"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PublishCommand = exports.factory = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const glob_1 = tslib_1.__importDefault(require("glob"));
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const os_1 = tslib_1.__importDefault(require("os"));
const path_1 = tslib_1.__importDefault(require("path"));
const crypto_1 = tslib_1.__importDefault(require("crypto"));
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const p_pipe_1 = tslib_1.__importDefault(require("p-pipe"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const temp_dir_1 = tslib_1.__importDefault(require("temp-dir"));
const version_1 = require("@lerna-lite/version");
const core_1 = require("@lerna-lite/core");
const get_current_tags_1 = require("./lib/get-current-tags");
const get_tagged_packages_1 = require("./lib/get-tagged-packages");
const get_unpublished_packages_1 = require("./lib/get-unpublished-packages");
const get_npm_username_1 = require("./lib/get-npm-username");
const verify_npm_package_access_1 = require("./lib/verify-npm-package-access");
const get_two_factor_auth_required_1 = require("./lib/get-two-factor-auth-required");
const get_current_sha_1 = require("./lib/get-current-sha");
const git_checkout_1 = require("./lib/git-checkout");
const pack_directory_1 = require("./lib/pack-directory");
const npm_publish_1 = require("./lib/npm-publish");
const log_packed_1 = require("./lib/log-packed");
const npm_dist_tag_1 = require("./lib/npm-dist-tag");
const override_publish_config_1 = require("./lib/override-publish-config");
const remove_temp_licenses_1 = require("./lib/remove-temp-licenses");
const create_temp_licenses_1 = require("./lib/create-temp-licenses");
const get_packages_without_license_1 = require("./lib/get-packages-without-license");
function factory(argv) {
    return new PublishCommand(argv);
}
exports.factory = factory;
class PublishCommand extends core_1.Command {
    get otherCommandConfigs() {
        // back-compat
        return ['version'];
    }
    get requiresGit() {
        // `lerna publish from-package` doesn't _need_ git, per se
        return this.options.bump !== 'from-package';
    }
    constructor(argv) {
        super(argv);
        /** command name */
        this.name = 'publish';
        this.gitReset = false;
        this.savePrefix = '';
        this.tagPrefix = '';
        this.hasRootedLeaf = false;
        this.npmSession = '';
        this.packagesToPublish = [];
        this.publishedPackages = [];
        this.packagesToBeLicensed = [];
        this.verifyAccess = false;
        this.toposort = false;
        this.twoFactorAuthRequired = false;
        this.updates = [];
    }
    configureProperties() {
        super.configureProperties();
        // For publish we want to enable topological sorting by default, but allow users to override with --no-sort
        this.toposort = this.options.sort !== false;
        // Defaults are necessary here because yargs defaults
        // override durable options provided by a config file
        const { 
        // prettier-ignore
        exact, gitHead, gitReset, tagVersionPrefix = 'v', verifyAccess, } = this.options;
        if (this.requiresGit && gitHead) {
            throw new core_1.ValidationError('EGITHEAD', '--git-head is only allowed with "from-package" positional');
        }
        // https://docs.npmjs.com/misc/config#save-prefix
        this.savePrefix = exact ? '' : '^';
        // https://docs.npmjs.com/misc/config#tag-version-prefix
        this.tagPrefix = tagVersionPrefix;
        // TODO: properly inherit from npm-conf
        // inverted boolean options are only respected if prefixed with `--no-`, e.g. `--no-verify-access`
        this.gitReset = gitReset !== false;
        // consumed by npm-registry-fetch (via libnpmpublish)
        this.npmSession = crypto_1.default.randomBytes(8).toString('hex');
        this.verifyAccess = verifyAccess;
    }
    get userAgent() {
        // consumed by npm-registry-fetch (via libnpmpublish)
        return `lerna/${this.options.lernaVersion}/node@${process.version}+${process.arch} (${process.platform})`;
    }
    initialize() {
        if (this.options.verifyAccess === false) {
            this.logger.warn('verify-access', '--verify-access=false and --no-verify-access are no longer needed, because the legacy preemptive access verification is now disabled by default. Requests will fail with appropriate errors when not authorized correctly.');
        }
        if (this.options.graphType === 'dependencies') {
            this.logger.warn('graph-type', '--graph-type=dependencies is deprecated and will be removed in the next major version of lerna-lite. If you have a use-case you feel requires it please open an issue to discuss: https://github.com/lerna/lerna/issues/new/choose');
        }
        if (this.options.workspaceStrictMatch === false) {
            this.logger.warn('deprecation', 'Providing --no-workspace-strict-match is deprecated and will be removed in future version, we will make "workspace:" protocol strict matching in every case.');
        }
        if (this.options.buildMetadata && this.options.canary) {
            throw new core_1.ValidationError('ENOTSATISFIED', 'Cannot use --build-metadata in conjunction with --canary option.');
        }
        else if (this.options.canary) {
            this.logger.info('canary', 'enabled');
        }
        // @deprecated, to be removed in next major
        if (this.options.requireScripts) {
            this.logger.info('require-scripts', 'enabled');
        }
        // npmSession and user-agent are consumed by npm-registry-fetch (via libnpmpublish)
        this.logger.verbose('session', this.npmSession);
        this.logger.verbose('user-agent', this.userAgent);
        this.conf = (0, core_1.npmConf)({
            lernaCommand: 'publish',
            _auth: this.options.legacyAuth,
            npmSession: this.npmSession,
            npmVersion: this.userAgent,
            otp: this.options.otp,
            registry: this.options.registry,
            'ignore-prepublish': this.options.ignorePrepublish,
            'ignore-scripts': this.options.ignoreScripts,
        });
        // cache to hold a one-time-password across publishes
        this.otpCache = { otp: this.conf.get('otp') };
        this.conf.set('user-agent', this.userAgent, 'cli');
        if (this.conf.get('registry') === 'https://registry.yarnpkg.com') {
            this.logger.warn('', `Yarn's registry proxy is broken, replacing with public npm registry`);
            this.logger.warn('', `If you don't have an npm token, you should exit and run "npm login"`);
            this.conf.set('registry', 'https://registry.npmjs.org/', 'cli');
        }
        // inject --dist-tag into opts, if present
        const distTag = this.getDistTag();
        if (distTag) {
            this.conf.set('tag', distTag.trim(), 'cli');
        }
        // a 'rooted leaf' is the regrettable pattern of adding '.' to the 'packages' config in lerna.json
        this.hasRootedLeaf = this.packageGraph.has(this.project.manifest.name);
        if (this.hasRootedLeaf) {
            this.logger.info('publish', 'rooted leaf detected, skipping synthetic root lifecycles');
        }
        this.runPackageLifecycle = (0, core_1.createRunner)(this.options);
        // don't execute recursively if run from a poorly-named script
        this.runRootLifecycle = /^(pre|post)?publish$/.test(process.env.npm_lifecycle_event || '')
            ? (stage) => this.logger.warn('lifecycle', 'Skipping root %j because it has already been called', stage)
            : (stage) => this.runPackageLifecycle(this.project.manifest, stage);
        let chain = Promise.resolve();
        if (this.options.bump === 'from-git') {
            chain = chain.then(() => this.detectFromGit());
        }
        else if (this.options.bump === 'from-package') {
            chain = chain.then(() => this.detectFromPackage());
        }
        else if (this.options.canary) {
            chain = chain.then(() => this.detectCanaryVersions());
        }
        else {
            chain = chain.then(() => new version_1.VersionCommand(this.argv));
        }
        return chain.then((result) => {
            if (!result) {
                // early return from nested VersionCommand
                return false;
            }
            if (!result.updates.length) {
                this.logger.success('No changed packages to publish');
                // still exits zero, aka 'ok'
                return false;
            }
            // (occasionally) redundant private filtering necessary to handle nested VersionCommand
            this.updates = result.updates.filter((node) => !node.pkg.private);
            this.updatesVersions = new Map(result.updatesVersions);
            this.packagesToPublish = this.updates.map((node) => node.pkg);
            if (this.options.contents) {
                // globally override directory to publish
                for (const pkg of this.packagesToPublish) {
                    pkg.contents = this.options.contents;
                }
            }
            if (result.needsConfirmation) {
                // only confirm for --canary, bump === 'from-git',
                // or bump === 'from-package', as VersionCommand
                // has its own confirmation prompt
                return this.confirmPublish();
            }
            return true;
        });
    }
    async execute() {
        var _a;
        this.enableProgressBar();
        this.logger.info('publish', 'Publishing packages to npm...');
        await this.prepareRegistryActions();
        await this.prepareLicenseActions();
        if (this.options.canary) {
            await this.updateCanaryVersions();
        }
        await this.resolveLocalDependencyLinks();
        await this.resolveLocalDependencyWorkspaceProtocols();
        if (this.options.removePackageFields) {
            await this.removePackageProperties();
        }
        if (this.options.publishConfigOverrides !== false) {
            await this.applyPublishConfigOverrides();
        }
        await this.annotateGitHead();
        await this.serializeChanges();
        await this.packUpdated();
        await this.publishPacked();
        if (this.gitReset) {
            await this.resetChanges();
        }
        if (this.options.tempTag) {
            await this.npmUpdateAsLatest();
        }
        const count = (_a = this.publishedPackages) === null || _a === void 0 ? void 0 : _a.length;
        const publishedPackagesSorted = this.publishedPackages.sort((a, b) => a.name.localeCompare(b.name));
        if (!count) {
            this.logger.success('All packages have already been published.');
            return;
        }
        (0, core_1.logOutput)('Successfully published:');
        if (this.options.summaryFile !== undefined) {
            // create a json object and output it to a file location.
            const filePath = this.options.summaryFile
                ? `${this.options.summaryFile}/lerna-publish-summary.json`
                : './lerna-publish-summary.json';
            const jsonObject = publishedPackagesSorted.map((pkg) => {
                return {
                    packageName: pkg.name,
                    version: pkg.version,
                };
            });
            (0, core_1.logOutput)(jsonObject);
            try {
                fs_extra_1.default.outputFileSync(filePath, JSON.stringify(jsonObject));
                (0, core_1.logOutput)('Publish summary created: ', filePath);
            }
            catch (error) {
                (0, core_1.logOutput)('Failed to create the summary report', error);
            }
        }
        else {
            const message = publishedPackagesSorted.map((pkg) => ` - ${pkg.name}@${pkg.version}`);
            (0, core_1.logOutput)(message.join(os_1.default.EOL));
        }
        // optionally cleanup temp packed files after publish, opt-in option
        if (this.options.cleanupTempFiles) {
            (0, glob_1.default)(path_1.default.join(temp_dir_1.default, '/lerna-*'), (_err, deleteFolders) => {
                // delete silently all files/folders that startsWith "lerna-"
                deleteFolders.forEach((folder) => fs_extra_1.default.removeSync(folder));
                this.logger.verbose('publish', `Found ${deleteFolders.length} temp folders to cleanup after publish.`);
            });
        }
        this.logger.success('published', '%d %s', count, count === 1 ? 'package' : 'packages');
    }
    verifyWorkingTreeClean() {
        return (0, core_1.describeRef)(this.execOpts, undefined, this.options.dryRun).then(core_1.throwIfUncommitted);
    }
    detectFromGit() {
        const matchingPattern = this.project.isIndependent() ? `*@*` : `${this.tagPrefix}*.*.*`;
        let chain = Promise.resolve();
        // attempting to publish a tagged release with local changes is not allowed
        chain = chain.then(() => this.verifyWorkingTreeClean());
        chain = chain.then(() => (0, get_current_tags_1.getCurrentTags)(this.execOpts, matchingPattern));
        chain = chain.then((taggedPackageNames) => {
            if (!taggedPackageNames.length) {
                this.logger.notice('from-git', 'No tagged release found. You might not have fetched tags.');
                return [];
            }
            if (this.project.isIndependent()) {
                return taggedPackageNames.map((name) => this.packageGraph.get(name));
            }
            return (0, get_tagged_packages_1.getTaggedPackages)(this.packageGraph, this.project.rootPath, this.execOpts);
        });
        // private packages are never published, full stop.
        chain = chain.then((updates) => updates.filter((node) => !node.pkg.private));
        return chain.then((updates) => {
            const updatesVersions = updates.map((node) => [node.name, node.version]);
            return {
                updates,
                updatesVersions,
                needsConfirmation: true,
            };
        });
    }
    detectFromPackage() {
        let chain = Promise.resolve();
        // attempting to publish a release with local changes is not allowed
        chain = chain
            .then(() => this.verifyWorkingTreeClean())
            .catch((err) => {
            // an execa error is thrown when git suffers a fatal error (such as no git repository present)
            if (err.failed && /git describe/.test(err.command)) {
                // (we tried)
                this.logger.silly('EWORKINGTREE', err.message);
                this.logger.notice('FYI', 'Unable to verify working tree, proceed at your own risk');
                process.exitCode = 0;
            }
            else {
                // validation errors should be preserved
                throw err;
            }
        });
        // private packages are already omitted by getUnpublishedPackages()
        chain = chain.then(() => (0, get_unpublished_packages_1.getUnpublishedPackages)(this.packageGraph, this.conf.snapshot));
        chain = chain.then((unpublished) => {
            if (!unpublished.length) {
                this.logger.notice('from-package', 'No unpublished release found');
            }
            return unpublished;
        });
        return chain.then((updates) => {
            const updatesVersions = updates.map((node) => [node.name, node.version]);
            return {
                updates,
                updatesVersions,
                needsConfirmation: true,
            };
        });
    }
    detectCanaryVersions() {
        const { cwd } = this.execOpts;
        const { bump = 'prepatch', preid = 'alpha', ignoreChanges, forcePublish, includeMergedTags } = this.options;
        // 'prerelease' and 'prepatch' are identical, for our purposes
        const release = bump.startsWith('pre') ? bump.replace('release', 'patch') : `pre${bump}`;
        let chain = Promise.resolve();
        // attempting to publish a canary release with local changes is not allowed
        chain = chain
            .then(() => this.verifyWorkingTreeClean())
            .catch((err) => {
            // an execa error is thrown when git suffers a fatal error (such as no git repository present)
            if (err.failed && /git describe/.test(err.command)) {
                // (we tried)
                this.logger.silly('EWORKINGTREE', err.message);
                this.logger.notice('FYI', 'Unable to verify working tree, proceed at your own risk');
            }
            else {
                // validation errors should be preserved
                throw err;
            }
        });
        const isIndependent = this.project.isIndependent();
        const describeTag = this.project.config.describeTag;
        // find changed packages since last release, if any
        chain = chain.then(() => {
            var _a, _b;
            return (0, core_1.collectUpdates)((_b = (_a = this.packageGraph) === null || _a === void 0 ? void 0 : _a.rawPackageList) !== null && _b !== void 0 ? _b : [], this.packageGraph, this.execOpts, {
                bump: 'prerelease',
                canary: true,
                ignoreChanges,
                forcePublish,
                includeMergedTags,
                isIndependent,
                describeTag,
                // private packages are never published, don't bother describing their refs.
            }, this.options.dryRun).filter((node) => !node.pkg.private);
        });
        // prettier-ignore
        const makeVersion = (fallback) => ({ lastVersion = fallback, refCount, sha }) => {
            // the next version is bumped without concern for preid or current index
            // prettier-ignore
            const nextVersion = semver_1.default.inc(lastVersion.replace(this.tagPrefix, ''), release.replace('pre', ''));
            // semver.inc() starts a new prerelease at .0, git describe starts at .1
            // and build metadata is always ignored when comparing dependency ranges
            return `${nextVersion}-${preid}.${Math.max(0, refCount - 1)}+${sha}`;
        };
        if (isIndependent) {
            // each package is described against its tags only
            chain = chain.then((updates) => (0, p_map_1.default)(updates, (node) => (0, core_1.describeRef)({
                match: `${node.name}@*`,
                cwd,
            }, includeMergedTags, this.options.dryRun)
                // an unpublished package will have no reachable git tag
                .then(makeVersion(node.version))
                .then((version) => [node.name, version])).then((updatesVersions) => ({
                updates,
                updatesVersions,
            })));
        }
        else {
            // all packages are described against the last tag
            chain = chain.then((updates) => (0, core_1.describeRef)({
                match: `${this.tagPrefix}*.*.*`,
                cwd,
            }, includeMergedTags, this.options.dryRun)
                // a repo with no tags should default to whatever lerna.json claims
                .then(makeVersion(this.project.version))
                .then((version) => updates.map((node) => [node.name, version]))
                .then((updatesVersions) => ({
                updates,
                updatesVersions,
            })));
        }
        return chain.then(({ updates, updatesVersions }) => ({
            updates,
            updatesVersions,
            needsConfirmation: true,
        }));
    }
    confirmPublish() {
        var _a, _b, _c;
        const count = (_a = this.packagesToPublish) === null || _a === void 0 ? void 0 : _a.length;
        const message = (_c = (_b = this.packagesToPublish) === null || _b === void 0 ? void 0 : _b.map((pkg) => { var _a; return ` - ${pkg.name} => ${(_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get(pkg.name)}`; })) !== null && _c !== void 0 ? _c : [];
        (0, core_1.logOutput)('');
        (0, core_1.logOutput)(`Found ${count} ${count === 1 ? 'package' : 'packages'} to publish:`);
        (0, core_1.logOutput)(message.join(os_1.default.EOL));
        (0, core_1.logOutput)('');
        if (this.options.yes) {
            this.logger.info('auto-confirmed', '');
            return true;
        }
        let confirmMessage = this.options.dryRun ? chalk_1.default.bgMagenta('[dry-run]') : '';
        confirmMessage += ' Are you sure you want to publish these packages?';
        return (0, core_1.promptConfirmation)(confirmMessage.trim());
    }
    prepareLicenseActions() {
        return Promise.resolve()
            .then(() => { var _a; return (0, get_packages_without_license_1.getPackagesWithoutLicense)(this.project, (_a = this.packagesToPublish) !== null && _a !== void 0 ? _a : []); })
            .then((packagesWithoutLicense) => {
            if (packagesWithoutLicense.length && !this.project.licensePath) {
                this.packagesToBeLicensed = [];
                const names = packagesWithoutLicense.map((pkg) => pkg.name);
                const noun = names.length > 1 ? 'Packages' : 'Package';
                const verb = names.length > 1 ? 'are' : 'is';
                const list = 
                // prettier-ignore
                names.length > 1
                    ? `${names.slice(0, -1).join(', ')}${names.length > 2 ? ',' : ''} and ${names[names.length - 1] /* oxford commas _are_ that important */}`
                    : names[0];
                this.logger.warn('ENOLICENSE', '%s %s %s missing a license.\n%s\n%s', noun, list, verb, 'One way to fix this is to add a LICENSE.md file to the root of this repository.', 'See https://choosealicense.com for additional guidance.');
            }
            else {
                this.packagesToBeLicensed = packagesWithoutLicense;
            }
        });
    }
    prepareRegistryActions() {
        let chain = Promise.resolve();
        if (this.conf.get('registry') !== 'https://registry.npmjs.org/') {
            this.logger.notice('', 'Skipping all user and access validation due to third-party registry');
            this.logger.notice('', `Make sure you're authenticated properly "\\_(ツ)_ /"`);
            return chain;
        }
        /* istanbul ignore if */
        if (process.env.LERNA_INTEGRATION) {
            return chain;
        }
        if (this.verifyAccess) {
            // validate user has valid npm credentials first,
            // by far the most common form of failed execution
            chain = chain.then(() => (0, get_npm_username_1.getNpmUsername)(this.conf.snapshot));
            chain = chain.then((username) => {
                var _a;
                // if no username was retrieved, don't bother validating
                if (username) {
                    return (0, verify_npm_package_access_1.verifyNpmPackageAccess)((_a = this.packagesToPublish) !== null && _a !== void 0 ? _a : [], username, this.conf.snapshot);
                }
            });
            // read profile metadata to determine if account-level 2FA is enabled
            chain = chain.then(() => (0, get_two_factor_auth_required_1.getTwoFactorAuthRequired)(this.conf.snapshot));
            chain = chain.then((isRequired) => {
                // notably, this still doesn't handle package-level 2FA requirements
                this.twoFactorAuthRequired = isRequired;
            });
        }
        return chain;
    }
    updateCanaryVersions() {
        return (0, p_map_1.default)(this.updates, (node) => {
            var _a, _b, _c;
            node.pkg.set('version', (_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get(node.name));
            for (const [depName, resolved] of node.localDependencies) {
                // other canary versions need to be updated, non-canary is a no-op
                const depVersion = ((_b = this.updatesVersions) === null || _b === void 0 ? void 0 : _b.get(depName)) || ((_c = this.packageGraph) === null || _c === void 0 ? void 0 : _c.get(depName).pkg.version);
                // it no longer matters if we mutate the shared Package instance
                node.pkg.updateLocalDependency(resolved, depVersion, this.savePrefix, this.options.allowPeerDependenciesUpdate, this.options.workspaceStrictMatch, this.commandName);
            }
            // writing changes to disk handled in serializeChanges()
        });
    }
    /**
     * It is possible to override some fields in the manifest before the package is packed
     * @see https://pnpm.io/package_json#publishconfig
     * @returns
     */
    applyPublishConfigOverrides() {
        // potentially apply any packages that might have publishConfig overrides
        return (0, p_map_1.default)(this.updates, (node) => (0, override_publish_config_1.overridePublishConfig)(node.pkg.manifest));
    }
    resolveLocalDependencyLinks() {
        // resolve relative file: links to their actual version range
        const updatesWithLocalLinks = this.updates.filter((node) => Array.from(node.localDependencies.values()).some((resolved) => resolved.type === 'directory'));
        return (0, p_map_1.default)(updatesWithLocalLinks, (node) => {
            var _a, _b;
            for (const [depName, resolved] of node.localDependencies) {
                // regardless of where the version comes from, we can't publish 'file:../sibling-pkg' specs
                const depVersion = ((_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get(depName)) || ((_b = this.packageGraph) === null || _b === void 0 ? void 0 : _b.get(depName).pkg.version);
                // it no longer matters if we mutate the shared Package instance
                node.pkg.updateLocalDependency(resolved, depVersion, this.savePrefix, this.options.allowPeerDependenciesUpdate, this.options.workspaceStrictMatch, this.commandName);
            }
            // writing changes to disk handled in serializeChanges()
        });
    }
    resolveLocalDependencyWorkspaceProtocols() {
        // resolve workspace protocol: translates to their actual version target/range
        const publishingPackagesWithLocalWorkspaces = this.updates.filter((node) => Array.from(node.localDependencies.values()).some((resolved) => resolved.workspaceSpec));
        return (0, p_map_1.default)(publishingPackagesWithLocalWorkspaces, (node) => {
            // regardless of where the version comes from, we can't publish 'workspace:*' specs, it has to be transformed for both local & external dependencies
            // e.g. considering version is `1.2.3` and we have `workspace:*` it will be converted to "^1.2.3" or to "1.2.3" with strict match range enabled
            var _a, _b;
            // 1. update & bump version of local dependencies
            for (const [depName, resolved] of node.localDependencies) {
                const depVersion = ((_a = this.updatesVersions) === null || _a === void 0 ? void 0 : _a.get(depName)) || ((_b = this.packageGraph) === null || _b === void 0 ? void 0 : _b.get(depName).pkg.version);
                // it no longer matters if we mutate the shared Package instance
                node.pkg.updateLocalDependency(resolved, depVersion, this.savePrefix, this.options.allowPeerDependenciesUpdate, this.options.workspaceStrictMatch, this.commandName);
            }
            // 2. remove any "workspace:" prefix from the package to be published any of external dependencies (without anything being bumped)
            // we will only accept "workspace:" with semver version, for example "workspace:1.2.3" is ok but "workspace:*" will log an error
            for (const [_depName, resolved] of node.externalDependencies) {
                node.pkg.removeDependencyWorkspaceProtocolPrefix(node.name, resolved);
            }
            // writing changes to disk handled in serializeChanges()
        });
    }
    annotateGitHead() {
        var _a;
        try {
            const gitHead = this.options.gitHead || (0, get_current_sha_1.getCurrentSHA)(this.execOpts);
            for (const pkg of (_a = this.packagesToPublish) !== null && _a !== void 0 ? _a : []) {
                // provide gitHead property that is normally added during npm publish
                pkg.set('gitHead', gitHead);
            }
        }
        catch (err) {
            // from-package should be _able_ to run without git, but at least we tried
            this.logger.silly('EGITHEAD', err.message);
            this.logger.notice('FYI', 'Unable to set temporary gitHead property, it will be missing from registry metadata');
        }
        // writing changes to disk handled in serializeChanges()
    }
    serializeChanges() {
        var _a;
        return (0, p_map_1.default)((_a = this.packagesToPublish) !== null && _a !== void 0 ? _a : [], (pkg) => pkg.serialize());
    }
    resetChanges() {
        // the package.json files are changed (by gitHead if not --canary)
        // and we should always __attempt_ to leave the working tree clean
        const { cwd } = this.execOpts;
        const gitOpts = {
            granularPathspec: this.options.granularPathspec !== false,
        };
        const dirtyManifests = [this.project.manifest]
            .concat(this.packagesToPublish)
            .map((pkg) => path_1.default.relative(cwd, pkg.manifestLocation));
        return (0, git_checkout_1.gitCheckout)(dirtyManifests, gitOpts, this.execOpts, this.options.dryRun).catch((err) => {
            this.logger.silly('EGITCHECKOUT', err.message);
            this.logger.notice('FYI', `Unable to reset working tree changes, this probably isn't a git repo.`);
        });
    }
    // @deprecated, see Lerna PR https://github.com/lerna/lerna/pull/1862/files
    execScript(pkg, script) {
        const scriptLocation = path_1.default.join(pkg.location, 'scripts', script);
        try {
            require(scriptLocation);
        }
        catch (ex) {
            this.logger.silly('execScript', `No ${script} script found at ${scriptLocation}`);
        }
        return pkg;
    }
    removePackageProperties() {
        const { removePackageFields } = this.options;
        return (0, p_map_1.default)(this.updates, (node) => {
            if (Array.isArray(removePackageFields)) {
                for (const removeField of removePackageFields) {
                    (0, core_1.deleteComplexObjectProp)(node.pkg.manifest, removeField, `"${node.pkg.name}" package`);
                }
            }
        });
    }
    removeTempLicensesOnError(error) {
        return Promise.resolve()
            .then(() => {
            var _a;
            return (0, remove_temp_licenses_1.removeTempLicenses)((_a = this.packagesToBeLicensed) !== null && _a !== void 0 ? _a : []).catch((removeError) => {
                this.logger.error('licenses', 'error removing temporary license files', removeError.stack || removeError);
            });
        })
            .then(() => {
            // restore original error into promise chain
            throw error;
        });
    }
    requestOneTimePassword() {
        if (this.options.dryRun) {
            this.logger.info(chalk_1.default.bold.magenta('[dry-run] >'), 'will ask OTP');
            return;
        }
        // if OTP has already been provided, skip prompt
        if (this.otpCache.otp) {
            return;
        }
        return Promise.resolve()
            .then(() => (0, version_1.getOneTimePassword)('Enter OTP:'))
            .then((otp) => (this.otpCache.otp = otp));
    }
    topoMapPackages(mapper) {
        return (0, core_1.runTopologically)(this.packagesToPublish, mapper, {
            concurrency: this.concurrency,
            rejectCycles: this.options.rejectCycles,
            /**
             * Previously `publish` had unique default behavior for graph creation vs other commands: it would only consider dependencies when finding
             * edges by default (i.e. relationships between packages specified via devDependencies would be ignored). It was documented to be the case
             * in order to try and reduce the chance of dependency cycles.
             *
             * We are removing this behavior altogether in v6 because we do not want to have different ways of constructing the graph,
             * only different ways of utilizing it (e.g. --no-sort vs topological sort).
             *
             * Therefore until we remove graphType altogether in v6, we provide a way for users to opt into the old default behavior
             * by setting the `graphType` option to `dependencies`.
             */
            // prettier-ignore
            graphType: this.options.graphType === 'dependencies'
                ? 'dependencies'
                : this.options.allowPeerDependenciesUpdate
                    ? 'allPlusPeerDependencies'
                    : 'allDependencies',
        });
    }
    packUpdated() {
        var _a;
        const tracker = this.logger.newItem('npm pack');
        tracker.addWork((_a = this.packagesToPublish) === null || _a === void 0 ? void 0 : _a.length);
        let chain = Promise.resolve();
        chain = chain.then(() => { var _a; return (0, create_temp_licenses_1.createTempLicenses)(this.project.licensePath, (_a = this.packagesToBeLicensed) !== null && _a !== void 0 ? _a : []); });
        if (!this.hasRootedLeaf) {
            // despite being deprecated for years...
            chain = chain.then(() => this.runRootLifecycle('prepublish'));
            // these lifecycles _should_ never be employed to run `lerna publish`...
            chain = chain.then(() => this.runPackageLifecycle(this.project.manifest, 'prepare'));
            chain = chain.then(() => this.runPackageLifecycle(this.project.manifest, 'prepublishOnly'));
            chain = chain.then(() => this.runPackageLifecycle(this.project.manifest, 'prepack'));
        }
        const opts = this.conf.snapshot;
        const mapper = (0, p_pipe_1.default)(...[
            this.options.requireScripts && ((pkg) => this.execScript(pkg, 'prepublish')),
            (pkg) => (0, core_1.pulseTillDone)((0, pack_directory_1.packDirectory)(pkg, pkg.location, opts)).then((packed) => {
                var _a;
                tracker.verbose('packed', path_1.default.relative((_a = this.project.rootPath) !== null && _a !== void 0 ? _a : '', pkg.contents));
                tracker.completeWork(1);
                // store metadata for use in this.publishPacked()
                pkg.packed = packed;
                // manifest may be mutated by any previous lifecycle
                return pkg.refresh();
            }),
        ].filter(Boolean));
        chain = chain.then(() => {
            if (this.toposort) {
                return this.topoMapPackages(mapper);
            }
            return (0, p_map_1.default)(this.packagesToPublish, mapper, { concurrency: this.concurrency });
        });
        chain = chain.then(() => { var _a; return (0, remove_temp_licenses_1.removeTempLicenses)((_a = this.packagesToBeLicensed) !== null && _a !== void 0 ? _a : []); });
        // remove temporary license files if _any_ error occurs _anywhere_ in the promise chain
        chain = chain.catch((error) => this.removeTempLicensesOnError(error));
        if (!this.hasRootedLeaf) {
            chain = chain.then(() => this.runPackageLifecycle(this.project.manifest, 'postpack'));
        }
        return chain.finally(() => tracker.finish());
    }
    publishPacked() {
        var _a;
        this.publishedPackages = [];
        const tracker = this.logger.newItem('publish');
        tracker.addWork((_a = this.packagesToPublish) === null || _a === void 0 ? void 0 : _a.length);
        let chain = Promise.resolve();
        // if account-level 2FA is enabled, prime the OTP cache
        if (this.twoFactorAuthRequired) {
            chain = chain.then(() => this.requestOneTimePassword());
        }
        const opts = Object.assign(this.conf.snapshot, {
            // distTag defaults to 'latest' OR whatever is in pkg.publishConfig.tag
            // if we skip temp tags we should tag with the proper value immediately
            tag: this.options.tempTag ? 'lerna-temp' : this.conf.get('tag'),
            'git-dry-run': this.options.dryRun || false,
        });
        const mapper = (0, p_pipe_1.default)(...[
            (pkg) => {
                const preDistTag = this.getPreDistTag(pkg);
                const tag = !this.options.tempTag && preDistTag ? preDistTag : opts.tag;
                const pkgOpts = Object.assign({}, opts, { tag });
                return (0, core_1.pulseTillDone)((0, npm_publish_1.npmPublish)(pkg, pkg.packed.tarFilePath, pkgOpts, this.otpCache))
                    .then(() => {
                    this.publishedPackages.push(pkg);
                    tracker.success('published', pkg.name, pkg.version);
                    tracker.completeWork(1);
                    (0, log_packed_1.logPacked)(pkg, this.options.dryRun);
                    return pkg;
                })
                    .catch((err) => {
                    if (err.code === 'EPUBLISHCONFLICT') {
                        tracker.warn('publish', `Package is already published: ${pkg.name}@${pkg.version}`);
                        tracker.completeWork(1);
                        return pkg;
                    }
                    this.logger.silly('', err);
                    this.logger.error(err.code, (err.body && err.body.error) || err.message);
                    // avoid dumping logs, this isn't a lerna problem
                    err.name = 'ValidationError';
                    // ensure process exits non-zero
                    process.exitCode = 'errno' in err ? err.errno : 1;
                    throw err;
                });
            },
            this.options.requireScripts && ((pkg) => this.execScript(pkg, 'postpublish')),
        ].filter(Boolean));
        chain = chain.then(() => {
            if (this.toposort) {
                return this.topoMapPackages(mapper);
            }
            return (0, p_map_1.default)(this.packagesToPublish, mapper, { concurrency: this.concurrency });
        });
        if (!this.hasRootedLeaf) {
            // cyclical 'publish' lifecycles are automatically skipped
            chain = chain.then(() => this.runRootLifecycle('publish'));
            chain = chain.then(() => this.runRootLifecycle('postpublish'));
        }
        return chain.finally(() => tracker.finish());
    }
    npmUpdateAsLatest() {
        var _a;
        const tracker = this.logger.newItem('npmUpdateAsLatest');
        tracker.addWork((_a = this.packagesToPublish) === null || _a === void 0 ? void 0 : _a.length);
        tracker.showProgress();
        let chain = Promise.resolve();
        const opts = this.conf.snapshot;
        const getDistTag = (publishConfig) => {
            if (opts.tag === 'latest' && (publishConfig === null || publishConfig === void 0 ? void 0 : publishConfig.tag)) {
                return publishConfig.tag;
            }
            return opts.tag;
        };
        const mapper = (pkg) => {
            const spec = `${pkg.name}@${pkg.version}`;
            const preDistTag = this.getPreDistTag(pkg);
            const distTag = preDistTag || getDistTag(pkg.get('publishConfig'));
            return Promise.resolve()
                .then(() => (0, core_1.pulseTillDone)((0, npm_dist_tag_1.remove)(spec, 'lerna-temp', opts, this.otpCache)))
                .then(() => (0, core_1.pulseTillDone)((0, npm_dist_tag_1.add)(spec, distTag, opts, this.otpCache)))
                .then(() => {
                tracker.success('dist-tag', '%s@%s => %j', pkg.name, pkg.version, distTag);
                tracker.completeWork(1);
                return pkg;
            });
        };
        chain = chain.then(() => {
            if (this.toposort) {
                return this.topoMapPackages(mapper);
            }
            return (0, p_map_1.default)(this.packagesToPublish, mapper, { concurrency: this.concurrency });
        });
        return chain.finally(() => tracker.finish());
    }
    getDistTag() {
        if (this.options.distTag) {
            return this.options.distTag;
        }
        if (this.options.canary) {
            return 'canary';
        }
        // undefined defaults to 'latest' OR whatever is in pkg.publishConfig.tag
    }
    getPreDistTag(pkg) {
        if (!this.options.preDistTag) {
            return;
        }
        const isPrerelease = (0, core_1.prereleaseIdFromVersion)(pkg.version);
        if (isPrerelease) {
            return this.options.preDistTag;
        }
    }
}
exports.PublishCommand = PublishCommand;
//# sourceMappingURL=publish-command.js.map