"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyNpmPackageAccess = void 0;
const tslib_1 = require("tslib");
const libnpmaccess_1 = tslib_1.__importDefault(require("libnpmaccess"));
const core_1 = require("@lerna-lite/core");
const fetch_config_1 = require("./fetch-config");
/**
 * Throw an error if the logged-in user does not have read-write access to all packages.
 * @param {{ name: string; }[]} packages
 * @param {string} username
 * @param {import('./fetch-config').FetchConfig} options
 * @returns {Promise<void>}
 */
function verifyNpmPackageAccess(packages, username, options) {
    const opts = (0, fetch_config_1.getFetchConfig)(options, {
        // don't wait forever for third-party failures to be dealt with
        fetchRetries: 0,
    });
    opts.log.silly('verifyNpmPackageAccess', '');
    return (0, core_1.pulseTillDone)(libnpmaccess_1.default.getPackages(username, opts)).then(success, failure);
    function success(result) {
        // when _no_ results received, access.getPackages returns null
        // we can only assume that the packages in question have never been published
        if (result === null) {
            opts.log.warn('', 'The logged-in user does not have any previously-published packages, skipping permission checks...');
        }
        else {
            for (const pkg of packages) {
                if ((pkg === null || pkg === void 0 ? void 0 : pkg.name) in result && result[pkg === null || pkg === void 0 ? void 0 : pkg.name] !== 'read-write') {
                    throw new core_1.ValidationError('EACCESS', `You do not have write permission required to publish "${pkg.name}"`);
                }
            }
        }
    }
    function failure(err) {
        // pass if registry does not support ls-packages endpoint
        if (err.code === 'E500' || err.code === 'E404') {
            // most likely a private registry (npm Enterprise, verdaccio, etc)
            opts.log.warn('EREGISTRY', 'Registry %j does not support `npm access ls-packages`, skipping permission checks...', 
            // registry
            opts.registry);
            // don't log redundant errors
            return;
        }
        // Log the error cleanly to stderr
        opts.log.pause();
        console.error(err.message); // eslint-disable-line no-console
        opts.log.resume();
        throw new core_1.ValidationError('EWHOAMI', 'Authentication error. Use `npm whoami` to troubleshoot.');
    }
}
exports.verifyNpmPackageAccess = verifyNpmPackageAccess;
//# sourceMappingURL=verify-npm-package-access.js.map