"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.packDirectory = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const version_1 = require("@lerna-lite/version");
const arborist_1 = tslib_1.__importDefault(require("@npmcli/arborist"));
const path_1 = tslib_1.__importDefault(require("path"));
const npm_packlist_1 = tslib_1.__importDefault(require("npm-packlist"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const tar_1 = tslib_1.__importDefault(require("tar"));
const get_packed_1 = require("./get-packed");
/**
 * Pack a directory suitable for publishing, writing tarball to a tempfile.
 * @param {Package|string} _pkg Package instance or path to manifest
 * @param {string} dir to pack
 * @param {PackConfig} options
 */
async function packDirectory(_pkg, dir, options) {
    const pkg = core_1.Package.lazy(_pkg, dir);
    const opts = {
        // @ts-ignore
        log: npmlog_1.default,
        ...options,
    };
    opts.log.verbose('pack-directory', path_1.default.relative('.', pkg.contents));
    if (opts.ignorePrepublish !== true) {
        await (0, core_1.runLifecycle)(pkg, 'prepublish', opts);
    }
    await (0, core_1.runLifecycle)(pkg, 'prepare', opts);
    if (opts.lernaCommand === 'publish') {
        opts.stdio = 'inherit';
        await pkg.refresh();
        await (0, core_1.runLifecycle)(pkg, 'prepublishOnly', opts);
        await pkg.refresh();
    }
    await (0, core_1.runLifecycle)(pkg, 'prepack', opts);
    await pkg.refresh();
    const arborist = new arborist_1.default({ path: pkg.contents });
    const tree = await arborist.loadActual();
    const files = await (0, npm_packlist_1.default)(tree);
    const stream = tar_1.default.create({
        cwd: pkg.contents,
        prefix: 'package/',
        portable: true,
        // Provide a specific date in the 1980s for the benefit of zip,
        // which is confounded by files dated at the Unix epoch 0.
        mtime: new Date('1985-10-26T08:15:00.000Z'),
        gzip: true,
    }, 
    // NOTE: node-tar does some Magic Stuff depending on prefixes for files
    //       specifically with @ signs, so we just neutralize that one
    //       and any such future 'features' by prepending `./`
    files.map((f) => `./${f}`));
    const tarFilePath = await (0, version_1.tempWrite)(stream, getTarballName(pkg));
    return (0, get_packed_1.getPacked)(pkg, tarFilePath).then((packed) => Promise.resolve()
        .then(() => (0, core_1.runLifecycle)(pkg, 'postpack', opts))
        .then(() => packed));
}
exports.packDirectory = packDirectory;
function getTarballName(pkg) {
    const name = pkg.name[0] === '@'
        ? // scoped packages get special treatment
            pkg.name.substring(1).replace(/\//g, '-')
        : pkg.name;
    return `${name}-${pkg.version}.tgz`;
}
//# sourceMappingURL=pack-directory.js.map