"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.npmPublish = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const version_1 = require("@lerna-lite/version");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const npm_package_arg_1 = tslib_1.__importDefault(require("npm-package-arg"));
const path_1 = tslib_1.__importDefault(require("path"));
const pify_1 = tslib_1.__importDefault(require("pify"));
const libnpmpublish_1 = require("libnpmpublish");
const read_package_json_1 = tslib_1.__importDefault(require("read-package-json"));
const readJSONAsync = (0, pify_1.default)(read_package_json_1.default);
/**
 * Alias dash-cased npmConf to camelCase
 * @param {NpmPublishOptions} obj
 * @returns {NpmPublishOptions}
 */
function flattenOptions(obj) {
    return {
        // eslint-disable-next-line dot-notation -- (npm v7 compat)
        defaultTag: obj['tag'] || 'latest',
        dryRun: obj['dry-run'] || obj['git-dry-run'],
        // libnpmpublish / npm-registry-fetch check strictSSL rather than strict-ssl
        strictSSL: obj['strict-ssl'],
        ...obj,
    };
}
/**
 * Publish a package to the configured registry.
 * @param {import("@lerna/package").Package} pkg
 * @param {string} tarFilePath
 * @param {LibNpmPublishOptions & NpmPublishOptions} [options]
 * @param {import("@lerna/otplease").OneTimePasswordCache} [otpCache]
 */
function npmPublish(pkg, tarFilePath, options = {}, otpCache) {
    var _a;
    const { dryRun, ...remainingOptions } = flattenOptions(options);
    const { scope } = (0, npm_package_arg_1.default)((_a = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _a !== void 0 ? _a : '');
    // pass only the package scope to libnpmpublish
    const opts = {
        log: npmlog_1.default,
        ...remainingOptions,
        projectScope: scope,
    };
    opts.log.verbose('publish', pkg.name);
    let chain = Promise.resolve();
    if (!dryRun) {
        chain = chain.then(() => {
            let { manifestLocation } = pkg;
            if (pkg.contents !== pkg.location) {
                // 'rebase' manifest used to generated directory
                manifestLocation = path_1.default.join(pkg.contents, 'package.json');
            }
            return Promise.all([fs_extra_1.default.readFile(tarFilePath), readJSONAsync(manifestLocation)]);
        });
        chain = chain.then(([tarData, manifest]) => {
            // non-default tag needs to override publishConfig.tag,
            // which is merged into opts below if necessary
            if (opts.defaultTag !== 'latest' &&
                manifest.publishConfig &&
                manifest.publishConfig.tag &&
                manifest.publishConfig.tag !== opts.defaultTag) {
                // eslint-disable-next-line no-param-reassign
                manifest.publishConfig.tag = opts.defaultTag;
            }
            // publishConfig is no longer consumed in n-r-f, so merge here
            if (manifest.publishConfig) {
                Object.assign(opts, publishConfigToOpts(manifest.publishConfig));
            }
            return (0, version_1.otplease)((innerOpts) => (0, libnpmpublish_1.publish)(manifest, tarData, innerOpts), opts, otpCache);
        });
    }
    chain = chain.then(() => (0, core_1.runLifecycle)(pkg, 'publish', opts));
    chain = chain.then(() => (0, core_1.runLifecycle)(pkg, 'postpublish', opts));
    return chain;
}
exports.npmPublish = npmPublish;
/**
 * Obtain an object suitable for assignment onto existing options from `pkg.publishConfig`.
 * @param {PackagePublishConfig} publishConfig
 * @returns {Omit<PackagePublishConfig, 'tag'> & { defaultTag?: string }}
 */
function publishConfigToOpts(publishConfig) {
    const opts = { ...publishConfig };
    // npm v7 renamed tag internally
    if (publishConfig.tag) {
        opts.defaultTag = publishConfig.tag;
        delete opts.tag;
    }
    return opts;
}
//# sourceMappingURL=npm-publish.js.map