"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.logPacked = void 0;
const tslib_1 = require("tslib");
const byte_size_1 = tslib_1.__importDefault(require("byte-size"));
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const columnify_1 = tslib_1.__importDefault(require("columnify"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const has_unicode_1 = tslib_1.__importDefault(require("has-unicode"));
function logPacked(pkg, dryRun = false) {
    var _a, _b, _c;
    const tarball = pkg.packed;
    npmlog_1.default.notice('', '');
    npmlog_1.default.notice('', `${(0, has_unicode_1.default)() ? '📦 ' : 'package:'} ${tarball.name}@${tarball.version}`);
    if (tarball.files && tarball.files.length) {
        npmlog_1.default.notice('=== Tarball Contents ===', '');
        npmlog_1.default.notice('', (0, columnify_1.default)(tarball.files.map((f) => {
            const bytes = (0, byte_size_1.default)(f.size);
            return {
                path: f.path,
                size: `${bytes.value}${bytes.unit}`,
            };
        }), {
            include: ['size', 'path'],
            showHeaders: false,
        }));
    }
    if (tarball.bundled && tarball.bundled.length) {
        npmlog_1.default.notice('=== Bundled Dependencies ===', '');
        tarball.bundled.forEach((name) => npmlog_1.default.notice('', name));
    }
    npmlog_1.default.notice('=== Tarball Details ===', '');
    npmlog_1.default.notice('', (0, columnify_1.default)([
        { name: 'name:', value: tarball.name },
        { name: 'version:', value: tarball.version },
        tarball.filename && { name: 'filename:', value: tarball.filename },
        tarball.size && { name: 'package size:', value: (0, byte_size_1.default)(tarball.size) },
        tarball.unpackedSize && { name: 'unpacked size:', value: (0, byte_size_1.default)(tarball.unpackedSize) },
        tarball.shasum && { name: 'shasum:', value: tarball.shasum },
        tarball.integrity && { name: 'integrity:', value: elideIntegrity(tarball.integrity) },
        ((_a = tarball.bundled) === null || _a === void 0 ? void 0 : _a.length) && { name: 'bundled deps:', value: tarball.bundled.length },
        ((_b = tarball.bundled) === null || _b === void 0 ? void 0 : _b.length) && { name: 'bundled files:', value: tarball.entryCount - tarball.files.length },
        ((_c = tarball.bundled) === null || _c === void 0 ? void 0 : _c.length) && { name: 'own files:', value: tarball.files.length },
        tarball.entryCount && { name: 'total files:', value: tarball.entryCount },
    ].filter((x) => x), {
        include: ['name', 'value'],
        showHeaders: false,
    }));
    // in dry-run mode, show tarball temp location and dependencies, devDependencies
    if (dryRun) {
        npmlog_1.default.notice('', `--- ${chalk_1.default.bgMagenta('DRY-RUN')} details ---`);
        npmlog_1.default.notice('', `temp location: ${tarball.tarFilePath}`);
        npmlog_1.default.notice('', `package name: ${pkg.name}`);
        if (pkg.dependencies) {
            npmlog_1.default.notice('dependencies:', '');
            npmlog_1.default.notice('', (0, columnify_1.default)(pkg.dependencies, { columnSplitter: ' | ', showHeaders: false }));
        }
        if (pkg.devDependencies) {
            npmlog_1.default.notice('devDependencies:', '');
            npmlog_1.default.notice('', (0, columnify_1.default)(pkg.devDependencies, { columnSplitter: ' | ', showHeaders: false }));
        }
    }
    // an empty line
    npmlog_1.default.notice('', '');
}
exports.logPacked = logPacked;
function elideIntegrity(integrity) {
    const str = integrity.toString();
    return `${str.substr(0, 20)}[...]${str.substr(80)}`;
}
//# sourceMappingURL=log-packed.js.map