"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTwoFactorAuthRequired = void 0;
const core_1 = require("@lerna-lite/core");
const fetch_config_1 = require("./fetch-config");
const get_profile_data_1 = require("./get-profile-data");
/**
 * Determine if the logged-in user has enabled two-factor auth.
 * @param {import('./fetch-config').FetchConfig} options
 * @returns {Promise<boolean>}
 */
function getTwoFactorAuthRequired(options) {
    const opts = (0, fetch_config_1.getFetchConfig)(options, {
        // don't wait forever for third-party failures to be dealt with
        fetchRetries: 0,
    });
    opts.log.info('', 'Checking two-factor auth mode');
    return (0, get_profile_data_1.getProfileData)(opts).then(success, failure);
    function success(result) {
        opts.log.silly('2FA', result.tfa.toString());
        if (result.tfa.pending) {
            // if 2FA is pending, it is disabled
            return false;
        }
        return result.tfa.mode === 'auth-and-writes';
    }
    function failure(err) {
        // pass if registry does not support profile endpoint
        if (err.code === 'E500' || err.code === 'E404') {
            // most likely a private registry (npm Enterprise, verdaccio, etc)
            opts.log.warn('EREGISTRY', `Registry "${opts.registry}" does not support 'npm profile get', skipping two-factor auth check...`);
            // don't log redundant errors
            return false;
        }
        // Log the error cleanly to stderr
        opts.log.pause();
        console.error(err.message); // eslint-disable-line no-console
        opts.log.resume();
        throw new core_1.ValidationError('ETWOFACTOR', 'Unable to obtain two-factor auth mode');
    }
}
exports.getTwoFactorAuthRequired = getTwoFactorAuthRequired;
//# sourceMappingURL=get-two-factor-auth-required.js.map