"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPacked = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const path_1 = tslib_1.__importDefault(require("path"));
const ssri_1 = tslib_1.__importDefault(require("ssri"));
const tar_1 = tslib_1.__importDefault(require("tar"));
function getPacked(pkg, tarFilePath) {
    const bundledWanted = new Set(/* pkg.bundleDependencies || pkg.bundledDependencies || */ []);
    const bundled = new Set();
    const files = [];
    let totalEntries = 0;
    let totalEntrySize = 0;
    return tar_1.default
        .list({
        file: tarFilePath,
        onentry(entry) {
            totalEntries += 1;
            totalEntrySize += entry.size;
            const p = entry.path;
            /* istanbul ignore if */
            if (p.startsWith('package/node_modules/')) {
                const name = p.match(/^package\/node_modules\/((?:@[^/]+\/)?[^/]+)/)[1];
                if (bundledWanted.has(name)) {
                    bundled.add(name);
                }
            }
            else {
                files.push({
                    path: entry.path.replace(/^package\//, ''),
                    size: entry.size,
                    mode: entry.mode,
                });
            }
        },
        strip: 1,
    })
        .then(() => Promise.all([
        fs_extra_1.default.stat(tarFilePath),
        ssri_1.default.fromStream(fs_extra_1.default.createReadStream(tarFilePath), {
            algorithms: ['sha1', 'sha512'],
        }),
    ]))
        .then(([{ size }, { sha1, sha512 }]) => {
        const shasum = sha1[0].hexDigest();
        return {
            id: `${pkg.name}@${pkg.version}`,
            name: pkg.name,
            version: pkg.version,
            size,
            unpackedSize: totalEntrySize,
            shasum,
            integrity: ssri_1.default.parse(sha512[0]),
            filename: path_1.default.basename(tarFilePath),
            files,
            entryCount: totalEntries,
            bundled: Array.from(bundled),
            tarFilePath,
        };
    });
}
exports.getPacked = getPacked;
//# sourceMappingURL=get-packed.js.map