"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getNpmUsername = void 0;
const fetch_config_1 = require("./fetch-config");
const get_profile_data_1 = require("./get-profile-data");
const get_whoami_1 = require("./get-whoami");
const core_1 = require("@lerna-lite/core");
/**
 * Retrieve username of logged-in user.
 * @param {import("./fetch-config").FetchConfig} options
 * @returns {Promise<string>}
 */
function getNpmUsername(options) {
    const opts = (0, fetch_config_1.getFetchConfig)(options, {
        // don't wait forever for third-party failures to be dealt with
        fetchRetries: 0,
    });
    opts.log.info('', 'Verifying npm credentials');
    return (0, get_profile_data_1.getProfileData)(opts)
        .catch((err) => {
        // Many third-party registries do not implement the user endpoint
        // Legacy npm Enterprise returns E500 instead of E404
        if (err.code === 'E500' || err.code === 'E404') {
            return (0, get_whoami_1.getWhoAmI)(opts);
        }
        // re-throw 401 Unauthorized (and all other unexpected errors)
        throw err;
    })
        .then(success, failure);
    function success(result) {
        opts.log.silly('get npm username', 'received %j', result);
        if (!result.username) {
            throw new core_1.ValidationError('ENEEDAUTH', 'You must be logged in to publish packages. Use `npm login` and try again.');
        }
        return result.username;
    }
    // catch request errors, not auth expired errors
    function failure(err) {
        // Log the error cleanly to stderr
        opts.log.pause();
        console.error(err.message); // eslint-disable-line no-console
        opts.log.resume();
        if (opts.registry === 'https://registry.npmjs.org/') {
            if (err.code === 'E403') {
                throw new core_1.ValidationError('ENEEDAUTH', 'Access verification failed. Ensure that your npm access token has both read and write access, or remove the verifyAccess option to skip this verification. Note that npm automation tokens do NOT have read access (https://docs.npmjs.com/creating-and-viewing-access-tokens).');
            }
            throw new core_1.ValidationError('EWHOAMI', 'Authentication error. Use `npm whoami` to troubleshoot.');
        }
        opts.log.warn('EWHOAMI', 'Unable to determine npm username from third-party registry, this command will likely fail soon!');
    }
}
exports.getNpmUsername = getNpmUsername;
//# sourceMappingURL=get-npm-username.js.map