"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitCommand = exports.factory = void 0;
const tslib_1 = require("tslib");
const core_1 = require("@lerna-lite/core");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const path_1 = tslib_1.__importDefault(require("path"));
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const write_json_file_1 = tslib_1.__importDefault(require("write-json-file"));
const LERNA_CLI_PKG_NAME = '@lerna-lite/cli';
function factory(argv) {
    return new InitCommand(argv);
}
exports.factory = factory;
class InitCommand extends core_1.Command {
    constructor(argv) {
        super(argv);
        /** command name */
        this.name = 'init';
        this.exact = false;
        this.lernaVersion = '';
    }
    get requiresGit() {
        return false;
    }
    runValidations() {
        this.logger.verbose(this.name, 'skipping validations');
    }
    runPreparations() {
        this.logger.verbose(this.name, 'skipping preparations');
    }
    initialize() {
        this.exact = this.options.exact;
        this.lernaVersion = this.options.lernaVersion;
        if (!this.gitInitialized()) {
            this.logger.info('', 'Initializing Git repository');
            return (0, core_1.exec)('git', ['init'], this.execOpts);
        }
    }
    async execute() {
        await this.ensurePackageJSON();
        await this.ensureLernaConfig();
        await this.ensurePackagesDir();
        this.logger.success('', 'Initialized Lerna files');
    }
    async ensurePackageJSON() {
        var _a;
        if (!this.project.manifest) {
            this.logger.info('', 'Creating package.json');
            // initialize with default indentation so write-pkg doesn't screw it up with tabs
            await (0, write_json_file_1.default)(path_1.default.join(this.project.rootPath, 'package.json'), { name: 'root', private: true }, { indent: 2 });
        }
        else {
            this.logger.info('', 'Updating package.json');
        }
        const rootPkg = this.project.manifest;
        let targetDependencies;
        if ((_a = rootPkg.dependencies) === null || _a === void 0 ? void 0 : _a[LERNA_CLI_PKG_NAME]) {
            // lerna is a dependency in the current project
            targetDependencies = rootPkg.dependencies;
        }
        else {
            // lerna is a devDependency or no dependency, yet
            if (!rootPkg.devDependencies) {
                // mutate raw JSON object
                rootPkg.set('devDependencies', {});
            }
            targetDependencies = rootPkg.devDependencies;
        }
        targetDependencies[LERNA_CLI_PKG_NAME] = this.exact ? this.lernaVersion : `^${this.lernaVersion}`;
        // add workspace packages in package.json when `useWorkspaces` enabled
        if (this.options.useWorkspaces && !rootPkg.workspaces) {
            rootPkg.workspaces = ['packages/*'];
        }
        return rootPkg.serialize();
    }
    ensureLernaConfig() {
        // config already defaulted to empty object in Project constructor
        const { config, version: projectVersion } = this.project;
        let version;
        if (this.options.independent) {
            version = 'independent';
        }
        else if (projectVersion) {
            version = projectVersion;
        }
        else {
            version = '0.0.0';
        }
        const logMessage = !projectVersion ? 'Creating lerna.json' : 'Updating lerna.json';
        this.logger.info('', logMessage);
        delete config[LERNA_CLI_PKG_NAME]; // no longer relevant
        if (this.exact) {
            // ensure --exact is preserved for future init commands
            const commandConfig = config.command || (config.command = {});
            const initConfig = commandConfig.init || (commandConfig.init = {});
            initConfig.exact = true;
        }
        const lernaConfig = {
            $schema: 'node_modules/@lerna-lite/cli/schemas/lerna-schema.json',
            version,
        };
        if (this.options.useWorkspaces) {
            lernaConfig.useWorkspaces = true;
        }
        else {
            lernaConfig.packages = ['packages/*'];
        }
        Object.assign(config, lernaConfig);
        return this.project.serializeConfig();
    }
    ensurePackagesDir() {
        this.logger.info('', 'Creating packages directory');
        return (0, p_map_1.default)(this.project.packageParentDirs, (dir) => fs_extra_1.default.mkdirp(dir));
    }
}
exports.InitCommand = InitCommand;
//# sourceMappingURL=init-command.js.map