"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.filterPackages = void 0;
const tslib_1 = require("tslib");
const multimatch_1 = tslib_1.__importDefault(require("multimatch"));
const util_1 = tslib_1.__importDefault(require("util"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const core_1 = require("@lerna-lite/core");
/**
 * Filters a list of packages, returning all packages that match the `include` glob[s]
 * and do not match the `exclude` glob[s].
 *
 * @param {import('@lerna/package').Package[]} packagesToFilter The packages to filter
 * @param {string[]} [include] A list of globs to match the package name against
 * @param {string[]} [exclude] A list of globs to filter the package name against
 * @param {boolean} [showPrivate] When false, filter out private packages
 * @param {boolean} [continueIfNoMatch] When true, do not throw if no package is matched
 * @throws when a given glob would produce an empty list of packages and `continueIfNoMatch` is not set.
 */
function filterPackages(packagesToFilter, include = [], exclude = [], showPrivate, continueIfNoMatch) {
    var _a;
    const filtered = new Set(packagesToFilter);
    const patterns = [].concat(arrify(include), negate(exclude));
    if (showPrivate === false) {
        for (const pkg of filtered) {
            if (pkg.private) {
                filtered.delete(pkg);
            }
        }
    }
    if (patterns.length) {
        npmlog_1.default.info('filter', JSON.stringify(patterns));
        if (!include.length) {
            // only excludes needs to select all items first
            // globstar is for matching scoped packages
            patterns.unshift('**');
        }
        const pnames = Array.from(filtered).map((pkg) => { var _a; return (_a = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _a !== void 0 ? _a : ''; });
        const chosen = new Set((0, multimatch_1.default)(pnames, patterns));
        for (const pkg of filtered) {
            if (!chosen.has((_a = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _a !== void 0 ? _a : '')) {
                filtered.delete(pkg);
            }
        }
        if (!filtered.size && !continueIfNoMatch) {
            throw new core_1.ValidationError('EFILTER', util_1.default.format('No packages remain after filtering', patterns));
        }
    }
    return Array.from(filtered);
}
exports.filterPackages = filterPackages;
/**
 * @param {string[]|string|undefined} thing
 */
function arrify(thing) {
    if (!thing) {
        return [];
    }
    if (!Array.isArray(thing)) {
        return [thing];
    }
    return thing;
}
/**
 * @param {string[]} patterns
 */
function negate(patterns) {
    return arrify(patterns).map((pattern) => `!${pattern}`);
}
//# sourceMappingURL=filter-packages.js.map