"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.runTopologically = void 0;
const tslib_1 = require("tslib");
const p_queue_1 = tslib_1.__importDefault(require("p-queue"));
const query_graph_1 = require("./query-graph");
/**
 * Run callback in maximally-saturated topological order.
 *
 * @template T
 * @param {import("@lerna/package").Package[]} packages List of `Package` instances
 * @param {(pkg: import("@lerna/package").Package) => Promise<T>} runner Callback to map each `Package` with
 * @param {TopologicalConfig} [options]
 * @returns {Promise<T[]>} when all executions complete
 */
function runTopologically(packages, runner, { concurrency, graphType, rejectCycles } = {}) {
    const queue = new p_queue_1.default({ concurrency });
    const graph = new query_graph_1.QueryGraph(packages, { graphType, rejectCycles });
    return new Promise((resolve, reject) => {
        const returnValues = [];
        const queueNextAvailablePackages = () => graph.getAvailablePackages().forEach(({ pkg, name }) => {
            graph.markAsTaken(name);
            queue
                .add(() => runner(pkg)
                .then((value) => returnValues.push(value))
                .then(() => graph.markAsDone(pkg))
                .then(() => queueNextAvailablePackages()))
                .catch(reject);
        });
        queueNextAvailablePackages();
        return queue.onIdle().then(() => resolve(returnValues));
    });
}
exports.runTopologically = runTopologically;
//# sourceMappingURL=run-topologically.js.map