"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isEmpty = exports.getComplexObjectValue = exports.deleteComplexObjectProp = void 0;
const tslib_1 = require("tslib");
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
/**
 * From a dot (.) notation path, find and delete a property within an object if found given a complex object path
 * @param {Object} object - object to search from
 * @param {String} path - complex object path to find descendant property from, must be a string with dot (.) notation
 * @param {String} [sourceName] - source name of which object name to delete the field from.
 */
function deleteComplexObjectProp(object, path, sourceName) {
    if (!object || !path) {
        return object;
    }
    const props = path.split('.');
    const lastProp = props.slice(-1).pop();
    return props.reduce((obj, prop) => {
        if (lastProp !== undefined && (obj === null || obj === void 0 ? void 0 : obj[prop]) !== undefined && prop === lastProp) {
            delete obj[prop];
            npmlog_1.default.verbose('mutation', `Removed "${path}" field from ${sourceName || 'n/a'}.`);
        }
        else {
            return obj === null || obj === void 0 ? void 0 : obj[prop];
        }
    }, object);
}
exports.deleteComplexObjectProp = deleteComplexObjectProp;
/**
 * From a dot (.) notation path, find and return a property within an object given a complex object path
 * Note that the object path does should not include the parent itself
 * for example if we want to get `address.zip` from `user` object, we would call `getComplexObjectValue(user, 'address.zip')`
 * @param object - object to search from
 * @param path - complex object path to find descendant property from, must be a string with dot (.) notation
 * @returns outputValue - the object property value found if any
 */
function getComplexObjectValue(object, path) {
    if (!object || !path) {
        return object;
    }
    return path.split('.').reduce((obj, prop) => obj === null || obj === void 0 ? void 0 : obj[prop], object);
}
exports.getComplexObjectValue = getComplexObjectValue;
/**
 * Check if an object is empty
 * @returns {Boolean}
 */
function isEmpty(obj) {
    return !obj || Object.keys(obj).length === 0;
}
exports.isEmpty = isEmpty;
//# sourceMappingURL=object-utils.js.map