"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.describeRefSync = exports.describeRef = void 0;
const tslib_1 = require("tslib");
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const child_process_1 = require("../child-process");
/**
 * Build `git describe` args.
 * @param {DescribeRefOptions} options
 * @param {boolean} [includeMergedTags]
 */
function getArgs(options, includeMergedTags) {
    let args = [
        'describe',
        // fallback to short sha if no tags located
        '--always',
        // always return full result, helps identify existing release
        '--long',
        // annotate if uncommitted changes present
        '--dirty',
        // prefer tags originating on upstream branch
        '--first-parent',
    ];
    if (options.match) {
        args.push('--match', options.match);
    }
    if (includeMergedTags) {
        // we want to consider all tags, also from merged branches
        args = args.filter((arg) => arg !== '--first-parent');
    }
    return args;
}
/**
 * @param {DescribeRefOptions} [options]
 * @param {boolean} [includeMergedTags]
 * @returns {Promise<DescribeRefFallbackResult|DescribeRefDetailedResult>}
 */
function describeRef(options = {}, includeMergedTags, dryRun = false) {
    const promise = (0, child_process_1.exec)('git', getArgs(options, includeMergedTags), options, dryRun);
    return promise.then(({ stdout } = { stdout: '' }) => {
        const result = parse(stdout, options.cwd);
        if (options === null || options === void 0 ? void 0 : options.match) {
            npmlog_1.default.verbose('git-describe', '%j => %j', options === null || options === void 0 ? void 0 : options.match, stdout);
        }
        if (stdout) {
            npmlog_1.default.silly('git-describe', 'parsed => %j', result);
        }
        return result;
    });
}
exports.describeRef = describeRef;
/**
 * @param {DescribeRefOptions} [options]
 * @param {boolean} [includeMergedTags]
 */
function describeRefSync(options = {}, includeMergedTags, dryRun = false) {
    const stdout = (0, child_process_1.execSync)('git', getArgs(options, includeMergedTags), options, dryRun);
    const result = parse(stdout, options.cwd);
    if (options === null || options === void 0 ? void 0 : options.match) {
        npmlog_1.default.verbose('git-describe.sync', '%j => %j', options === null || options === void 0 ? void 0 : options.match, stdout);
    }
    if (stdout) {
        npmlog_1.default.silly('git-describe', 'parsed => %j', result);
    }
    return result;
}
exports.describeRefSync = describeRefSync;
/**
 * Parse git output and return relevant metadata.
 * @param {string} stdout Result of `git describe`
 * @param {string} [cwd] Defaults to `process.cwd()`
 * @returns {DescribeRefFallbackResult|DescribeRefDetailedResult}
 */
function parse(stdout, cwd) {
    const minimalShaRegex = /^([0-9a-f]{7,40})(-dirty)?$/;
    // when git describe fails to locate tags, it returns only the minimal sha
    if (minimalShaRegex.test(stdout)) {
        // repo might still be dirty
        const [, sha, isDirty] = minimalShaRegex.exec(stdout);
        // count number of commits since beginning of time
        const refCount = (0, child_process_1.execSync)('git', ['rev-list', '--count', sha], { cwd });
        return { refCount, sha, isDirty: Boolean(isDirty) };
    }
    const [, lastTagName, lastVersion, refCount, sha, isDirty] = /^((?:.*@)?(.*))-(\d+)-g([0-9a-f]+)(-dirty)?$/.exec(stdout) || [];
    return { lastTagName, lastVersion, refCount, sha, isDirty: Boolean(isDirty) };
}
//# sourceMappingURL=describe-ref.js.map