"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeDiffPredicate = void 0;
const tslib_1 = require("tslib");
const globby_1 = tslib_1.__importDefault(require("globby"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const minimatch_1 = tslib_1.__importDefault(require("minimatch"));
const path_1 = tslib_1.__importDefault(require("path"));
const slash_1 = tslib_1.__importDefault(require("slash"));
const child_process_1 = require("../../../child-process");
/**
 * @param {string} committish
 * @param {import("@lerna/child-process").ExecOpts} execOpts
 * @param {string[]} ignorePatterns
 */
function makeDiffPredicate(committish, execOpts, ignorePatterns = [], diffOpts) {
    const ignoreFilters = new Set(ignorePatterns.map((p) => minimatch_1.default.filter(`!${p}`, {
        matchBase: true,
        // dotfiles inside ignored directories should also match
        dot: true,
    })));
    if (ignoreFilters.size) {
        npmlog_1.default.info('ignoring diff in paths matching', ignorePatterns.join(' '));
    }
    return function hasDiffSinceThatIsntIgnored(/** @type {import("@lerna/package-graph").PackageGraphNode} */ node) {
        const diff = diffSinceIn(committish, node.location, execOpts, diffOpts);
        if (diff === '') {
            npmlog_1.default.silly('', 'no diff found in %s', node.name);
            return false;
        }
        npmlog_1.default.silly('found diff in', diff);
        let changedFiles = diff.split('\n');
        if (ignoreFilters.size) {
            for (const ignored of ignoreFilters) {
                changedFiles = changedFiles.filter(ignored);
            }
        }
        if (changedFiles.length) {
            npmlog_1.default.verbose('filtered diff', changedFiles.join(' '));
        }
        else {
            npmlog_1.default.verbose('', 'no diff found in %s (after filtering)', node.name);
        }
        return changedFiles.length > 0;
    };
}
exports.makeDiffPredicate = makeDiffPredicate;
/**
 * @param {string} committish
 * @param {string} location
 * @param {import("@lerna/child-process").ExecOpts} execOpts
 */
function diffSinceIn(committish, location, execOpts, diffOpts) {
    const args = ['diff', '--name-only', committish];
    const formattedLocation = (0, slash_1.default)(path_1.default.relative(execOpts.cwd, location));
    if (formattedLocation) {
        // avoid same-directory path.relative() === ""
        let independentSubpackages = [];
        // optionally exclude sub-packages
        if (diffOpts === null || diffOpts === void 0 ? void 0 : diffOpts.independentSubpackages) {
            independentSubpackages = globby_1.default
                .sync('**/*/package.json', {
                cwd: formattedLocation,
                nodir: true,
                ignore: '**/node_modules/**',
            })
                .map((file) => `:^${formattedLocation}/${path_1.default.dirname(file)}`);
        }
        // avoid same-directory path.relative() === ""
        args.push('--', formattedLocation, ...independentSubpackages);
    }
    npmlog_1.default.silly('checking diff', formattedLocation);
    return (0, child_process_1.execSync)('git', args, execOpts);
}
//# sourceMappingURL=make-diff-predicate.js.map