"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.throwIfUncommitted = exports.mkThrowIfUncommitted = exports.throwIfReleased = exports.checkWorkingTree = void 0;
const collect_uncommitted_1 = require("./collect-uncommitted");
const describe_ref_1 = require("./describe-ref");
const validation_error_1 = require("../validation-error");
function checkWorkingTree({ cwd } = {}, dryRun = false) {
    let chain = Promise.resolve();
    chain = chain.then(() => (0, describe_ref_1.describeRef)({ cwd }, undefined, dryRun));
    // wrap each test separately to allow all applicable errors to be reported
    const tests = [
        // prevent duplicate versioning
        chain.then(throwIfReleased),
        // prevent publish of uncommitted changes
        chain.then(mkThrowIfUncommitted({ cwd }, dryRun)),
    ];
    // passes through result of describeRef() to aid composability
    return chain.then((result) => Promise.all(tests).then(() => result));
}
exports.checkWorkingTree = checkWorkingTree;
function throwIfReleased({ refCount }) {
    if (refCount === '0') {
        throw new validation_error_1.ValidationError('ERELEASED', 'The current commit has already been released. Please make new commits before continuing.');
    }
}
exports.throwIfReleased = throwIfReleased;
const EUNCOMMIT_MSG = 'Working tree has uncommitted changes, please commit or remove the following changes before continuing:\n';
function mkThrowIfUncommitted(options = {}, dryRun = false) {
    return function ({ isDirty }) {
        if (isDirty) {
            return (0, collect_uncommitted_1.collectUncommitted)(options, dryRun).then((uncommitted) => {
                throw new validation_error_1.ValidationError('EUNCOMMIT', `${EUNCOMMIT_MSG}${uncommitted.join('\n')}`);
            });
        }
    };
}
exports.mkThrowIfUncommitted = mkThrowIfUncommitted;
function throwIfUncommitted() {
    mkThrowIfUncommitted();
}
exports.throwIfUncommitted = throwIfUncommitted;
//# sourceMappingURL=check-working-tree.js.map