"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Project = void 0;
const tslib_1 = require("tslib");
const cosmiconfig_1 = require("cosmiconfig");
const dedent_1 = tslib_1.__importDefault(require("dedent"));
const globby_1 = tslib_1.__importDefault(require("globby"));
const glob_parent_1 = tslib_1.__importDefault(require("glob-parent"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const path_1 = tslib_1.__importDefault(require("path"));
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const load_json_file_1 = tslib_1.__importDefault(require("load-json-file"));
const write_json_file_1 = tslib_1.__importDefault(require("write-json-file"));
const package_1 = require("../package");
const apply_extends_1 = require("./lib/apply-extends");
const validation_error_1 = require("../validation-error");
const make_file_finder_1 = require("./lib/make-file-finder");
/**
 * A representation of the entire project managed by Lerna.
 *
 * Wherever the lerna.json file is located, that is the project root.
 * All package globs are rooted from this location.
 */
class Project {
    /**
     * @param {string} [cwd] Defaults to process.cwd()
     */
    constructor(cwd) {
        var _a, _b;
        const explorer = (0, cosmiconfig_1.cosmiconfigSync)('lerna', {
            searchPlaces: ['lerna.json', 'package.json'],
            transform(obj) {
                // cosmiconfig returns null when nothing is found
                if (!obj) {
                    return {
                        // No need to distinguish between missing and empty,
                        // saves a lot of noisy guards elsewhere
                        config: {},
                        configNotFound: true,
                        // path.resolve(".", ...) starts from process.cwd()
                        filepath: path_1.default.resolve(cwd || '.', 'lerna.json'),
                    };
                }
                obj.config = (0, apply_extends_1.applyExtends)(obj.config, path_1.default.dirname(obj.filepath));
                return obj;
            },
        });
        let loaded;
        try {
            loaded = explorer.search(cwd);
        }
        catch (err) {
            // redecorate JSON syntax errors, avoid debug dump
            if (err.name === 'JSONError') {
                throw new validation_error_1.ValidationError(err.name, err.message);
            }
            // re-throw other errors, could be ours or third-party
            throw err;
        }
        this.config = loaded === null || loaded === void 0 ? void 0 : loaded.config;
        this.configNotFound = loaded === null || loaded === void 0 ? void 0 : loaded.configNotFound;
        this.rootConfigLocation = (_a = loaded === null || loaded === void 0 ? void 0 : loaded.filepath) !== null && _a !== void 0 ? _a : '';
        this.rootPath = path_1.default.dirname((_b = loaded === null || loaded === void 0 ? void 0 : loaded.filepath) !== null && _b !== void 0 ? _b : '');
        npmlog_1.default.verbose('rootPath', this.rootPath);
    }
    /**
     * @param {string} [cwd] Defaults to process.cwd()
     */
    static getPackages(cwd) {
        return new Project(cwd).getPackages();
    }
    /**
     * @param {string} [cwd] Defaults to process.cwd()
     */
    static getPackagesSync(cwd) {
        return new Project(cwd).getPackagesSync();
    }
    get version() {
        return this.config.version;
    }
    set version(val) {
        this.config.version = val;
    }
    get packageConfigs() {
        var _a;
        if (this.config.useWorkspaces) {
            const workspaces = (_a = this.manifest) === null || _a === void 0 ? void 0 : _a.get('workspaces');
            if (!workspaces) {
                throw new validation_error_1.ValidationError('EWORKSPACES', (0, dedent_1.default) `
            Yarn workspaces need to be defined in the root package.json.
            See: https://github.com/lerna/lerna/blob/master/commands/bootstrap/README.md#--use-workspaces
          `);
            }
            const workspaceList = (workspaces.packages || workspaces);
            npmlog_1.default.verbose('project workspaces packages', workspaceList.join(' '));
            return workspaceList;
        }
        npmlog_1.default.verbose('project packages', (this.config.packages || [Project.PACKAGE_GLOB]).join(' '));
        return this.config.packages || [Project.PACKAGE_GLOB];
    }
    get packageParentDirs() {
        return this.packageConfigs
            .map(glob_parent_1.default)
            .map((parentDir) => path_1.default.resolve(this.rootPath, parentDir));
    }
    get manifest() {
        let manifest;
        try {
            const manifestLocation = path_1.default.join(this.rootPath, 'package.json');
            const packageJson = load_json_file_1.default.sync(manifestLocation);
            if (!packageJson.name) {
                // npm-lifecycle chokes if this is missing, so default like npm init does
                packageJson.name = path_1.default.basename(path_1.default.dirname(manifestLocation));
            }
            // Encapsulate raw JSON in Package instance
            manifest = new package_1.Package(packageJson, this.rootPath);
            // redefine getter to lazy-loaded value
            Object.defineProperty(this, 'manifest', {
                value: manifest,
            });
        }
        catch (err) {
            // redecorate JSON syntax errors, avoid debug dump
            if (err.name === 'JSONError') {
                throw new validation_error_1.ValidationError(err.name, err.message);
            }
            // try again next time
        }
        return manifest;
    }
    get licensePath() {
        let licensePath;
        try {
            const search = globby_1.default.sync(Project.LICENSE_GLOB, {
                cwd: this.rootPath,
                absolute: true,
                caseSensitiveMatch: false,
                // Project license is always a sibling of the root manifest
                deep: 0,
            });
            licensePath = search.shift();
            if (licensePath) {
                // POSIX results always need to be normalized
                licensePath = path_1.default.normalize(licensePath);
                // redefine getter to lazy-loaded value
                Object.defineProperty(this, 'licensePath', {
                    value: licensePath,
                });
            }
        }
        catch (err) {
            /* istanbul ignore next */
            throw new validation_error_1.ValidationError(err.name, err.message);
        }
        return licensePath;
    }
    get fileFinder() {
        const finder = (0, make_file_finder_1.makeFileFinder)(this.rootPath, this.packageConfigs);
        // redefine getter to lazy-loaded value
        Object.defineProperty(this, 'fileFinder', {
            value: finder,
        });
        return finder;
    }
    /**
     * @returns {Promise<Package[]>} A promise resolving to a list of Package instances
     */
    getPackages() {
        const mapper = (packageConfigPath) => {
            var _a;
            return (_a = (0, load_json_file_1.default)(packageConfigPath)) === null || _a === void 0 ? void 0 : _a.then((packageJson) => new package_1.Package(packageJson, path_1.default.dirname(packageConfigPath), this.rootPath));
        };
        return this.fileFinder('package.json', (filePaths) => (0, p_map_1.default)(filePaths, mapper, { concurrency: 50 }));
    }
    /**
     * @returns {Package[]} A list of Package instances
     */
    getPackagesSync() {
        return (0, make_file_finder_1.makeSyncFileFinder)(this.rootPath, this.packageConfigs)('package.json', (packageConfigPath) => {
            return new package_1.Package(load_json_file_1.default.sync(packageConfigPath), path_1.default.dirname(packageConfigPath), this.rootPath);
        });
    }
    getPackageLicensePaths() {
        return this.fileFinder(Project.LICENSE_GLOB, null, { caseSensitiveMatch: false });
    }
    isIndependent() {
        return this.version === 'independent';
    }
    serializeConfig() {
        // TODO: might be package.json prop
        return (0, write_json_file_1.default)(this.rootConfigLocation, this.config, { indent: 2, detectIndent: true }).then(() => this.rootConfigLocation);
    }
}
exports.Project = Project;
Project.PACKAGE_GLOB = 'packages/*';
Project.LICENSE_GLOB = 'LICEN{S,C}E{,.*}';
Project.PACKAGE_GLOB = 'packages/*';
Project.LICENSE_GLOB = 'LICEN{S,C}E{,.*}';
//# sourceMappingURL=project.js.map