"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeSyncFileFinder = exports.makeFileFinder = void 0;
const tslib_1 = require("tslib");
const globby_1 = tslib_1.__importDefault(require("globby"));
const path_1 = tslib_1.__importDefault(require("path"));
const p_map_1 = tslib_1.__importDefault(require("p-map"));
const validation_error_1 = require("../../validation-error");
/**
 * @param {string[]} results
 */
function normalize(results) {
    return results.map((fp) => path_1.default.normalize(fp));
}
function getGlobOpts(rootPath, packageConfigs) {
    const globOpts = {
        cwd: rootPath,
        absolute: true,
        expandDirectories: false,
        followSymbolicLinks: false,
    };
    if (packageConfigs.some((cfg) => cfg.indexOf('**') > -1)) {
        if (packageConfigs.some((cfg) => cfg.indexOf('node_modules') > -1)) {
            throw new validation_error_1.ValidationError('EPKGCONFIG', 'An explicit node_modules package path does not allow globstars (**)');
        }
        globOpts.ignore = [
            // allow globs like "packages/**",
            // but avoid picking up node_modules/**/package.json
            '**/node_modules/**',
        ];
    }
    return globOpts;
}
function makeFileFinder(rootPath, packageConfigs) {
    const globOpts = getGlobOpts(rootPath, packageConfigs);
    return (fileName, fileMapper, customGlobOpts) => {
        const options = Object.assign({}, customGlobOpts, globOpts);
        const promise = (0, p_map_1.default)(Array.from(packageConfigs).sort(), (globPath) => {
            let chain = (0, globby_1.default)(path_1.default.posix.join(globPath, fileName), options);
            // fast-glob does not respect pattern order, so we re-sort by absolute path
            chain = chain.then((results) => results.sort());
            // POSIX results always need to be normalized
            chain = chain.then(normalize);
            if (fileMapper) {
                chain = chain.then(fileMapper);
            }
            return chain;
        }, { concurrency: 4 });
        // always flatten the results
        return promise.then((results) => results.reduce((acc, result) => acc.concat(result), []));
    };
}
exports.makeFileFinder = makeFileFinder;
function makeSyncFileFinder(rootPath, packageConfigs) {
    const globOpts = getGlobOpts(rootPath, packageConfigs);
    return (fileName, fileMapper, customGlobOpts) => {
        const options = Object.assign({}, customGlobOpts, globOpts);
        const patterns = packageConfigs.map((globPath) => path_1.default.posix.join(globPath, fileName)).sort();
        let results = globby_1.default.sync(patterns, options);
        // POSIX results always need to be normalized
        results = normalize(results);
        /* istanbul ignore else */
        if (fileMapper) {
            results = results.map(fileMapper);
        }
        return results;
    };
}
exports.makeSyncFileFinder = makeSyncFileFinder;
//# sourceMappingURL=make-file-finder.js.map