"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Package = void 0;
const tslib_1 = require("tslib");
const load_json_file_1 = tslib_1.__importDefault(require("load-json-file"));
const npm_package_arg_1 = tslib_1.__importDefault(require("npm-package-arg"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const path_1 = tslib_1.__importDefault(require("path"));
const write_pkg_1 = tslib_1.__importDefault(require("write-pkg"));
// symbol used to 'hide' internal state
const PKG = Symbol('pkg');
// private fields
const _location = Symbol('location');
const _resolved = Symbol('resolved');
const _rootPath = Symbol('rootPath');
const _scripts = Symbol('scripts');
const _contents = Symbol('contents');
/**
 * @param {import('npm-package-arg').Result} result
 */
function binSafeName({ name, scope }) {
    return scope ? name.substring(scope.length + 1) : name;
}
// package.json files are not that complicated, so this is intentionally naïve
function shallowCopy(json) {
    return Object.keys(json).reduce((obj, key) => {
        const val = json[key];
        /* istanbul ignore if */
        if (Array.isArray(val)) {
            obj[key] = val.slice();
        }
        else if (val && typeof val === 'object') {
            obj[key] = Object.assign({}, val);
        }
        else {
            obj[key] = val;
        }
        return obj;
    }, {});
}
/**
 * Lerna's internal representation of a local package, with
 * many values resolved directly from the original JSON.
 */
class Package {
    /**
     * Create a Package instance from parameters, possibly reusing existing instance.
     * @param {string|Package|RawManifest} ref A path to a package.json file, Package instance, or JSON object
     * @param {string} [dir] If `ref` is a JSON object, this is the location of the manifest
     * @returns {Package}
     */
    static lazy(ref, dir = '.') {
        if (typeof ref === 'string') {
            const location = path_1.default.resolve(path_1.default.basename(ref) === 'package.json' ? path_1.default.dirname(ref) : ref);
            const manifest = load_json_file_1.default.sync(path_1.default.join(location, 'package.json'));
            return new Package(manifest, location);
        }
        // don't use instanceof because it fails across nested module boundaries
        if ('__isLernaPackage' in ref) {
            return ref;
        }
        // assume ref is a json object
        return new Package(ref, dir);
    }
    /**
     * @param {RawManifest} pkg
     * @param {string} location
     * @param {string} [rootPath]
     */
    constructor(pkg, location, rootPath = location) {
        var _a, _b;
        this._id = '';
        this.licensePath = '';
        this.localDependencies = new Map();
        // npa will throw an error if the name is invalid
        const resolved = npm_package_arg_1.default.resolve((_a = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _a !== void 0 ? _a : '', `file:${path_1.default.relative(rootPath, location)}`, rootPath);
        this.name = (_b = pkg === null || pkg === void 0 ? void 0 : pkg.name) !== null && _b !== void 0 ? _b : '';
        this[PKG] = pkg;
        // omit raw pkg from default util.inspect() output, but preserve internal mutability
        Object.defineProperty(this, PKG, { enumerable: false, writable: true });
        this[_location] = location;
        this[_resolved] = resolved;
        this[_rootPath] = rootPath;
        this[_scripts] = { ...pkg.scripts };
    }
    // readonly getters
    get location() {
        return this[_location];
    }
    get private() {
        return Boolean(this[PKG].private);
    }
    get resolved() {
        return this[_resolved];
    }
    get rootPath() {
        return this[_rootPath];
    }
    get scripts() {
        return this[_scripts];
    }
    get bin() {
        const pkg = this[PKG];
        return typeof pkg.bin === 'string'
            ? { [binSafeName(this.resolved)]: pkg.bin }
            : Object.assign({}, pkg.bin);
    }
    get binLocation() {
        return path_1.default.join(this.location, 'node_modules', '.bin');
    }
    /** alias to pkg getter (to avoid calling duplicate prop like `node.pkg.pkg` in which node is PackageGraphNode) */
    get manifest() {
        return this[PKG];
    }
    get manifestLocation() {
        return path_1.default.join(this.location, 'package.json');
    }
    get nodeModulesLocation() {
        return path_1.default.join(this.location, 'node_modules');
    }
    // eslint-disable-next-line class-methods-use-this
    get __isLernaPackage() {
        // safer than instanceof across module boundaries
        return true;
    }
    // accessors
    get version() {
        return this[PKG].version;
    }
    set version(version) {
        this[PKG].version = version;
    }
    get workspaces() {
        return this[PKG].workspaces;
    }
    set workspaces(workspaces) {
        this[PKG].workspaces = workspaces;
    }
    get contents() {
        // if modified with setter, use that value
        if (this[_contents]) {
            return this[_contents];
        }
        // if provided by pkg.publishConfig.directory value
        if (this[PKG].publishConfig && this[PKG].publishConfig.directory) {
            return path_1.default.join(this.location, this[PKG].publishConfig.directory);
        }
        // default to package root
        return this.location;
    }
    set contents(subDirectory) {
        this[_contents] = path_1.default.join(this.location, subDirectory);
    }
    // 'live' collections
    get dependencies() {
        return this[PKG].dependencies;
    }
    get devDependencies() {
        return this[PKG].devDependencies;
    }
    get optionalDependencies() {
        return this[PKG].optionalDependencies;
    }
    get peerDependencies() {
        return this[PKG].peerDependencies;
    }
    get pkg() {
        return this[PKG];
    }
    /**
     * Map-like retrieval of arbitrary values
     * @template {keyof RawManifest} K
     * @param {K} key field name to retrieve value
     * @returns {RawManifest[K]} value stored under key, if present
     */
    get(key) {
        return this[PKG][key];
    }
    /**
     * Map-like storage of arbitrary values
     * @template {keyof RawManifest} K
     * @param {T} key field name to store value
     * @param {RawManifest[K]} val value to store
     * @returns {Package} instance for chaining
     */
    set(key, val) {
        this[PKG][key] = val;
        return this;
    }
    /**
     * Provide shallow copy for munging elsewhere
     * @returns {Object}
     */
    toJSON() {
        return shallowCopy(this[PKG]);
    }
    /**
     * Refresh internal state from disk (e.g., changed by external lifecycles)
     */
    refresh() {
        return (0, load_json_file_1.default)(this.manifestLocation).then((pkg) => {
            this[PKG] = pkg;
            return this;
        });
    }
    /**
     * Write manifest changes to disk
     * @returns {Promise} resolves when write finished
     */
    serialize() {
        return (0, write_pkg_1.default)(this.manifestLocation, this[PKG]).then(() => this);
    }
    /**
     * Mutate given dependency (could be local/external) spec according to type
     * @param {String} pkgName - package name
     * @param {Object} resolved npa metadata
     */
    removeDependencyWorkspaceProtocolPrefix(pkgName, resolved) {
        var _a, _b;
        const depName = resolved.name;
        const workspaceSpec = (_a = resolved === null || resolved === void 0 ? void 0 : resolved.workspaceSpec) !== null && _a !== void 0 ? _a : '';
        const localDependencies = this.retrievePackageDependencies(depName);
        const inspectDependencies = [localDependencies];
        // package could be found in both a local dependencies and peerDependencies, so we need to include it when found
        if ((_b = this.peerDependencies) === null || _b === void 0 ? void 0 : _b[depName]) {
            inspectDependencies.push(this.peerDependencies);
        }
        for (const depCollection of inspectDependencies) {
            if (depCollection &&
                (resolved.registry || resolved.type === 'directory') &&
                /^(workspace:)+(.*)$/.test(workspaceSpec)) {
                if (workspaceSpec) {
                    if (resolved.fetchSpec === 'latest' || resolved.fetchSpec === '') {
                        npmlog_1.default.error(`publish`, [
                            `Your package named "${pkgName}" has external dependencies not handled by Lerna-Lite and without workspace version suffix, `,
                            `we recommend using defined versions with workspace protocol. `,
                            `Your dependency is currently being published with "${depName}": "${resolved.fetchSpec}".`,
                        ].join(''));
                    }
                    depCollection[depName] = resolved.fetchSpec;
                }
            }
        }
    }
    /**
     * Mutate local dependency spec according to type
     * @param {Object} resolved npa metadata
     * @param {String} depVersion semver
     * @param {String} savePrefix npm_config_save_prefix
     * @param {Boolean} [workspaceStrictMatch] - are we using `workspace:` protocol strict match?
     * @param {String} [updatedByCommand] - which command called this update?
     */
    updateLocalDependency(resolved, depVersion, savePrefix, allowPeerDependenciesUpdate = false, workspaceStrictMatch = true, updatedByCommand) {
        var _a, _b;
        const depName = resolved.name;
        const localDependencies = this.retrievePackageDependencies(depName);
        const updatingDependencies = [localDependencies];
        // when we have peer dependencies, we might need to perform certain things
        if ((_a = this.peerDependencies) === null || _a === void 0 ? void 0 : _a[depName]) {
            // when user allows peer bump and is a regular semver version, we'll push it to the array of dependencies to potentially bump
            // however we won't when the semver has a range with operator, ie this would bump ("^2.0.0") but these would not (">=2.0.0" or "workspace:<2.0.0" or "workspace:*")
            // prettier-ignore
            if (allowPeerDependenciesUpdate && /^(workspace:)?[~^]?[\d\.]+([\-]+[\w\.\-\+]+)*$/i.test(this.peerDependencies[depName] || '')) {
                updatingDependencies.push(this.peerDependencies);
            }
            // when peer bump is disabled, we could end up with peerDependencies not being reviewed
            // and some might still have the `workspace:` prefix so make sure to remove any of these prefixes
            else if (updatedByCommand === 'publish' && this.peerDependencies[depName].startsWith('workspace:')) {
                this.peerDependencies[depName] = this.peerDependencies[depName].replace('workspace:', '');
            }
        }
        for (const depCollection of updatingDependencies) {
            if (depCollection && (resolved.registry || resolved.type === 'directory')) {
                // a version (1.2.3) OR range (^1.2.3) OR directory (file:../foo-pkg)
                depCollection[depName] = `${savePrefix}${depVersion}`;
                // when using explicit `workspace:` protocol
                if (resolved.workspaceSpec) {
                    const workspaceSpec = (_b = resolved === null || resolved === void 0 ? void 0 : resolved.workspaceSpec) !== null && _b !== void 0 ? _b : '';
                    const [_, _wsTxt, operatorPrefix, rangePrefix, semver] = workspaceSpec.match(/^(workspace:)?([<>=]{0,2})?([*|~|^])?(.*)$/) || [];
                    if (operatorPrefix) {
                        // package with range operator should never be bumped, we'll use same version range but without prefix "workspace:>=1.2.3" will assign ">=1.2.3"
                        depCollection[depName] = `${operatorPrefix}${rangePrefix || ''}${semver}`;
                    }
                    else if (workspaceStrictMatch) {
                        // with workspace in strict mode we might have empty range prefix like "workspace:1.2.3"
                        depCollection[depName] = `${rangePrefix || ''}${depVersion}`;
                    }
                    if (updatedByCommand === 'publish') {
                        // when publishing, workspace protocol will be transformed to semver range
                        // e.g.: considering version is `1.2.3` and we have `workspace:*` it will be converted to "^1.2.3" or to "1.2.3" with strict match range enabled
                        if (workspaceStrictMatch) {
                            if (workspaceSpec === 'workspace:*') {
                                depCollection[depName] = depVersion; // (*) exact range, "1.5.0"
                            }
                            else if (workspaceSpec === 'workspace:~') {
                                depCollection[depName] = `~${depVersion}`; // (~) patch range, "~1.5.0"
                            }
                            else if (workspaceSpec === 'workspace:^') {
                                depCollection[depName] = `^${depVersion}`; // (^) minor range, "^1.5.0"
                            }
                        }
                        // anything else will fall under what Lerna previously found to be the version,
                        // typically by this line: depCollection[depName] = `${savePrefix}${depVersion}`;
                    }
                    else {
                        // when versioning we'll only bump workspace protocol that have semver range like `workspace:^1.2.3`
                        // any other workspace will remain the same in `package.json` file, for example `workspace:^`
                        // keep target workspace or bump when it's a workspace semver range (like `workspace:^1.2.3`)
                        depCollection[depName] = /^workspace:[*|~|^]{1}$/.test(workspaceSpec)
                            ? resolved.workspaceSpec // target like `workspace:^` => `workspace:^` (remains untouched in package.json)
                            : `workspace:${depCollection[depName]}`; // range like `workspace:^1.2.3` => `workspace:^1.3.3` (bump minor example)
                    }
                }
            }
            else if (resolved.gitCommittish) {
                // a git url with matching committish (#v1.2.3 or #1.2.3)
                const [tagPrefix] = /^\D*/.exec(resolved.gitCommittish);
                // update committish
                const { hosted } = resolved; // take that, lint!
                hosted.committish = `${tagPrefix}${depVersion}`;
                // always serialize the full url (identical to previous resolved.saveSpec)
                depCollection[depName] = hosted.toString({ noGitPlus: false, noCommittish: false });
            }
            else if (resolved.gitRange) {
                // a git url with matching gitRange (#semver:^1.2.3)
                const { hosted } = resolved; // take that, lint!
                hosted.committish = `semver:${savePrefix}${depVersion}`;
                // always serialize the full url (identical to previous resolved.saveSpec)
                depCollection[depName] = hosted.toString({ noGitPlus: false, noCommittish: false });
            }
        }
    }
    /**
     * Retrieve the dependencies collection which contain the dependency name provided, we'll search in all type of dependencies/devDependencies/...
     * @param {String} depName - dependency name
     * @returns {Array<String>} - array of dependencies that contains the dependency name provided
     */
    retrievePackageDependencies(depName) {
        // first, try runtime dependencies
        let depCollection = this.dependencies;
        // try optionalDependencies if that didn't work
        if (!depCollection || !depCollection[depName]) {
            depCollection = this.optionalDependencies;
        }
        // fall back to devDependencies
        if (!depCollection || !depCollection[depName]) {
            depCollection = this.devDependencies;
        }
        return depCollection;
    }
}
exports.Package = Package;
//# sourceMappingURL=package.js.map