import { CommandType, NpaResolveResult, RawManifest } from './models';
/**
 * Lerna's internal representation of a local package, with
 * many values resolved directly from the original JSON.
 */
export declare class Package {
    _id: string;
    name: string;
    licensePath: string;
    localDependencies: Map<string, any>;
    /**
     * Create a Package instance from parameters, possibly reusing existing instance.
     * @param {string|Package|RawManifest} ref A path to a package.json file, Package instance, or JSON object
     * @param {string} [dir] If `ref` is a JSON object, this is the location of the manifest
     * @returns {Package}
     */
    static lazy(ref: string | Package | RawManifest, dir?: string): Package;
    /**
     * @param {RawManifest} pkg
     * @param {string} location
     * @param {string} [rootPath]
     */
    constructor(pkg: RawManifest, location: string, rootPath?: string);
    get location(): string;
    get private(): boolean;
    get resolved(): {
        name: string;
        bin: any;
        scope: any;
    };
    get rootPath(): string;
    get scripts(): any;
    get bin(): any;
    get binLocation(): string;
    /** alias to pkg getter (to avoid calling duplicate prop like `node.pkg.pkg` in which node is PackageGraphNode) */
    get manifest(): RawManifest;
    get manifestLocation(): string;
    get nodeModulesLocation(): string;
    get __isLernaPackage(): boolean;
    get version(): string;
    set version(version: string);
    get workspaces(): string[] | {
        packages: string[];
    };
    set workspaces(workspaces: string[] | {
        packages: string[];
    });
    get contents(): string;
    set contents(subDirectory: string);
    get dependencies(): {
        [depName: string]: string;
    };
    get devDependencies(): {
        [depName: string]: string;
    };
    get optionalDependencies(): {
        [depName: string]: string;
    };
    get peerDependencies(): {
        [depName: string]: string;
    };
    get pkg(): Package;
    /**
     * Map-like retrieval of arbitrary values
     * @template {keyof RawManifest} K
     * @param {K} key field name to retrieve value
     * @returns {RawManifest[K]} value stored under key, if present
     */
    get<K extends keyof RawManifest>(key: string): K;
    /**
     * Map-like storage of arbitrary values
     * @template {keyof RawManifest} K
     * @param {T} key field name to store value
     * @param {RawManifest[K]} val value to store
     * @returns {Package} instance for chaining
     */
    set(key: string, val: RawManifest[keyof RawManifest]): Package;
    /**
     * Provide shallow copy for munging elsewhere
     * @returns {Object}
     */
    toJSON(): {};
    /**
     * Refresh internal state from disk (e.g., changed by external lifecycles)
     */
    refresh(): Promise<this>;
    /**
     * Write manifest changes to disk
     * @returns {Promise} resolves when write finished
     */
    serialize(): Promise<this>;
    /**
     * Mutate given dependency (could be local/external) spec according to type
     * @param {String} pkgName - package name
     * @param {Object} resolved npa metadata
     */
    removeDependencyWorkspaceProtocolPrefix(pkgName: string, resolved: NpaResolveResult): void;
    /**
     * Mutate local dependency spec according to type
     * @param {Object} resolved npa metadata
     * @param {String} depVersion semver
     * @param {String} savePrefix npm_config_save_prefix
     * @param {Boolean} [workspaceStrictMatch] - are we using `workspace:` protocol strict match?
     * @param {String} [updatedByCommand] - which command called this update?
     */
    updateLocalDependency(resolved: NpaResolveResult, depVersion: string, savePrefix: string, allowPeerDependenciesUpdate?: boolean, workspaceStrictMatch?: boolean, updatedByCommand?: CommandType): void;
    /**
     * Retrieve the dependencies collection which contain the dependency name provided, we'll search in all type of dependencies/devDependencies/...
     * @param {String} depName - dependency name
     * @returns {Array<String>} - array of dependencies that contains the dependency name provided
     */
    private retrievePackageDependencies;
}
