import { CyclicPackageGraphNode, PackageGraphNode } from './lib';
import { Package } from '../package';
/**
 * A graph of packages in the current project.
 *
 * @extends {Map<string, PackageGraphNode>}
 */
export declare class PackageGraph extends Map<string, PackageGraphNode> {
    /**
     * @param {Package[]} packages - An array of Packages to build the graph out of.
     * @param {'allDependencies' | 'allPlusPeerDependencies' | 'dependencies'} [graphType]
     *    Pass "dependencies" to create a graph of only dependencies,
     *    excluding the devDependencies that would normally be included.
     * @param {boolean|'auto'|'force'|'explicit'} [localDependencies] Treatment of local sibling dependencies, default "auto"
     */
    constructor(packages: Package[], graphType?: 'allDependencies' | 'allPlusPeerDependencies' | 'dependencies', localDependencies?: boolean | 'auto' | 'force' | 'explicit' | 'forceLocal');
    get rawPackageList(): Package[];
    /**
     * Takes a list of Packages and returns a list of those same Packages with any Packages
     * they depend on. i.e if packageA depended on packageB `graph.addDependencies([packageA])`
     * would return [packageA, packageB].
     *
     * @param {Package[]} filteredPackages - The packages to include dependencies for.
     */
    addDependencies(filteredPackages: Package[]): Package[];
    /**
     * Takes a list of Packages and returns a list of those same Packages with any Packages
     * that depend on them. i.e if packageC depended on packageD `graph.addDependents([packageD])`
     * would return [packageD, packageC].
     *
     * @param {Package[]} filteredPackages - The packages to include dependents for.
     */
    addDependents(filteredPackages: Package[]): Package[];
    /**
     * Extends a list of packages by traversing on a given property, which must refer to a
     * `PackageGraphNode` property that is a collection of `PackageGraphNode`s.
     * Returns input packages with any additional packages found by traversing `nodeProp`.
     *
     * @param {Package[]} packageList - The list of packages to extend
     * @param {'localDependencies'|'localDependents'} nodeProp - The property on `PackageGraphNode` used to traverse
     */
    extendList(packageList: Package[], nodeProp: 'localDependencies' | 'localDependents'): Package[];
    /**
     * Return a tuple of cycle paths and nodes.
     *
     * @deprecated Use collapseCycles instead.
     *
     * @param {boolean} rejectCycles Whether or not to reject cycles
     * @returns {[Set<string[]>, Set<PackageGraphNode>]}
     */
    partitionCycles(rejectCycles?: boolean): (Set<PackageGraphNode> | Set<string[]>)[];
    /**
     * Returns the cycles of this graph. If two cycles share some elements, they will
     * be returned as a single cycle.
     *
     * @param {boolean} rejectCycles Whether or not to reject cycles
     * @returns {Set<CyclicPackageGraphNode>}
     */
    collapseCycles(rejectCycles?: boolean): Set<CyclicPackageGraphNode>;
    /**
     * Remove cycle nodes.
     *
     * @deprecated Spread set into prune() instead.
     *
     * @param {Set<PackageGraphNode>} cycleNodes
     */
    pruneCycleNodes(cycleNodes: Set<PackageGraphNode>): void;
    /**
     * Remove all candidate nodes.
     * @param {PackageGraphNode[]} candidates
     */
    prune(...candidates: PackageGraphNode[]): void;
    /**
     * Delete by value (instead of key), as well as removing pointers
     * to itself in the other node's internal collections.
     * @param {PackageGraphNode} candidateNode instance to remove
     */
    remove(candidateNode: PackageGraphNode): void;
}
