"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Command = void 0;
const tslib_1 = require("tslib");
const clone_deep_1 = tslib_1.__importDefault(require("clone-deep"));
const dedent_1 = tslib_1.__importDefault(require("dedent"));
const execa_1 = tslib_1.__importDefault(require("execa"));
const is_ci_1 = tslib_1.__importDefault(require("is-ci"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const os_1 = tslib_1.__importDefault(require("os"));
const clean_stack_1 = require("./utils/clean-stack");
const child_process_1 = require("./child-process");
const log_package_error_1 = require("./utils/log-package-error");
const warn_if_hanging_1 = require("./utils/warn-if-hanging");
const write_log_file_1 = require("./utils/write-log-file");
const project_1 = require("./project/project");
const validation_error_1 = require("./validation-error");
const package_graph_1 = require("./package-graph/package-graph");
// maxBuffer value for running exec
const DEFAULT_CONCURRENCY = os_1.default.cpus().length;
class Command {
    constructor(_argv) {
        this.commandName = '';
        npmlog_1.default.pause();
        npmlog_1.default.heading = 'lerna-lite';
        const argv = (0, clone_deep_1.default)(_argv);
        npmlog_1.default.silly('argv', argv.toString());
        // 'FooCommand' => 'foo'
        this.commandName = this.constructor.name.replace(/Command$/, '').toLowerCase();
        // composed commands are called from other commands, like publish -> version
        this.composed = typeof argv.composed === 'string' && argv.composed !== this.commandName;
        if (!this.composed) {
            // composed commands have already logged the lerna version
            npmlog_1.default.notice('cli', `v${argv.lernaVersion}`);
        }
        // launch the command
        let runner = new Promise((resolve, reject) => {
            // run everything inside a Promise chain
            let chain = Promise.resolve();
            chain = chain.then(() => (this.project = new project_1.Project(argv.cwd)));
            chain = chain.then(() => this.configureEnvironment());
            chain = chain.then(() => this.configureOptions());
            chain = chain.then(() => this.configureProperties());
            chain = chain.then(() => this.configureLogging());
            chain = chain.then(() => this.runValidations());
            chain = chain.then(() => this.runPreparations());
            chain = chain.then(() => this.runCommand());
            chain.then((result) => {
                (0, warn_if_hanging_1.warnIfHanging)();
                resolve(result);
            }, (err) => {
                if (err.pkg) {
                    // Cleanly log specific package error details
                    (0, log_package_error_1.logPackageError)(err, this.options.stream);
                }
                else if (err.name !== 'ValidationError') {
                    // npmlog does some funny stuff to the stack by default,
                    // so pass it directly to avoid duplication.
                    npmlog_1.default.error('', (0, clean_stack_1.cleanStack)(err, this.constructor.name));
                }
                // ValidationError does not trigger a log dump, nor do external package errors
                if (err.name !== 'ValidationError' && !err.pkg) {
                    (0, write_log_file_1.writeLogFile)(this.project.rootPath);
                }
                (0, warn_if_hanging_1.warnIfHanging)();
                // error code is handled by cli.fail()
                reject(err);
                if (err.name === 'ValidationError') {
                    npmlog_1.default.error(err.code, (err.body && err.body.error) || err.message);
                }
            });
        });
        // passed via yargs context in tests, never actual CLI
        /* istanbul ignore else */
        if (argv.onResolved || argv.onRejected) {
            runner = runner.then(argv.onResolved, argv.onRejected);
            // when nested, never resolve inner with outer callbacks
            delete argv.onResolved; // eslint-disable-line no-param-reassign
            delete argv.onRejected; // eslint-disable-line no-param-reassign
        }
        // 'hide' irrelevant argv keys from options
        for (const key of ['cwd', '$0']) {
            Object.defineProperty(argv, key, { enumerable: false });
        }
        Object.defineProperty(this, 'argv', {
            value: Object.freeze(argv),
        });
        Object.defineProperty(this, 'runner', {
            value: runner,
        });
    }
    // proxy 'Promise' methods to 'private' instance
    then(onResolved, onRejected) {
        var _a;
        return (_a = this.runner) === null || _a === void 0 ? void 0 : _a.then(onResolved, onRejected);
    }
    /* istanbul ignore next */
    catch(onRejected) {
        var _a;
        return (_a = this.runner) === null || _a === void 0 ? void 0 : _a.catch(onRejected);
    }
    get requiresGit() {
        return true;
    }
    // Override this to inherit config from another command.
    // For example `changed` inherits config from `publish`.
    get otherCommandConfigs() {
        return [];
    }
    configureEnvironment() {
        let loglevel;
        let progress;
        /* istanbul ignore next */
        if (is_ci_1.default || !process.stderr.isTTY || process.env.TERM === 'dumb') {
            npmlog_1.default.disableColor();
            progress = false;
        }
        else if (!process.stdout.isTTY) {
            // stdout is being piped, don't log non-errors or progress bars
            progress = false;
            loglevel = 'error';
        }
        else if (process.stderr.isTTY) {
            npmlog_1.default.enableColor();
            npmlog_1.default.enableUnicode();
        }
        Object.defineProperty(this, 'envDefaults', {
            value: {
                ci: is_ci_1.default,
                progress,
                loglevel,
            },
        });
    }
    configureOptions() {
        // Command config object normalized to 'command' namespace
        const commandConfig = this.project.config.command || {};
        // The current command always overrides otherCommandConfigs
        const overrides = [this.commandName, ...this.otherCommandConfigs].map((key) => commandConfig[key]);
        this.options = defaultOptions(
        // CLI flags, which if defined overrule subsequent values
        this.argv, 
        // Namespaced command options from `lerna.json`
        ...overrides, 
        // Global options from `lerna.json`
        this.project.config, 
        // Environmental defaults prepared in previous step
        this.envDefaults);
        if (this.options.verbose && this.options.loglevel !== 'silly') {
            this.options.loglevel = 'verbose';
        }
    }
    configureProperties() {
        var _a;
        const { concurrency, sort, maxBuffer } = this.options;
        this.concurrency = Math.max(1, +concurrency || DEFAULT_CONCURRENCY);
        this.toposort = sort === undefined || sort;
        this.execOpts = {
            cwd: (_a = this.project.rootPath) !== null && _a !== void 0 ? _a : '',
            maxBuffer,
        };
    }
    configureLogging() {
        const { loglevel } = this.options;
        if (loglevel) {
            npmlog_1.default.level = loglevel;
        }
        // handle log.success()
        npmlog_1.default.addLevel('success', 3001, { fg: 'green', bold: true });
        // create logger that subclasses use
        Object.defineProperty(this, 'logger', {
            value: npmlog_1.default.newGroup(this.commandName),
        });
        // emit all buffered logs at configured level and higher
        npmlog_1.default.resume();
    }
    enableProgressBar() {
        /* istanbul ignore next */
        if (this.options.progress !== false) {
            npmlog_1.default.enableProgress();
        }
    }
    gitInitialized() {
        var _a;
        const opts = {
            cwd: (_a = this.project.rootPath) !== null && _a !== void 0 ? _a : '',
            // don't throw, just want boolean
            reject: false,
            // only return code, no stdio needed
            stdio: 'ignore',
        };
        const gitCommand = 'git';
        const gitArgs = ['rev-parse'];
        if (this.options.dryRun) {
            (0, child_process_1.logExecCommand)(gitCommand, gitArgs);
            return true;
        }
        return execa_1.default.sync(gitCommand, gitArgs, opts).exitCode === 0;
    }
    runValidations() {
        if ((this.options.since !== undefined || this.requiresGit) && !this.gitInitialized()) {
            throw new validation_error_1.ValidationError('ENOGIT', 'The git binary was not found, or this is not a git repository.');
        }
        if (!this.project.manifest) {
            throw new validation_error_1.ValidationError('ENOPKG', 'No `package.json` file found, make sure it exist in the root of your project.');
        }
        if (this.project.configNotFound) {
            throw new validation_error_1.ValidationError('ENOLERNA', '`lerna.json` does not exist, have you run `lerna init`?');
        }
        if (!this.project.version) {
            throw new validation_error_1.ValidationError('ENOVERSION', 'Required property `version` does not exist in `lerna.json`, make sure to provide one of two modes (fixed or independent). For example "version": "independent" OR "version": "1.0.0"');
        }
        if (this.options.independent && !this.project.isIndependent()) {
            throw new validation_error_1.ValidationError('EVERSIONMODE', (0, dedent_1.default) `
          You ran lerna-lite with --independent or -i, but the repository is not set to independent mode.
          To use independent mode you need to set lerna.json's "version" property to "independent".
          Then you won't need to pass the --independent or -i flags.
        `);
        }
    }
    runPreparations() {
        if (!this.composed && this.project.isIndependent()) {
            // composed commands have already logged the independent status
            npmlog_1.default.info('versioning', 'independent');
        }
        if (!this.composed && this.options.ci) {
            npmlog_1.default.info('ci', 'enabled');
        }
        let chain = Promise.resolve();
        if (this.commandName !== 'info') {
            chain = chain.then(() => this.project.getPackages());
            chain = chain.then((packages) => {
                this.packageGraph = new package_graph_1.PackageGraph(packages || [], this.options.allowPeerDependenciesUpdate
                    ? 'allPlusPeerDependencies'
                    : 'allDependencies');
            });
        }
        return chain;
    }
    async runCommand() {
        return Promise.resolve()
            .then(() => this.initialize())
            .then((proceed) => {
            if (proceed !== false) {
                return this.execute();
            }
            // early exits set their own exitCode (if non-zero)
        });
    }
    initialize() {
        throw new validation_error_1.ValidationError(this.commandName, 'initialize() needs to be implemented.');
    }
    execute() {
        throw new validation_error_1.ValidationError(this.commandName, 'execute() needs to be implemented.');
    }
}
exports.Command = Command;
// _.defaults(), but simplified:
//  * All inputs are plain objects
//  * Only own keys, not inherited
function defaultOptions(...sources) {
    const options = {};
    for (const source of sources) {
        if (source !== null && source !== undefined) {
            for (const key of Object.keys(source)) {
                if (options[key] === undefined) {
                    options[key] = source[key];
                }
            }
        }
    }
    return options;
}
//# sourceMappingURL=command.js.map