"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.logExecCommand = exports.wrapError = exports.spawnProcess = exports.getExitCode = exports.getChildProcessCount = exports.spawnStreaming = exports.spawn = exports.execSync = exports.exec = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const execa_1 = tslib_1.__importDefault(require("execa"));
const npmlog_1 = tslib_1.__importDefault(require("npmlog"));
const os_1 = tslib_1.__importDefault(require("os"));
const strong_log_transformer_1 = tslib_1.__importDefault(require("strong-log-transformer"));
// bookkeeping for spawned processes
const children = new Set();
// when streaming processes are spawned, use this color for prefix
const colorWheel = ['cyan', 'magenta', 'blue', 'yellow', 'green', 'red'];
const NUM_COLORS = colorWheel.length;
// ever-increasing index ensures colors are always sequential
let currentColor = 0;
/**
 * Execute a command asynchronously, piping stdio by default.
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").Options} [opts]
 */
function exec(command, args, opts, dryRun = false) {
    const options = Object.assign({ stdio: 'pipe' }, opts);
    const spawned = spawnProcess(command, args, options, dryRun);
    return dryRun ? Promise.resolve() : wrapError(spawned);
}
exports.exec = exec;
/**
 * Execute a command synchronously.
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").SyncOptions} [opts]
 */
function execSync(command, args, opts, dryRun = false) {
    // prettier-ignore
    return dryRun
        ? logExecCommand(command, args)
        : execa_1.default.sync(command, args, opts).stdout;
}
exports.execSync = execSync;
/**
 * Spawn a command asynchronously, _always_ inheriting stdio.
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").Options} [opts]
 */
function spawn(command, args, opts, dryRun = false) {
    const options = Object.assign({}, opts, { stdio: 'inherit' });
    const spawned = spawnProcess(command, args, options, dryRun);
    return wrapError(spawned);
}
exports.spawn = spawn;
/**
 * Spawn a command asynchronously, streaming stdio with optional prefix.
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").Options} [opts]
 * @param {string} [prefix]
 */
// istanbul ignore next
function spawnStreaming(command, args, opts, prefix, dryRun = false) {
    const options = Object.assign({}, opts);
    options.stdio = ['ignore', 'pipe', 'pipe'];
    const spawned = spawnProcess(command, args, options, dryRun);
    const stdoutOpts = {};
    const stderrOpts = {}; // mergeMultiline causes escaped newlines :P
    if (prefix) {
        const colorName = colorWheel[currentColor % NUM_COLORS];
        const color = chalk_1.default[colorName];
        currentColor += 1;
        stdoutOpts.tag = `${color.bold(prefix)}:`;
        stderrOpts.tag = `${color(prefix)}:`;
    }
    // Avoid 'Possible EventEmitter memory leak detected' warning due to piped stdio
    if (children.size > process.stdout.listenerCount('close')) {
        process.stdout.setMaxListeners(children.size);
        process.stderr.setMaxListeners(children.size);
    }
    spawned.stdout.pipe((0, strong_log_transformer_1.default)(stdoutOpts)).pipe(process.stdout);
    spawned.stderr.pipe((0, strong_log_transformer_1.default)(stderrOpts)).pipe(process.stderr);
    return wrapError(spawned);
}
exports.spawnStreaming = spawnStreaming;
function getChildProcessCount() {
    return children.size;
}
exports.getChildProcessCount = getChildProcessCount;
function getExitCode(result) {
    var _a, _b, _c;
    // https://nodejs.org/docs/latest-v6.x/api/child_process.html#child_process_event_close
    if (typeof result.code === 'number' || typeof result.exitCode === 'number') {
        return (_a = result.code) !== null && _a !== void 0 ? _a : result.exitCode;
    }
    // https://nodejs.org/docs/latest-v6.x/api/errors.html#errors_error_code
    // istanbul ignore else
    if (typeof result.code === 'string' || typeof result.exitCode === 'string') {
        return os_1.default.constants.errno[(_b = result.code) !== null && _b !== void 0 ? _b : result.exitCode];
    }
    // istanbul ignore next: extremely weird
    throw new TypeError(`Received unexpected exit code value ${JSON.stringify((_c = result.code) !== null && _c !== void 0 ? _c : result.exitCode)}`);
}
exports.getExitCode = getExitCode;
/**
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").Options} opts
 */
function spawnProcess(command, args, opts, dryRun = false) {
    if (dryRun) {
        return logExecCommand(command, args);
    }
    const child = (0, execa_1.default)(command, args, opts);
    const drain = (_code, signal) => {
        children.delete(child);
        // don't run repeatedly if this is the error event
        if (signal === undefined) {
            child.removeListener('exit', drain);
        }
    };
    child.once('exit', drain);
    child.once('error', drain);
    if (opts.pkg) {
        child.pkg = opts.pkg;
    }
    children.add(child);
    return child;
}
exports.spawnProcess = spawnProcess;
/**
 * Spawn a command asynchronously, _always_ inheriting stdio.
 * @param {string} command
 * @param {string[]} args
 * @param {import("execa").Options} [opts]
 */
function wrapError(spawned) {
    if (spawned.pkg) {
        return spawned.catch((err) => {
            // ensure exit code is always a number
            err.exitCode = getExitCode(err);
            // log non-lerna error cleanly
            err.pkg = spawned.pkg;
            throw err;
        });
    }
    return spawned;
}
exports.wrapError = wrapError;
/**
 * Log the child-process command and its arguments as dry-run (without executing the process)
 * @param {string} command
 * @param {string[]} args
 */
function logExecCommand(command, args) {
    var _a;
    const argStr = (_a = (Array.isArray(args) ? args.join(' ') : args)) !== null && _a !== void 0 ? _a : '';
    const cmdList = [];
    for (const c of [command, argStr]) {
        cmdList.push(Array.isArray(c) ? c.join(' ') : c);
    }
    npmlog_1.default.info(chalk_1.default.bold.magenta('[dry-run] >'), cmdList.join(' '));
    return '';
}
exports.logExecCommand = logExecCommand;
//# sourceMappingURL=child-process.js.map