import type { BivariantCallback } from "@ariakit/core/utils/types";
import type { ElementType, SyntheticEvent } from "react";
import type { Options, Props } from "../utils/types.ts";
declare const TagName = "div";
type TagName = typeof TagName;
declare const safariFocusAncestorSymbol: unique symbol;
type SafariFocusAncestor = Element & {
    [safariFocusAncestorSymbol]?: boolean;
};
export declare function isSafariFocusAncestor(element: SafariFocusAncestor | null): boolean;
/**
 * Returns props to create a `Focusable` component.
 * @see https://ariakit.org/components/focusable
 * @example
 * ```jsx
 * const props = useFocusable();
 * <Role {...props}>Focusable</Role>
 * ```
 */
export declare const useFocusable: import("../utils/types.ts").Hook<"div", FocusableOptions<"div">>;
/**
 * Renders a focusable element. When this element gains keyboard focus, it gets
 * a
 * [`data-focus-visible`](https://ariakit.org/guide/styling#data-focus-visible)
 * attribute and triggers the
 * [`onFocusVisible`](https://ariakit.org/reference/focusable#onfocusvisible)
 * prop.
 *
 * The `Focusable` component supports the
 * [`disabled`](https://ariakit.org/reference/focusable#disabled) prop for all
 * elements, even those not supporting the native `disabled` attribute. Disabled
 * elements using the `Focusable` component may be still accessible via keyboard
 * by using the the
 * [`accessibleWhenDisabled`](https://ariakit.org/reference/focusable#accessiblewhendisabled)
 * prop.
 * @see https://ariakit.org/components/focusable
 * @example
 * ```jsx
 * <Focusable>Focusable</Focusable>
 * ```
 */
export declare const Focusable: (props: FocusableProps) => import("react").ReactElement<any, string | import("react").JSXElementConstructor<any>>;
export interface FocusableOptions<_T extends ElementType = TagName> extends Options {
    /**
     * Determines if the element is disabled. This sets the `aria-disabled`
     * attribute accordingly, enabling support for all elements, including those
     * that don't support the native `disabled` attribute.
     *
     * This feature can be combined with the
     * [`accessibleWhenDisabled`](https://ariakit.org/reference/focusable#accessiblewhendisabled)
     * prop to make disabled elements still accessible via keyboard.
     *
     * **Note**: For this prop to work, the
     * [`focusable`](https://ariakit.org/reference/command#focusable) prop must be
     * set to `true`, if it's not set by default.
     *
     * Live examples:
     * - [Submenu](https://ariakit.org/examples/menu-nested)
     * - [Combobox with Tabs](https://ariakit.org/examples/combobox-tabs)
     * - [Context Menu](https://ariakit.org/examples/menu-context-menu)
     * @default false
     */
    disabled?: boolean;
    /**
     * Automatically focuses the element upon mounting, similar to the native
     * `autoFocus` prop. This addresses an issue where the element with the native
     * `autoFocus` attribute might receive focus before React effects are
     * executed.
     *
     * The `autoFocus` prop can also be used with
     * [Focusable](https://ariakit.org/components/focusable) elements within a
     * [Dialog](https://ariakit.org/components/dialog) component, establishing the
     * initial focus as the dialog opens.
     *
     * **Note**: For this prop to work, the
     * [`focusable`](https://ariakit.org/reference/command#focusable) prop must be
     * set to `true`, if it's not set by default.
     *
     * Live examples:
     * - [Warning on Dialog
     *   hide](https://ariakit.org/examples/dialog-hide-warning)
     * - [Dialog with React
     *   Router](https://ariakit.org/examples/dialog-react-router)
     * - [Nested Dialog](https://ariakit.org/examples/dialog-nested)
     * @default false
     */
    autoFocus?: boolean;
    /**
     * Determines if [Focusable](https://ariakit.org/components/focusable)
     * features should be active on non-native focusable elements.
     *
     * **Note**: This prop only turns off the additional features provided by the
     * [`Focusable`](https://ariakit.org/reference/focusable) component.
     * Non-native focusable elements will lose their focusability entirely.
     * However, native focusable elements will retain their inherent focusability,
     * but without added features such as improved
     * [`autoFocus`](https://ariakit.org/reference/focusable#autofocus),
     * [`accessibleWhenDisabled`](https://ariakit.org/reference/focusable#accessiblewhendisabled),
     * [`onFocusVisible`](https://ariakit.org/reference/focusable#onfocusvisible),
     * etc.
     * @default true
     */
    focusable?: boolean;
    /**
     * Indicates whether the element should be focusable even when it is
     * [`disabled`](https://ariakit.org/reference/focusable#disabled).
     *
     * This is important when discoverability is a concern. For example:
     *
     * > A toolbar in an editor contains a set of special smart paste functions
     * that are disabled when the clipboard is empty or when the function is not
     * applicable to the current content of the clipboard. It could be helpful to
     * keep the disabled buttons focusable if the ability to discover their
     * functionality is primarily via their presence on the toolbar.
     *
     * Learn more on [Focusability of disabled
     * controls](https://www.w3.org/WAI/ARIA/apg/practices/keyboard-interface/#focusabilityofdisabledcontrols).
     *
     * Live examples:
     * - [Combobox with Tabs](https://ariakit.org/examples/combobox-tabs)
     * - [Command Menu with
     *   Tabs](https://ariakit.org/examples/dialog-combobox-tab-command-menu)
     */
    accessibleWhenDisabled?: boolean;
    /**
     * Custom event handler invoked when the element gains focus through keyboard
     * interaction or a key press occurs while the element is in focus. This is
     * the programmatic equivalent of the
     * [`data-focus-visible`](https://ariakit.org/guide/styling#data-focus-visible)
     * attribute.
     *
     * **Note**: For this prop to work, the
     * [`focusable`](https://ariakit.org/reference/command#focusable) prop must be
     * set to `true`, if it's not set by default.
     *
     * Live examples:
     * - [Navigation Menubar](https://ariakit.org/examples/menubar-navigation)
     * - [Custom Checkbox](https://ariakit.org/examples/checkbox-custom)
     */
    onFocusVisible?: BivariantCallback<(event: SyntheticEvent<HTMLElement>) => void>;
}
export type FocusableProps<T extends ElementType = TagName> = Props<T, FocusableOptions<T>>;
export {};
