import type { AnyFunction } from "@ariakit/core/utils/types";
import type { ComponentType, DependencyList, EffectCallback, MutableRefObject, Ref, RefCallback, RefObject, SetStateAction } from "react";
import * as React from "react";
import type { WrapElement } from "./types.ts";
/**
 * `React.useLayoutEffect` that fallbacks to `React.useEffect` on server side.
 */
export declare const useSafeLayoutEffect: typeof React.useLayoutEffect;
/**
 * Returns a value that never changes even if the argument is updated.
 * @example
 * function Component({ prop }) {
 *   const initialProp = useInitialValue(prop);
 * }
 */
export declare function useInitialValue<T>(value: T | (() => T)): T;
/**
 * Returns a value that is lazily initiated and never changes.
 * @example
 * function Component() {
 *   const set = useLazyValue(() => new Set());
 * }
 */
export declare function useLazyValue<T>(init: () => T): T;
/**
 * Creates a `React.RefObject` that is constantly updated with the incoming
 * value.
 * @example
 * function Component({ prop }) {
 *   const propRef = useLiveRef(prop);
 * }
 */
export declare function useLiveRef<T>(value: T): MutableRefObject<T>;
/**
 * Keeps the reference of the previous value to be used in the render phase.
 */
export declare function usePreviousValue<T>(value: T): T;
/**
 * Creates a stable callback function that has access to the latest state and
 * can be used within event handlers and effect callbacks. Throws when used in
 * the render phase.
 * @example
 * function Component(props) {
 *   const onClick = useEvent(props.onClick);
 *   React.useEffect(() => {}, [onClick]);
 * }
 */
export declare function useEvent<T extends AnyFunction>(callback?: T): T;
/**
 * Creates a React state that calls a callback function whenever the state
 * changes and rolls back to the previous state on cleanup.
 */
export declare function useTransactionState<T>(callback?: ((state: SetStateAction<T | null>) => void) | null): readonly [T | null, React.Dispatch<SetStateAction<T | null>>];
/**
 * Merges React Refs into a single memoized function ref so you can pass it to
 * an element.
 * @example
 * const Component = React.forwardRef((props, ref) => {
 *   const internalRef = React.useRef();
 *   return <div {...props} ref={useMergeRefs(internalRef, ref)} />;
 * });
 */
export declare function useMergeRefs(...refs: Array<Ref<any> | undefined>): ((value: unknown) => void) | undefined;
/**
 * Generates a unique ID. Uses React's useId if available.
 */
export declare function useId(defaultId?: string): string | undefined;
/**
 * Uses React's useDeferredValue if available.
 */
export declare function useDeferredValue<T>(value: T): T;
/**
 * Returns the tag name by parsing an element ref.
 * @example
 * function Component(props) {
 *   const ref = React.useRef();
 *   const tagName = useTagName(ref, "button"); // div
 *   return <div ref={ref} {...props} />;
 * }
 */
export declare function useTagName(refOrElement?: RefObject<HTMLElement> | HTMLElement | null, type?: string | ComponentType): string | undefined;
/**
 * Returns the attribute value of an element.
 * @example
 * function Component(props) {
 *   const ref = React.useRef();
 *   const role = useAttribute(ref, "role", props.role);
 *   return <div ref={ref} {...props} />;
 * }
 */
export declare function useAttribute(refOrElement: RefObject<HTMLElement> | HTMLElement | null, attributeName: string, defaultValue?: string): string | undefined;
/**
 * A `React.useEffect` that will not run on the first render.
 */
export declare function useUpdateEffect(effect: EffectCallback, deps?: DependencyList): void;
/**
 * A `React.useLayoutEffect` that will not run on the first render.
 */
export declare function useUpdateLayoutEffect(effect: EffectCallback, deps?: DependencyList): void;
/**
 * A React hook similar to `useState` and `useReducer`, but with the only
 * purpose of re-rendering the component.
 */
export declare function useForceUpdate(): [never[], React.DispatchWithoutAction];
/**
 * Returns an event callback similar to `useEvent`, but this also accepts a
 * boolean value, which will be turned into a function.
 */
export declare function useBooleanEvent<T extends unknown[]>(booleanOrCallback: boolean | ((...args: T) => boolean)): (...args: T) => boolean;
/**
 * Returns props with an additional `wrapElement` prop.
 */
export declare function useWrapElement<P>(props: P & {
    wrapElement?: WrapElement;
}, callback: WrapElement, deps?: DependencyList): P & {
    wrapElement: WrapElement;
};
/**
 * Merges the portalRef prop and returns a `domReady` to be used in the
 * components that use Portal underneath.
 */
export declare function usePortalRef(portalProp?: boolean, portalRefProp?: RefCallback<HTMLElement> | MutableRefObject<HTMLElement | null>): {
    portalRef: ((value: unknown) => void) | undefined;
    portalNode: HTMLElement | null;
    domReady: true | HTMLElement | null;
};
/**
 * A hook that passes metadata props around without leaking them to the DOM.
 */
export declare function useMetadataProps<T, K extends keyof any>(props: {
    onLoadedMetadataCapture?: AnyFunction & {
        [key in K]?: T;
    };
}, key: K, value: T): readonly [(AnyFunction & { [key in K]?: T | undefined; })[K] | undefined, {
    readonly onLoadedMetadataCapture: () => void;
}];
/**
 * Returns a function that checks whether the mouse is moving.
 */
export declare function useIsMouseMoving(): () => boolean;
