import type { ElementType, RefObject } from "react";
import type { CollectionItemOptions } from "../collection/collection-item.tsx";
import type { DisclosureContentOptions } from "../disclosure/disclosure-content.tsx";
import type { FocusableOptions } from "../focusable/focusable.tsx";
import type { Props } from "../utils/types.ts";
import type { TabStore } from "./tab-store.ts";
declare const TagName = "div";
type TagName = typeof TagName;
/**
 * Returns props to create a `TabPanel` component.
 * @see https://ariakit.org/components/tab
 * @example
 * ```jsx
 * const store = useTabStore();
 * const props = useTabPanel({ store });
 * <TabList store={store}>
 *   <Tab>Tab 1</Tab>
 * </TabList>
 * <Role {...props}>Panel 1</Role>
 * ```
 */
export declare const useTabPanel: import("../utils/types.ts").Hook<"div", TabPanelOptions<"div">>;
/**
 * Renders a tab panel element that's controlled by a
 * [`Tab`](https://ariakit.org/reference/tab) component.
 *
 * If the [`tabId`](https://ariakit.org/reference/tab-panel#tabid) prop isn't
 * provided, the tab panel will automatically associate with a
 * [`Tab`](https://ariakit.org/reference/tab) based on its position in the DOM.
 * Alternatively, you can render a single tab panel with a dynamic
 * [`tabId`](https://ariakit.org/reference/tab-panel#tabid) value pointing to
 * the selected tab.
 * @see https://ariakit.org/components/tab
 * @example
 * ```jsx {6,7}
 * <TabProvider>
 *   <TabList>
 *     <Tab>Tab 1</Tab>
 *     <Tab>Tab 2</Tab>
 *   </TabList>
 *   <TabPanel>Panel 1</TabPanel>
 *   <TabPanel>Panel 2</TabPanel>
 * </TabProvider>
 * ```
 */
export declare const TabPanel: (props: TabPanelProps) => import("react").ReactElement<any, string | import("react").JSXElementConstructor<any>>;
export interface TabPanelOptions<T extends ElementType = TagName> extends FocusableOptions<T>, CollectionItemOptions<T>, Omit<DisclosureContentOptions<T>, "store"> {
    /**
     * Object returned by the
     * [`useTabStore`](https://ariakit.org/reference/use-tab-store) hook. If not
     * provided, the closest
     * [`TabProvider`](https://ariakit.org/reference/tab-provider) component's
     * context will be used.
     */
    store?: TabStore;
    /**
     * The [`id`](https://ariakit.org/reference/tab#id) of the tab controlling
     * this panel. This connection is used to assign the `aria-labelledby`
     * attribute to the tab panel and to determine if the tab panel should be
     * visible.
     *
     * This link is automatically established by matching the order of
     * [`Tab`](https://ariakit.org/reference/tab) and
     * [`TabPanel`](https://ariakit.org/reference/tab-panel) elements in the DOM.
     * If you're rendering a single tab panel, this can be set to a dynamic value
     * that refers to the selected tab.
     *
     * Live examples:
     * - [Combobox with Tabs](https://ariakit.org/examples/combobox-tabs)
     * - [Tab with React Router](https://ariakit.org/examples/tab-react-router)
     * - [Animated TabPanel](https://ariakit.org/examples/tab-panel-animated)
     * - [Select with Combobox and
     *   Tabs](https://ariakit.org/examples/select-combobox-tab)
     * - [Command Menu with
     *   Tabs](https://ariakit.org/examples/dialog-combobox-tab-command-menu)
     */
    tabId?: string | null;
    /**
     * Manages the scrolling behavior of the tab panel when it is hidden and then
     * shown again.
     *
     * This is especially useful when using a single tab panel for multiple tabs,
     * where you dynamically change the
     * [`tabId`](https://ariakit.org/reference/tab-panel#tabid) prop and the
     * panel's children, which would otherwise retain the current scroll position
     * when switching tabs.
     *
     * When set to `true`, the component will save the scroll position and restore
     * it when the panel is shown again. When set to `"reset"`, the scroll
     * position will reset to the top when the panel is displayed again.
     *
     * The default scroll element is the tab panel itself. To scroll a different
     * element, use the
     * [`scrollElement`](https://ariakit.org/reference/tab-panel#scrollelement)
     * prop.
     * @default false
     */
    scrollRestoration?: boolean | "reset";
    /**
     * When using the
     * [`scrollRestoration`](https://ariakit.org/reference/tab-panel#scrollrestoration)
     * prop, the tab panel element serves as the default scroll element. You can
     * use this prop to designate a different element for scrolling.
     *
     * If a function is provided, it will be called with the tab panel element as
     * an argument. The function should return the element to scroll.
     * @example
     * ```jsx
     * <TabPanel
     *   scrollRestoration
     *   scrollElement={(panel) => panel.querySelector(".scrollable")}
     * />
     * ```
     */
    scrollElement?: HTMLElement | RefObject<HTMLElement> | ((panel: HTMLElement) => HTMLElement | null);
}
export type TabPanelProps<T extends ElementType = TagName> = Props<T, TabPanelOptions<T>>;
export {};
