import type { BooleanOrCallback, StringWithValue } from "@ariakit/core/utils/types";
import type { AriaAttributes, ChangeEvent, ElementType, MouseEvent, KeyboardEvent as ReactKeyboardEvent } from "react";
import type { CompositeOptions } from "../composite/composite.tsx";
import type { PopoverAnchorOptions } from "../popover/popover-anchor.tsx";
import type { Props } from "../utils/types.ts";
import type { ComboboxStore, ComboboxStoreState } from "./combobox-store.ts";
declare const TagName = "input";
type TagName = typeof TagName;
/**
 * Returns props to create a `Combobox` component.
 * @see https://ariakit.org/components/combobox
 * @example
 * ```jsx
 * const store = useComboboxStore();
 * const props = useCombobox({ store });
 * <Role {...props} />
 * <ComboboxPopover store={store}>
 *   <ComboboxItem value="Apple" />
 *   <ComboboxItem value="Banana" />
 *   <ComboboxItem value="Orange" />
 * </ComboboxPopover>
 * ```
 */
export declare const useCombobox: import("../utils/types.ts").Hook<"input", ComboboxOptions<"input">>;
/**
 * Renders a combobox input element that can be used to filter a list of items.
 * @see https://ariakit.org/components/combobox
 * @example
 * ```jsx {2}
 * <ComboboxProvider>
 *   <Combobox />
 *   <ComboboxPopover>
 *     <ComboboxItem value="Apple" />
 *     <ComboboxItem value="Banana" />
 *     <ComboboxItem value="Orange" />
 *   </ComboboxPopover>
 * </ComboboxProvider>
 * ```
 */
export declare const Combobox: (props: ComboboxProps) => import("react").ReactElement<any, string | import("react").JSXElementConstructor<any>>;
export interface ComboboxOptions<T extends ElementType = TagName> extends CompositeOptions<T>, PopoverAnchorOptions<T> {
    /**
     * Object returned by the
     * [`useComboboxStore`](https://ariakit.org/reference/use-combobox-store)
     * hook. If not provided, the closest
     * [`ComboboxProvider`](https://ariakit.org/reference/combobox-provider)
     * component's context will be used.
     */
    store?: ComboboxStore;
    /**
     * Determines if the first enabled item will be automatically focused when the
     * combobox input value changes. If set to `true` or `"always"`, the exact
     * behavior hinges on the value of the
     * [`autoComplete`](https://ariakit.org/reference/combobox#autocomplete) prop:
     * - If [`autoComplete`](https://ariakit.org/reference/combobox#autocomplete)
     *   is `both` or `inline`, the first enabled item is automatically focused as
     *   the user types in the input. The value gets appended with the completion
     *   string if it matches. The inline completion string will be highlighted
     *   and selected.
     * - If [`autoComplete`](https://ariakit.org/reference/combobox#autocomplete)
     *   is `list` or `none`, the first enabled item is automatically focused as
     *   the user types in the input, but the input value is not appended with the
     *   item value.
     *
     * If set to `"always"`, the first enabled item is auto-highlighted when the
     * combobox list opens, not just when the input value changes.
     *
     * To change which item gets auto-selected, use the
     * [`getAutoSelectId`](https://ariakit.org/reference/combobox#getautoselectid)
     * prop.
     *
     * Live examples:
     * - [Command Menu with
     *   Tabs](https://ariakit.org/examples/dialog-combobox-tab-command-menu)
     * - [ComboboxGroup](https://ariakit.org/examples/combobox-group)
     * - [Combobox with links](https://ariakit.org/examples/combobox-links)
     * - [Textarea with inline
     *   Combobox](https://ariakit.org/examples/combobox-textarea)
     * - [Menu with Combobox](https://ariakit.org/examples/menu-combobox)
     * - [Select with Combobox](https://ariakit.org/examples/select-combobox)
     * @default false
     */
    autoSelect?: boolean | "always";
    /**
     * Function that takes the currently rendered items and returns the id of the
     * item to be auto selected when the
     * [`autoSelect`](https://ariakit.org/reference/combobox#autoselect) prop is
     * `true`.
     *
     * By default, the first enabled item is auto selected. This function is handy
     * if you prefer a different item to be auto selected. Returning `undefined`
     * from this function will result in the default behavior.
     * @example
     * ```jsx
     * <Combobox
     *   autoSelect
     *   getAutoSelectId={(items) => {
     *     // Auto select the first enabled item with a value
     *     const item = items.find((item) => {
     *       if (item.disabled) return false;
     *       if (!item.value) return false;
     *       return true;
     *     });
     *     return item?.id;
     *   }}
     * />
     * ```
     */
    getAutoSelectId?: (renderedItems: ComboboxStoreState["renderedItems"]) => string | null | undefined;
    /**
     * Whether the items will be filtered based on
     * [`value`](https://ariakit.org/reference/combobox-provider#value) and
     * whether the input value will temporarily change based on the active item.
     *
     * This prop is based on the standard
     * [`aria-autocomplete`](https://w3c.github.io/aria/#aria-autocomplete)
     * attribute, accepting the same values:
     * - `list` (default): indicates that the items will be dynamically rendered
     *   based on [`value`](https://ariakit.org/reference/combobox-provider#value)
     *   and the input value will _not_ change based on the active item. The
     *   filtering logic must be implemented by the consumer of this component.
     * - `inline`: indicates that the items are static, that is, they won't be
     *   filtered, but the input value will temporarily change based on the active
     *   item. Ariakit will automatically provide the inline autocompletion
     *   behavior.
     * - `both`: indicates that the items will be dynamically rendered based on
     *   [`value`](https://ariakit.org/reference/combobox-provider#value) and the
     *   input value will temporarily change based on the active item. The
     *   filtering logic must be implemented by the consumer of this component,
     *   whereas Ariakit will automatically provide the inline autocompletion
     *   behavior.
     * - `none`: the items are static and the input value will _not_ change based
     *   on the active item.
     *
     * Live examples:
     * - [ComboboxGroup](https://ariakit.org/examples/combobox-group)
     * @default "list"
     */
    autoComplete?: StringWithValue<Required<AriaAttributes>["aria-autocomplete"]>;
    /**
     * Determines if the highlighted item should lose focus when the user clicks
     * on the combobox input element. By default, this prop's value is set
     * according to the
     * [`includesBaseElement`](https://ariakit.org/reference/combobox-provider#includesbaseelement)
     * value.
     */
    blurActiveItemOnClick?: BooleanOrCallback<MouseEvent<HTMLElement>>;
    /**
     * Specifies the minimum character count the input value should have before
     * the [`ComboboxList`](https://ariakit.org/reference/combobox-list) or
     * [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components are displayed.
     *
     * The [`showOnChange`](https://ariakit.org/reference/combobox#showonchange),
     * [`showOnClick`](https://ariakit.org/reference/combobox#showonclick), and
     * [`showOnKeyPress`](https://ariakit.org/reference/combobox#showonkeypress)
     * props allow you to tailor the behavior for each unique event.
     * @default 0
     * @example
     * In the following example, the combobox list will be shown when the input
     * value has at least one character. However, if the user presses the arrow
     * keys, the list will be shown regardless of the input value length.
     * ```jsx
     * <Combobox showMinLength={1} showOnKeyPress />
     * ```
     */
    showMinLength?: number;
    /**
     * Whether the [`ComboboxList`](https://ariakit.org/reference/combobox-list)
     * or [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components should be shown when the input value changes.
     *
     * Live examples:
     * - [Textarea with inline
     *   Combobox](https://ariakit.org/examples/combobox-textarea)
     * @default true
     * @example
     * ```jsx
     * <Combobox showOnChange={(event) => event.target.value.length > 1} />
     * ```
     */
    showOnChange?: BooleanOrCallback<ChangeEvent<HTMLElement>>;
    /**
     * Whether the [`ComboboxList`](https://ariakit.org/reference/combobox-list)
     * or [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components should be shown when the input is clicked.
     * @deprecated Use
     * [`showOnClick`](https://ariakit.org/reference/combobox#showonclick)
     * instead.
     * @default true
     */
    showOnMouseDown?: BooleanOrCallback<MouseEvent<HTMLElement>>;
    /**
     * Whether the [`ComboboxList`](https://ariakit.org/reference/combobox-list)
     * or [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components should be shown when the input is clicked.
     *
     * Live examples:
     * - [Textarea with inline
     *   Combobox](https://ariakit.org/examples/combobox-textarea)
     * @default true
     * @example
     * ```jsx
     * <Combobox showOnClick={value.length > 1} />
     * ```
     */
    showOnClick?: BooleanOrCallback<MouseEvent<HTMLElement>>;
    /**
     * Whether the [`ComboboxList`](https://ariakit.org/reference/combobox-list)
     * or [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components should be shown when the user presses the arrow up or down keys
     * while focusing on the combobox input element.
     * @deprecated Use
     * [`showOnKeyPress`](https://ariakit.org/reference/combobox#showonkeypress)
     * instead.
     * @default true
     */
    showOnKeyDown?: BooleanOrCallback<ReactKeyboardEvent<HTMLElement>>;
    /**
     * Whether the [`ComboboxList`](https://ariakit.org/reference/combobox-list)
     * or [`ComboboxPopover`](https://ariakit.org/reference/combobox-popover)
     * components should be shown when the user presses the arrow up or down keys
     * while focusing on the combobox input element.
     *
     * Live examples:
     * - [Textarea with inline
     *   Combobox](https://ariakit.org/examples/combobox-textarea)
     * @default true
     * @example
     * ```jsx
     * <Combobox showOnKeyPress={value.length > 1} />
     * ```
     */
    showOnKeyPress?: BooleanOrCallback<ReactKeyboardEvent<HTMLElement>>;
    /**
     * Whether the combobox
     * [`value`](https://ariakit.org/reference/combobox-provider#value) state
     * should be updated when the input value changes. This is useful if you want
     * to customize how the store
     * [`value`](https://ariakit.org/reference/combobox-provider#value) is updated
     * based on the input element's value.
     *
     * Live examples:
     * - [Textarea with inline
     *   Combobox](https://ariakit.org/examples/combobox-textarea)
     * @default true
     */
    setValueOnChange?: BooleanOrCallback<ChangeEvent<HTMLElement>>;
    /**
     * Whether the combobox
     * [`value`](https://ariakit.org/reference/combobox-provider#value) state
     * should be updated when the combobox input element gets clicked. This
     * usually only applies when
     * [`autoComplete`](https://ariakit.org/reference/combobox#autocomplete) is
     * `both` or `inline`, because the input value will temporarily change based
     * on the active item and the store
     * [`value`](https://ariakit.org/reference/combobox-provider#value) will not
     * be updated until the user confirms the selection.
     * @default true
     */
    setValueOnClick?: BooleanOrCallback<MouseEvent<HTMLElement>>;
}
export type ComboboxProps<T extends ElementType = TagName> = Props<T, ComboboxOptions<T>>;
export {};
