import type { CollectionStore, CollectionStoreItem } from "../collection/collection-store.ts";
import type { ComboboxStore } from "../combobox/combobox-store.ts";
import type { CompositeStore, CompositeStoreFunctions, CompositeStoreItem, CompositeStoreOptions, CompositeStoreState } from "../composite/composite-store.ts";
import type { SelectStore } from "../select/select-store.ts";
import type { Store, StoreOptions, StoreProps } from "../utils/store.ts";
import type { SetState } from "../utils/types.ts";
export declare function createTabStore({ composite: parentComposite, combobox, ...props }?: TabStoreProps): TabStore;
export interface TabStoreItem extends CompositeStoreItem {
    dimmed?: boolean;
}
export interface TabStorePanel extends CollectionStoreItem {
    tabId?: string | null;
}
export interface TabStoreState extends CompositeStoreState<TabStoreItem> {
    /** @default "horizontal" */
    orientation: CompositeStoreState<TabStoreItem>["orientation"];
    /** @default true */
    focusLoop: CompositeStoreState<TabStoreItem>["focusLoop"];
    /**
     * The id of the tab whose panel is currently visible. If it's `undefined`, it
     * will be automatically set to the first enabled tab.
     *
     * Live examples:
     * - [Tab with React Router](https://ariakit.org/examples/tab-react-router)
     * - [Combobox with Tabs](https://ariakit.org/examples/combobox-tabs)
     * - [Select with Combobox and
     *   Tabs](https://ariakit.org/examples/select-combobox-tab)
     * - [Command Menu with
     *   Tabs](https://ariakit.org/examples/dialog-combobox-tab-command-menu)
     */
    selectedId: TabStoreState["activeId"];
    /**
     * Determines if the tab should be selected when it receives focus. If set to
     * `false`, the tab will only be selected upon clicking, not when using arrow
     * keys to shift focus.
     *
     * Live examples:
     * - [Tab with React Router](https://ariakit.org/examples/tab-react-router)
     * - [Select with Combobox and
     *   Tabs](https://ariakit.org/examples/select-combobox-tab)
     * @default true
     */
    selectOnMove?: boolean;
}
export interface TabStoreFunctions extends CompositeStoreFunctions<TabStoreItem> {
    /**
     * Sets the
     * [`selectedId`](https://ariakit.org/reference/tab-provider#selectedid) state
     * without moving focus. If you want to move focus, use the
     * [`select`](https://ariakit.org/reference/use-tab-store#select) function
     * instead.
     * @example
     * // Selects the tab with id "tab-1"
     * store.setSelectedId("tab-1");
     * // Toggles between "tab-1" and "tab-2"
     * store.setSelectedId((id) => id === "tab-1" ? "tab-2" : "tab-1"));
     * // Selects the first tab
     * store.setSelectedId(store.first());
     * // Selects the next tab
     * store.setSelectedId(store.next());
     */
    setSelectedId: SetState<TabStoreState["selectedId"]>;
    /**
     * A collection store containing the tab panels.
     *
     * Live examples:
     * - [Animated TabPanel](https://ariakit.org/examples/tab-panel-animated)
     */
    panels: CollectionStore<TabStorePanel>;
    /**
     * Selects the tab for the given id and moves focus to it. If you want to set
     * the [`selectedId`](https://ariakit.org/reference/tab-provider#selectedid)
     * state without moving focus, use the
     * [`setSelectedId`](https://ariakit.org/reference/use-tab-store#setselectedid-1)
     * function instead.
     * @example
     * // Selects the tab with id "tab-1"
     * store.select("tab-1");
     * // Selects the first tab
     * store.select(store.first());
     * // Selects the next tab
     * store.select(store.next());
     */
    select: TabStore["move"];
}
export interface TabStoreOptions extends StoreOptions<TabStoreState, "orientation" | "focusLoop" | "selectedId" | "selectOnMove">, CompositeStoreOptions<TabStoreItem> {
    /**
     * A reference to another [composite
     * store](https://ariakit.org/reference/use-composite-store). This is used when
     * rendering tabs as part of another composite widget such as
     * [Combobox](https://ariakit.org/components/combobox) or
     * [Select](https://ariakit.org/components/select). The stores will share the
     * same state.
     */
    composite?: CompositeStore | SelectStore | null;
    /**
     * A reference to a [combobox
     * store](https://ariakit.org/reference/use-combobox-store). This is used when
     * rendering tabs inside a
     * [Combobox](https://ariakit.org/components/combobox).
     */
    combobox?: ComboboxStore | null;
    /**
     * The id of the tab whose panel is currently visible. If it's `undefined`, it
     * will be automatically set to the first enabled tab.
     *
     * Live examples:
     * - [Combobox with Tabs](https://ariakit.org/examples/combobox-tabs)
     * - [Animated TabPanel](https://ariakit.org/examples/tab-panel-animated)
     * - [Select with Combobox and
     *   Tabs](https://ariakit.org/examples/select-combobox-tab)
     * - [Command Menu with
     *   Tabs](https://ariakit.org/examples/dialog-combobox-tab-command-menu)
     */
    defaultSelectedId?: TabStoreState["selectedId"];
}
export interface TabStoreProps extends TabStoreOptions, StoreProps<TabStoreState> {
}
export interface TabStore extends TabStoreFunctions, Store<TabStoreState> {
}
