import type { ComboboxStore } from "../combobox/combobox-store.ts";
import type { CompositeStoreFunctions, CompositeStoreItem, CompositeStoreOptions, CompositeStoreState } from "../composite/composite-store.ts";
import type { PopoverStoreFunctions, PopoverStoreOptions, PopoverStoreState } from "../popover/popover-store.ts";
import type { Store, StoreOptions, StoreProps } from "../utils/store.ts";
import type { PickRequired, SetState } from "../utils/types.ts";
type MutableValue<T extends SelectStoreValue = SelectStoreValue> = T extends string ? string : T;
export declare function createSelectStore<T extends SelectStoreValue = SelectStoreValue>(props: PickRequired<SelectStoreProps<T>, "value" | "defaultValue">): SelectStore<T>;
export declare function createSelectStore(props?: SelectStoreProps): SelectStore;
export type SelectStoreValue = string | string[];
export interface SelectStoreItem extends CompositeStoreItem {
    value?: string;
}
export interface SelectStoreState<T extends SelectStoreValue = SelectStoreValue> extends CompositeStoreState<SelectStoreItem>, PopoverStoreState {
    /** @default true */
    virtualFocus: CompositeStoreState<SelectStoreItem>["virtualFocus"];
    /** @default null */
    activeId: CompositeStoreState<SelectStoreItem>["activeId"];
    /** @default "vertical" */
    orientation: CompositeStoreState<SelectStoreItem>["orientation"];
    /** @default "bottom-start" */
    placement: PopoverStoreState["placement"];
    /**
     * The select value.
     *
     * Live examples:
     * - [Form with Select](https://ariakit.org/examples/form-select)
     * - [Select Grid](https://ariakit.org/examples/select-grid)
     * - [Select with custom
     *   items](https://ariakit.org/examples/select-item-custom)
     * - [Multi-Select](https://ariakit.org/examples/select-multiple)
     * - [Toolbar with Select](https://ariakit.org/examples/toolbar-select)
     * - [Select with Next.js App
     *   Router](https://ariakit.org/examples/select-next-router)
     */
    value: MutableValue<T>;
    /**
     * Whether the select
     * [`value`](https://ariakit.org/reference/select-provider#value) should be
     * set when the active item changes by moving (which usually happens when
     * moving to an item using the keyboard).
     *
     * Live examples:
     * - [Select Grid](https://ariakit.org/examples/select-grid)
     * - [Select with custom
     *   items](https://ariakit.org/examples/select-item-custom)
     * @default false
     */
    setValueOnMove: boolean;
    /**
     * The select label element.
     */
    labelElement: HTMLElement | null;
    /**
     * The select button element.
     *
     * Live examples:
     * - [Form with Select](https://ariakit.org/examples/form-select)
     */
    selectElement: HTMLElement | null;
    /**
     * The select list element.
     */
    listElement: HTMLElement | null;
}
export interface SelectStoreFunctions<T extends SelectStoreValue = SelectStoreValue> extends Pick<SelectStoreOptions<T>, "combobox">, CompositeStoreFunctions<SelectStoreItem>, PopoverStoreFunctions {
    /**
     * Sets the [`value`](https://ariakit.org/reference/select-provider#value)
     * state.
     * @example
     * store.setValue("Apple");
     * store.setValue(["Apple", "Banana"]);
     * store.setValue((value) => value === "Apple" ? "Banana" : "Apple"));
     */
    setValue: SetState<SelectStoreState<T>["value"]>;
    /**
     * Sets the `labelElement` state.
     */
    setLabelElement: SetState<SelectStoreState<T>["labelElement"]>;
    /**
     * Sets the `selectElement` state.
     */
    setSelectElement: SetState<SelectStoreState<T>["selectElement"]>;
    /**
     * Sets the `listElement` state.
     */
    setListElement: SetState<SelectStoreState<T>["listElement"]>;
}
export interface SelectStoreOptions<T extends SelectStoreValue = SelectStoreValue> extends StoreOptions<SelectStoreState<T>, "virtualFocus" | "activeId" | "orientation" | "placement" | "value" | "setValueOnMove">, CompositeStoreOptions<SelectStoreItem>, PopoverStoreOptions {
    /**
     * A reference to a combobox store. This is used when combining the combobox
     * with a select (e.g., select with a search input). The stores will share the
     * same state.
     */
    combobox?: ComboboxStore | null;
    /**
     * The default value. If not set, the first non-disabled item will be used.
     *
     * Live examples:
     * - [Form with Select](https://ariakit.org/examples/form-select)
     * - [Animated Select](https://ariakit.org/examples/select-animated)
     * - [Select with Combobox](https://ariakit.org/examples/select-combobox)
     * - [SelectGroup](https://ariakit.org/examples/select-group)
     * - [Select with Next.js App
     *   Router](https://ariakit.org/examples/select-next-router)
     * - [Select with Combobox and
     *   Tabs](https://ariakit.org/examples/select-combobox-tab)
     */
    defaultValue?: SelectStoreState<T>["value"];
}
export interface SelectStoreProps<T extends SelectStoreValue = SelectStoreValue> extends SelectStoreOptions<T>, StoreProps<SelectStoreState<T>> {
}
export interface SelectStore<T extends SelectStoreValue = SelectStoreValue> extends SelectStoreFunctions<T>, Store<SelectStoreState<T>> {
}
export {};
