package com.simpol.permissionssummary

import android.content.pm.ApplicationInfo
import android.content.pm.PackageManager
import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawing
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Check
import androidx.compose.material.icons.filled.ClearAll
import androidx.compose.material3.Checkbox
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.derivedStateOf
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.graphics.painter.BitmapPainter
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.core.graphics.drawable.toBitmap
import androidx.navigation.NavController
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

@Composable
fun TrustedListScreen(
    navController: NavController
) {
    val context = LocalContext.current
    val dataStore = remember { TrustedAppsDataStore(context) }
    val coroutineScope = rememberCoroutineScope()
    val density = LocalDensity.current

    val apps = remember { mutableStateListOf<App>() }
    var searchQuery by remember { mutableStateOf("") }
    val trustedApps = remember { mutableStateListOf<String>() }
    var isLoading by remember { mutableStateOf(true) }

    // Derived state for filtered apps - only recalculates when apps or searchQuery changes
    val filteredApps by remember {
        derivedStateOf {
            if (searchQuery.isBlank()) {
                apps
            } else {
                apps.filter { it.name.contains(searchQuery, ignoreCase = true) }
            }
        }
    }

    // Load trusted apps from DataStore
    LaunchedEffect(Unit) {
        dataStore.trustedApps.collect { saved ->
            trustedApps.clear()
            trustedApps.addAll(saved)
        }
    }

    // Load apps in background with better performance
    LaunchedEffect(Unit) {
        withContext(Dispatchers.IO) {
            val pm = context.packageManager
            val installed = pm.getInstalledPackages(PackageManager.GET_PERMISSIONS)
            val userApps = installed.mapNotNull { pkg ->
                try {
                    val appInfo = pm.getApplicationInfo(pkg.packageName, 0)
                    val isSystemApp = (appInfo.flags and ApplicationInfo.FLAG_SYSTEM) != 0 ||
                            (appInfo.flags and ApplicationInfo.FLAG_UPDATED_SYSTEM_APP) != 0
                    if (isSystemApp) return@mapNotNull null

                    App(
                        name = pm.getApplicationLabel(appInfo).toString(),
                        packageName = pkg.packageName,
                        permissions = emptyList()
                    )
                } catch (_: Exception) {
                    null
                }
            }.sortedBy { it.name }

            // Update UI on main thread
            withContext(Dispatchers.Main) {
                apps.clear()
                apps.addAll(userApps)
                isLoading = false
            }
        }
    }

    Scaffold(
        modifier = Modifier.fillMaxSize(),
        contentWindowInsets = WindowInsets.safeDrawing,
        floatingActionButton = {
            Column(
                verticalArrangement = Arrangement.spacedBy(12.dp),
                horizontalAlignment = Alignment.End
            ) {
                FloatingActionButton(
                    onClick = {
                        trustedApps.clear()
                        coroutineScope.launch(Dispatchers.IO) {
                            dataStore.saveTrustedApps(trustedApps.toSet())
                        }
                        navController.popBackStack()
                    }
                ) {
                    Icon(Icons.Default.ClearAll, contentDescription = "Clear Trusted List")
                }

                FloatingActionButton(
                    onClick = {
                        val toSave = trustedApps.toSet()
                        coroutineScope.launch(Dispatchers.IO) {
                            dataStore.saveTrustedApps(toSave)
                        }
                        navController.popBackStack()
                    }
                ) {
                    Icon(Icons.Default.Check, contentDescription = "Save Trusted List")
                }
            }
        }
    ) { innerPadding ->
        Column(
            Modifier
                .fillMaxSize()
                .padding(innerPadding)
                .padding(16.dp)
        ) {
            OutlinedTextField(
                value = searchQuery,
                onValueChange = { searchQuery = it },
                label = { Text("Search apps") },
                modifier = Modifier.fillMaxWidth()
            )

            Spacer(Modifier.height(12.dp))

            if (isLoading) {
                Box(
                    modifier = Modifier.fillMaxSize(),
                    contentAlignment = Alignment.Center
                ) {
                    Column(
                        horizontalAlignment = Alignment.CenterHorizontally,
                        verticalArrangement = Arrangement.Center
                    ) {
                        CircularProgressIndicator(
                            modifier = Modifier.size(48.dp),
                            strokeWidth = 4.dp
                        )
                        Spacer(modifier = Modifier.height(16.dp))
                        Text(
                            text = "Loading apps...",
                            style = MaterialTheme.typography.bodyLarge,
                            color = MaterialTheme.colorScheme.onSurfaceVariant
                        )
                    }
                }
            } else {
                LazyColumn {
                    items(
                        items = filteredApps,
                        key = { it.packageName } // Add key for better performance
                    ) { app ->
                        AppListItem(
                            app = app,
                            isChecked = trustedApps.contains(app.packageName),
                            onToggle = { checked ->
                                if (checked) {
                                    trustedApps.add(app.packageName)
                                } else {
                                    trustedApps.remove(app.packageName)
                                }
                            }
                        )
                    }
                }
            }
        }
    }
}

@Composable
private fun AppListItem(
    app: App,
    isChecked: Boolean,
    onToggle: (Boolean) -> Unit
) {
    Row(
        verticalAlignment = Alignment.CenterVertically,
        modifier = Modifier
            .fillMaxWidth()
            .padding(vertical = 6.dp)
            .clickable { onToggle(!isChecked) }
    ) {
        AppIcon(
            packageName = app.packageName,
            size = 48
        )
        Spacer(Modifier.width(12.dp))
        Text(
            text = app.name,
            modifier = Modifier.weight(1f),
            fontWeight = FontWeight.Medium
        )
        Checkbox(
            checked = isChecked,
            onCheckedChange = onToggle
        )
    }
}

@Composable
fun AppIcon(packageName: String, size: Int = 48) {
    val context = LocalContext.current
    val density = LocalDensity.current
    var iconBitmap by remember(packageName) { mutableStateOf<android.graphics.Bitmap?>(null) }
    var isLoading by remember(packageName) { mutableStateOf(true) }

    LaunchedEffect(packageName) {
        withContext(Dispatchers.IO) {
            try {
                val pm = context.packageManager
                val drawable = pm.getApplicationIcon(packageName)

                // Convert dp to pixels for high quality
                val sizeInPixels = with(density) { (size * 1.5f).dp.roundToPx() }

                // Create high-quality bitmap
                val bitmap = drawable.toBitmap(
                    width = sizeInPixels,
                    height = sizeInPixels,
                    config = android.graphics.Bitmap.Config.ARGB_8888
                )

                withContext(Dispatchers.Main) {
                    iconBitmap = bitmap
                    isLoading = false
                }
            } catch (_: Exception) {
                withContext(Dispatchers.Main) {
                    isLoading = false
                }
            }
        }
    }

    Box(
        modifier = Modifier
            .size(size.dp)
            .clip(RoundedCornerShape(8.dp)),
        contentAlignment = Alignment.Center
    ) {
        iconBitmap?.let { bitmap ->
            Image(
                painter = BitmapPainter(bitmap.asImageBitmap()),
                contentDescription = null,
                modifier = Modifier.fillMaxSize(),
                contentScale = ContentScale.Fit
            )
        } ?: run {
            if (isLoading) {
                CircularProgressIndicator(
                    modifier = Modifier.size(16.dp),
                    strokeWidth = 2.dp
                )
            } else {
                // Fallback icon or empty space
                Box(
                    modifier = Modifier
                        .fillMaxSize()
                        .clip(RoundedCornerShape(8.dp))
                ) {
                    Icon(
                        imageVector = Icons.Default.Check, // Use any default icon
                        contentDescription = null,
                        modifier = Modifier.fillMaxSize(),
                        tint = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
            }
        }
    }
}