package com.simpol.permissionssummary

import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.ApplicationInfo
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.content.pm.PermissionInfo
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

class AppRepository(private val context: Context) {

    suspend fun getPermissionGroups(): List<PermissionsGroup> = withContext(Dispatchers.IO) {
        val pm = context.packageManager
        val packages = pm.getInstalledPackages(PackageManager.GET_PERMISSIONS)
        val processedPackages =
            packages.mapNotNull { pkg -> createAppInfo(pkg, pm) }.sortedBy { it.name }
        mapAppsToPermissionsGroups(processedPackages)
    }

    private fun createAppInfo(packageInfo: PackageInfo, pm: PackageManager): App? {
        return try {
            val appInfo = pm.getApplicationInfo(packageInfo.packageName, 0)
            if (appInfo.flags and ApplicationInfo.FLAG_SYSTEM != 0) return null
            val grantedPermissions = extractGrantedPermissions(packageInfo, pm)
            if (grantedPermissions.isEmpty()) return null
            val permissionObjects = grantedPermissions.mapNotNull { getPermissionObject(it) }
            App(
                name = pm.getApplicationLabel(appInfo).toString(),
                packageName = packageInfo.packageName,
                permissions = permissionObjects
            )
        } catch (_: Exception) {
            null
        }
    }

    private fun extractGrantedPermissions(pkg: PackageInfo, pm: PackageManager): List<String> {
        val result = mutableListOf<String>()
        val permissions = pkg.requestedPermissions ?: return emptyList()
        val flags = pkg.requestedPermissionsFlags ?: return emptyList()
        for (i in permissions.indices) {
            val permission = permissions[i]
            val isGranted = flags[i] and PackageInfo.REQUESTED_PERMISSION_GRANTED != 0
            if (!isGranted) continue
            try {
                val info = pm.getPermissionInfo(permission, 0)
                val isDangerous = info.protectionLevel and PermissionInfo.PROTECTION_DANGEROUS != 0
                if (isDangerous)
                    result.add(permission)
            } catch (_: Exception) {
                // Ignore missing permission info
            }
        }
        return result
    }

    private fun getPermissionObject(permission: String): String? {
        val group = Constants.permissionMapping[permission] ?: return null
        return group
    }

    @SuppressLint("UseCompatLoadingForDrawables")
    fun mapAppsToPermissionsGroups(apps: List<App>): List<PermissionsGroup> {
        val groupMap = mutableMapOf<String, MutableList<Pair<App, String>>>()
        for (app in apps) {
            for (perm in app.permissions) {
                val group = perm
                groupMap.getOrPut(group) { mutableListOf() }.add(app to perm)
            }
        }
        return groupMap.mapNotNull { (groupName, appPermPairs) ->
            val iconResId = Constants.groupIcons[groupName] ?: return@mapNotNull null
            val groupIcon = iconResId
            val appsInGroup = appPermPairs.map { it.first }.distinct()
            PermissionsGroup(
                groupName = groupName,
                groupIcon = groupIcon,
                apps = appsInGroup
            )
        }
    }
}
