package com.simpol.permissionssummary

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch

class AppListViewModel(application: Application) : AndroidViewModel(application) {

    private val repository = AppRepository(application)
    private val dataStore = TrustedAppsDataStore(application)

    private val _permissionsGroups = MutableStateFlow<List<PermissionsGroup>>(emptyList())
    val permissionsGroups: StateFlow<List<PermissionsGroup>> = _permissionsGroups.asStateFlow()

    private val _isLoading = MutableStateFlow(false)
    val isLoading: StateFlow<Boolean> = _isLoading.asStateFlow()

    init {
        // Reactively reload permissions whenever trusted list changes
        viewModelScope.launch {
            dataStore.trustedApps.collect { trusted ->
                loadPermissionGroups(trusted)
            }
        }
    }

    private fun loadPermissionGroups(trustedApps: Set<String>) {
        viewModelScope.launch {
            _isLoading.value = true
            try {
                _permissionsGroups.value = repository
                    .getPermissionGroups()
                    .map { group ->
                        group.copy(
                            apps = group.apps.filterNot { trustedApps.contains(it.packageName) }
                        )
                    }
                    .filter { it.apps.isNotEmpty() }
            } catch (e: Exception) {
                e.printStackTrace()
            } finally {
                _isLoading.value = false
            }
        }
    }
}
