package com.shub39.grit.core.habits.presentation.ui.component

import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.rounded.ArrowBack
import androidx.compose.material.icons.automirrored.rounded.ArrowForward
import androidx.compose.material.icons.rounded.CalendarMonth
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import com.kizitonwose.calendar.compose.CalendarState
import com.kizitonwose.calendar.compose.HorizontalCalendar
import com.kizitonwose.calendar.core.minusDays
import com.kizitonwose.calendar.core.plusDays
import com.shub39.grit.core.habits.domain.HabitWithAnalytics
import com.shub39.grit.core.habits.presentation.HabitsAction
import grit.shared.core.generated.resources.Res
import grit.shared.core.generated.resources.monthly_progress
import kotlinx.coroutines.launch
import kotlinx.datetime.DateTimeUnit
import kotlinx.datetime.TimeZone
import kotlinx.datetime.YearMonth
import kotlinx.datetime.format
import kotlinx.datetime.format.MonthNames
import kotlinx.datetime.format.char
import kotlinx.datetime.minus
import kotlinx.datetime.plus
import kotlinx.datetime.todayIn
import org.jetbrains.compose.resources.stringResource
import kotlin.time.Clock
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
@Composable
fun CalendarMap(
    canSeeContent: Boolean,
    onAction: (HabitsAction) -> Unit,
    calendarState: CalendarState,
    currentHabit: HabitWithAnalytics,
    primary: Color,
    modifier: Modifier = Modifier
) {
    val today = Clock.System.todayIn(TimeZone.currentSystemDefault())
    val scope = rememberCoroutineScope()

    AnalyticsCard(
        title = stringResource(Res.string.monthly_progress),
        icon = Icons.Rounded.CalendarMonth,
        canSeeContent = canSeeContent,
        onPlusClick = { onAction(HabitsAction.OnShowPaywall) },
        header = {
            Row {
                IconButton(
                    onClick = {
                        scope.launch {
                            calendarState.animateScrollToMonth(
                                calendarState.firstVisibleMonth.yearMonth.minus(
                                    1,
                                    DateTimeUnit.MONTH
                                )
                            )
                        }
                    }
                ) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Rounded.ArrowBack,
                        contentDescription = null
                    )
                }
                IconButton(
                    onClick = {
                        scope.launch {
                            calendarState.animateScrollToMonth(
                                calendarState.firstVisibleMonth.yearMonth.plus(
                                    1,
                                    DateTimeUnit.MONTH
                                )
                            )
                        }
                    }
                ) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Rounded.ArrowForward,
                        contentDescription = null
                    )
                }
            }
        },
        modifier = modifier
    ) {
        HorizontalCalendar(
            state = calendarState,
            modifier = Modifier
                .height(350.dp)
                .padding(bottom = 16.dp),
            contentPadding = PaddingValues(horizontal = 16.dp),
            userScrollEnabled = canSeeContent,
            monthHeader = {
                Box(
                    modifier = Modifier.padding(4.dp)
                ) {
                    Text(
                        text = it.yearMonth.format(YearMonth.Format {
                            monthName(MonthNames.ENGLISH_FULL)
                            char(' ')
                            year()
                        }),
                        color = MaterialTheme.colorScheme.secondary,
                        style = MaterialTheme.typography.titleMedium,
                        modifier = Modifier.align(Alignment.Center)
                    )
                }
            },
            dayContent = { day ->
                if (day.position.name == "MonthDate") {
                    val done = currentHabit.statuses.any { it.date == day.date }
                    val validDate =
                        day.date <= today && canSeeContent && day.date.dayOfWeek in currentHabit.habit.days

                    Box(
                        modifier = Modifier
                            .padding(2.dp)
                            .fillMaxWidth()
                            .height(45.dp)
                            .clickable(enabled = validDate) {
                                onAction(
                                    HabitsAction.InsertStatus(
                                        currentHabit.habit,
                                        day.date
                                    )
                                )
                            }
                            .then(
                                if (done) {
                                    val donePrevious =
                                        currentHabit.statuses.any { it.date == day.date.minusDays(1) }
                                    val doneAfter =
                                        currentHabit.statuses.any { it.date == day.date.plusDays(1) }

                                    Modifier.background(
                                        color = primary.copy(alpha = 0.2f),
                                        shape = if (donePrevious && doneAfter) {
                                            RoundedCornerShape(5.dp)
                                        } else if (donePrevious) {
                                            RoundedCornerShape(
                                                topStart = 5.dp,
                                                bottomStart = 5.dp,
                                                topEnd = 20.dp,
                                                bottomEnd = 20.dp
                                            )
                                        } else if (doneAfter) {
                                            RoundedCornerShape(
                                                topStart = 20.dp,
                                                bottomStart = 20.dp,
                                                topEnd = 5.dp,
                                                bottomEnd = 5.dp
                                            )
                                        } else {
                                            RoundedCornerShape(20.dp)
                                        }
                                    )
                                } else Modifier
                            ),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(
                            text = day.date.day.toString(),
                            style = MaterialTheme.typography.bodyLarge,
                            fontWeight = if (done) FontWeight.Bold else FontWeight.Normal,
                            color = if (done) primary
                            else if (!validDate) MaterialTheme.colorScheme.onSurface.copy(alpha = 0.5f)
                            else MaterialTheme.colorScheme.onSurface
                        )
                    }
                }
            }
        )
    }
}