package com.seafile.seadroid2.ui.account.sso;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.http.SslCertificate;
import android.net.http.SslError;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Pair;
import android.view.MenuItem;
import android.view.View;
import android.view.animation.AnimationUtils;
import android.webkit.CookieManager;
import android.webkit.SslErrorHandler;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.LinearLayout;

import androidx.appcompat.widget.Toolbar;
import androidx.lifecycle.Observer;

import com.blankj.utilcode.util.NetworkUtils;
import com.seafile.seadroid2.R;
import com.seafile.seadroid2.SeadroidApplication;
import com.seafile.seadroid2.SeafException;
import com.seafile.seadroid2.account.Account;
import com.seafile.seadroid2.databinding.SingleSignOnAuthorizeLayoutBinding;
import com.seafile.seadroid2.framework.util.DeviceIdManager;
import com.seafile.seadroid2.framework.util.SLogs;
import com.seafile.seadroid2.framework.util.Toasts;
import com.seafile.seadroid2.framework.util.Utils;
import com.seafile.seadroid2.ssl.CertsManager;
import com.seafile.seadroid2.ui.account.AccountViewModel;
import com.seafile.seadroid2.ui.account.SeafileAuthenticatorActivity;
import com.seafile.seadroid2.ui.base.BaseActivityWithVM;
import com.seafile.seadroid2.ui.dialog.SslConfirmDialog;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URLEncoder;
import java.security.cert.X509Certificate;

/**
 * Single sign on Authorize page, use cookie to get authorized data
 */
public class SingleSignOnAuthorizeActivity extends BaseActivityWithVM<AccountViewModel> implements Toolbar.OnMenuItemClickListener {
    public final String TAG = "SingleSignOnAuthorizeActivity";

    public static final String SEAHUB_SHIB_COOKIE_NAME = "seahub_auth";

    public String serverUrl;
    private SingleSignOnAuthorizeLayoutBinding binding;

    @SuppressLint("SetJavaScriptEnabled")
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        binding = SingleSignOnAuthorizeLayoutBinding.inflate(getLayoutInflater());

        setContentView(binding.getRoot());

        applyEdgeToEdge(binding.getRoot());


        binding.webview.getSettings().setLoadsImagesAutomatically(true);
        binding.webview.getSettings().setJavaScriptEnabled(true);
        binding.webview.getSettings().setUserAgentString(System.getProperty("http.agent"));
        binding.webview.setScrollBarStyle(View.SCROLLBARS_INSIDE_OVERLAY);

        CustomWebviewClient client = new CustomWebviewClient();
        binding.webview.setWebViewClient(client);

        Toolbar toolbar = getActionBarToolbar();
        setSupportActionBar(toolbar);
        toolbar.setOnMenuItemClickListener(this);
        getSupportActionBar().setDisplayHomeAsUpEnabled(true);
        getSupportActionBar().setTitle(R.string.shib_login_title);

        initViewModel();

        String url = getIntent().getStringExtra(SeafileAuthenticatorActivity.SINGLE_SIGN_ON_SERVER_URL);
        CookieManager.getInstance().removeAllCookie();
        openAuthorizePage(url);
    }

    private void openAuthorizePage(String url) {
        SLogs.d(TAG, "server url is " + url);
        if (TextUtils.isEmpty(url)) {
            Toasts.show(R.string.err_server_andress_empty);
            return;
        }

        serverUrl = url;

        if (!NetworkUtils.isConnected()) {
            Toasts.show(R.string.network_down);
            return;
        }

        String appVersion = "";
        Context context = SeadroidApplication.getAppContext();
        try {
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            appVersion = pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            // ignore
        }

        if (!url.endsWith("/")) {
            url += "/shib-login";
        } else
            url += "shib-login";

        //local device id
        String deviceId = DeviceIdManager.getInstance().getOrSet();

        try {
            url += String.format("?shib_platform_version=%s&shib_device_name=%s&shib_platform=%s&shib_device_id=%s&shib_client_version=%s",
                    URLEncoder.encode(Build.VERSION.RELEASE, "UTF-8"),
                    URLEncoder.encode(Build.MODEL, "UTF-8"),
                    "android",
                    deviceId,
                    appVersion);
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }

        SLogs.d(TAG, "url " + url);

        binding.webview.loadUrl(url);

        showPageLoading(true);
    }

    private void initViewModel() {
        getViewModel().getAccountSeafExceptionLiveData().observe(this, new Observer<Pair<Account, SeafException>>() {
            @Override
            public void onChanged(Pair<Account, SeafException> pair) {
                onLoginException(pair.first, pair.second);
            }
        });
        getViewModel().getAccountLiveData().observe(this, new Observer<Account>() {
            @Override
            public void onChanged(Account account) {
                onLoggedIn(account);
            }
        });
    }

    private void onLoginException(Account account, SeafException err) {
        if (err == SeafException.NETWORK_SSL_EXCEPTION) {
            SslConfirmDialog dialog = new SslConfirmDialog(account,
                    new SslConfirmDialog.Listener() {
                        @Override
                        public void onAccepted(boolean rememberChoice) {
                            CertsManager.instance().saveCertForAccount(account, rememberChoice);
                            getViewModel().loadAccountInfo(account, account.getToken());
                        }

                        @Override
                        public void onRejected() {

                        }
                    });
            dialog.show(getSupportFragmentManager(), SslConfirmDialog.FRAGMENT_TAG);
        }
    }

    private void onLoggedIn(Account account) {
        Intent retData = new Intent();
        retData.putExtras(getIntent());
        retData.putExtra(android.accounts.AccountManager.KEY_ACCOUNT_NAME, account.getSignature());
        retData.putExtra(android.accounts.AccountManager.KEY_AUTHTOKEN, account.getToken());
        retData.putExtra(android.accounts.AccountManager.KEY_ACCOUNT_TYPE, getIntent().getStringExtra(SeafileAuthenticatorActivity.ARG_ACCOUNT_TYPE));
        retData.putExtra(SeafileAuthenticatorActivity.ARG_EMAIL, account.getEmail());
        retData.putExtra(SeafileAuthenticatorActivity.ARG_NAME, account.getName());
        retData.putExtra(SeafileAuthenticatorActivity.ARG_SHIB, account.is_shib);
        retData.putExtra(SeafileAuthenticatorActivity.ARG_SERVER_URI, account.getServer());
        setResult(RESULT_OK, retData);
        finish();
    }

    private void showPageLoading(boolean pageLoading) {

        if (!pageLoading) {
            binding.singleSignOnLoadingLl.startAnimation(AnimationUtils.loadAnimation(
                    this, android.R.anim.fade_out));
            binding.webview.startAnimation(AnimationUtils.loadAnimation(
                    this, android.R.anim.fade_in));
            binding.singleSignOnLoadingLl.setVisibility(View.GONE);
            binding.webview.setVisibility(View.VISIBLE);
        } else {
            binding.singleSignOnLoadingLl.startAnimation(AnimationUtils.loadAnimation(
                    this, android.R.anim.fade_in));
            binding.webview.startAnimation(AnimationUtils.loadAnimation(
                    this, android.R.anim.fade_out));

            binding.singleSignOnLoadingLl.setVisibility(View.VISIBLE);
            binding.webview.setVisibility(View.INVISIBLE);
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == android.R.id.home) {
            finish();
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public boolean onMenuItemClick(MenuItem item) {
        return true;
    }

    private void displaySSLError() {
        showPageLoading(false);
        Toasts.show(R.string.ssl_error);
    }

    class CustomWebviewClient extends WebViewClient {
        @Override
        public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
            // Display error messages
            Toasts.show(String.format((R.string.shib_load_page_error) + description));

            showPageLoading(false);
        }

        @Override
        public void onReceivedSslError(WebView view, final SslErrorHandler handler, SslError error) {
            SLogs.d(TAG, "onReceivedSslError " + error.getCertificate().toString());

            final Account account = new Account(serverUrl, null, null, null, null, false);
            SslCertificate sslCert = error.getCertificate();

            X509Certificate savedCert = CertsManager.instance().getCertificate(account);
            if (Utils.isSameCert(sslCert, savedCert)) {
                SLogs.d(TAG, "trust this cert");
                handler.proceed();
            } else {
                SLogs.d(TAG, "cert is not trusted");
                SslConfirmDialog dialog = new SslConfirmDialog(account,
                        Utils.getX509CertFromSslCertHack(sslCert),
                        new SslConfirmDialog.Listener() {
                            @Override
                            public void onAccepted(boolean rememberChoice) {
                                CertsManager.instance().saveCertForAccount(account, rememberChoice);
                                // Ignore SSL certificate validate
                                handler.proceed();
                            }

                            @Override
                            public void onRejected() {
                                displaySSLError();
                                handler.cancel();
                            }
                        });
                dialog.show(getSupportFragmentManager(), SslConfirmDialog.FRAGMENT_TAG);
            }
        }

        @Override
        public void onPageFinished(WebView webView, String url) {
            SLogs.d(TAG, "onPageFinished " + serverUrl);
            showPageLoading(false);

            String cookie = getCookie(serverUrl, SEAHUB_SHIB_COOKIE_NAME);
            if (cookie == null)
                return;

            Account account;
            try {
                account = parseAccount(Utils.cleanServerURL(serverUrl), cookie);
                if (account == null) {
                    return;
                }

                getViewModel().loadAccountInfo(account, account.getToken());
            } catch (MalformedURLException e) {
                SLogs.e(e);
            }
        }
    }

    /**
     * The cookie value is like seahub_shib="foo@test.com@bd8cc1138", where
     * foo@test.com is username and bd8cc1138 is api token"
     */
    private Account parseAccount(String url, String cookie) {
        if (url == null || cookie.isEmpty())
            return null;

        if (cookie.startsWith("\"")) {
            cookie = cookie.substring(1, cookie.length() - 1);
        }

        String email = cookie.substring(0, cookie.lastIndexOf("@"));
        String token = cookie.substring(cookie.lastIndexOf("@") + 1);

        if (email.isEmpty() || token.isEmpty())
            return null;
        SLogs.d(TAG, "email: " + email);
        SLogs.d(TAG, "token: *********");

        return new Account(url, email, "", null, token, true);
    }

    public String getCookie(String url, String key) {
        String CookieValue = "";

        CookieManager.getInstance().setAcceptCookie(true);
        CookieManager cookieManager = CookieManager.getInstance();

        String cookies = cookieManager.getCookie(url);
        if (cookies == null)
            return null;

        SLogs.d(TAG, "find all the cookies");

        String[] allCookies = cookies.split(";");

        for (String cookie : allCookies) {
            if (cookie.contains(key)) {
                String[] pair = cookie.split("=");
                CookieValue = pair[1];
            }
        }
        return CookieValue;
    }

}
