# Lua Project Structure

This directory contains Lua source code, tooling, and build artifacts for Track & Graph's Lua integration.

## Community Functions

Community functions are Lua scripts that transform data point streams for use in the node editor. They live in `src/community/functions/` with each function in its own directory:

```
src/community/functions/
├── multiply/
│   ├── multiply.lua
│   └── test_multiply.lua
├── override-label/
│   ├── override-label.lua
│   └── test_override-label.lua
└── filter-by-label/
    ├── filter-by-label.lua
    └── test_filter-by-label.lua
```

## Building the Function Catalog 

The function catalog is a single Lua file that bundles all community functions for distribution. The Android app downloads this catalog at runtime to discover available functions. The catalog is generated by `tools/pack-functions.lua` and output to `catalog/community-functions.lua`.

### Prerequisites

Install required Lua dependencies:

```bash
brew install lua luarocks
luarocks install serpent
```

### Running the Packer

From the `lua/` directory, run:

```bash
lua tools/pack-functions.lua
```

## Other Development Tools

The packer will run several steps before generating the catalog. You can run these steps independently as described below:

### Validate API Specs
Ensures all TNG API exports have corresponding API level specifications:
```bash
lua tools/verify-api-specs.lua
```

### Get Max API Level
Returns the highest API level defined across all specs:
```bash
lua tools/get-max-api-level.lua
```

### Validate Functions
Validates all community functions have required fields and translations:
```bash
lua tools/validate-functions.lua
```

### Validate Function Categories
Validates that all function categories are defined and used:
```bash
lua tools/validate-function-categories.lua
```

### Print Catalog
Prints all functions with their English titles and descriptions. Optionally filter by API level:
```bash
# From repository root (shows all functions):
make lua-print-catalog

# From lua/ directory (with optional API level filter):
cd lua
lua tools/print-catalog.lua          # All functions
lua tools/print-catalog.lua 1        # Only functions compatible with API level 1
```

### Detect Changes
Compares current functions against the published catalog to detect changes:
```bash
lua tools/detect-changes.lua
```

### Publish for Debug
Signs the catalog and publishes to debug assets with a fresh keypair:
```bash
lua tools/publish-functions-debug.lua
```
Creates three files in `app/src/debug/assets/functions-catalog/`:
- `community-functions.lua` - The catalog
- `community-functions.sig.json` - Signature metadata
- `debug-<timestamp>.pub` - Public key (referenced by keyId in JSON)

---

# Advanced: Adding New Lua Configuration Types

Guide for adding new configuration input types (text, number, checkbox, enum, etc.) to Track & Graph's UI.

## Quick Reference

7 files to update + 5 test files:

**Core Implementation:**
1. `LuaFunctionConfigSpec` - Data DTO (what Lua defines)
2. `LuaScriptConfigurationValue` - Database serialization
3. `LuaScriptConfigurationInput` - ViewModel with mutable state
4. `LuaScriptConfigurationInputFactory` - Creates inputs from specs
5. `LuaScriptConfigurationEncoder` - ViewModel → Database
6. `ConfigurationValueParser` - Database → Lua VM
7. `ConfigurationInputField` - UI component

**Tests (will fail if you miss steps):**
8. `LuaFunctionMetadataTests` - Parsing from Lua
9. `LuaScriptConfigurationInputFactoryTest` - Factory coverage
10. `LuaScriptConfigurationEncoderTest` - Encoding coverage
11. `ConfigurationValueParserTest` - Parser coverage
12. `FunctionGraphSerializerTest` - Serialization coverage

## Implementation Steps

### 1. Add to LuaFunctionConfigSpec

**File**: `data/src/main/java/com/samco/trackandgraph/data/lua/dto/LuaFunctionMetadata.kt`

```kotlin
sealed class LuaFunctionConfigSpec {
    abstract val id: String
    abstract val name: TranslatedString?

    data class YourType(
        override val id: String,
        override val name: TranslatedString?,
        val defaultValue: YourValueType? = null
    ) : LuaFunctionConfigSpec()
}
```

### 2. Add to LuaScriptConfigurationValue

**File**: `data/src/main/java/com/samco/trackandgraph/data/database/dto/FunctionGraph.kt`

```kotlin
@Serializable
@SerialName("YourType")  // PascalCase
data class YourType(
    override val id: String,
    val value: YourValueType
) : LuaScriptConfigurationValue()
```

### 3. Add to LuaScriptConfigurationInput

**File**: `app/src/main/java/com/samco/trackandgraph/functions/viewmodel/LuaScriptConfigurationInput.kt`

```kotlin
data class YourType(
    override val name: TranslatedString?,
    val value: MutableState<YourUIType> = mutableStateOf(defaultValue)
) : LuaScriptConfigurationInput()
```

### 4. Update LuaScriptConfigurationInputFactory

**File**: `app/src/main/java/com/samco/trackandgraph/functions/viewmodel/LuaScriptConfigurationInputFactory.kt`

Add case to `createConfigurationInput()`:
```kotlin
is LuaFunctionConfigSpec.YourType -> createYourTypeInput(config, savedValue)
```

Add case to `isCompatibleType()`:
```kotlin
is LuaFunctionConfigSpec.YourType -> existingInput is LuaScriptConfigurationInput.YourType
```

Add method:
```kotlin
private fun createYourTypeInput(
    config: LuaFunctionConfigSpec.YourType,
    savedValue: LuaScriptConfigurationValue?
): LuaScriptConfigurationInput.YourType {
    val typedValue = savedValue as? LuaScriptConfigurationValue.YourType
    val initialValue = typedValue?.value ?: config.defaultValue ?: fallback

    return LuaScriptConfigurationInput.YourType(
        name = config.name,
        value = mutableStateOf(initialValue)
    )
}
```

### 5. Update LuaScriptConfigurationEncoder

**File**: `app/src/main/java/com/samco/trackandgraph/functions/viewmodel/LuaScriptConfigurationEncoder.kt`

```kotlin
is LuaScriptConfigurationInput.YourType -> {
    LuaScriptConfigurationValue.YourType(
        id = id,
        value = input.value.value
    )
}
```

### 6. Update ConfigurationValueParser

**File**: `data/src/main/java/com/samco/trackandgraph/data/lua/apiimpl/ConfigurationValueParser.kt`

```kotlin
is LuaScriptConfigurationValue.YourType -> LuaValue.valueOf(value.value)
```

### 7. Update LuaFunctionMetadataAdapter

**File**: `data/src/main/java/com/samco/trackandgraph/data/lua/functionadapters/LuaFunctionMetadataAdapter.kt`

Add case to `parseConfigItem()`:
```kotlin
"your_type" -> parseYourTypeConfig(id, nameTranslations, configItem)
```

Add method:
```kotlin
private fun parseYourTypeConfig(
    id: String,
    name: TranslatedString?,
    configItem: LuaValue
): LuaFunctionConfigSpec.YourType {
    val defaultValue = configItem[DEFAULT]
        .takeUnless { it.isnil() }?.checkYourType()

    return LuaFunctionConfigSpec.YourType(
        id = id,
        name = name,
        defaultValue = defaultValue
    )
}
```

### 8. Add UI Component

**File**: `app/src/main/java/com/samco/trackandgraph/functions/node_editor/ConfigurationInputField.kt`

Add case:
```kotlin
is LuaScriptConfigurationInput.YourType -> YourTypeField(input)
```

Add component:
```kotlin
@Composable
private fun YourTypeField(input: LuaScriptConfigurationInput.YourType) {
    // TODO: Implement UI
}
```

## Test Updates

### 1. LuaFunctionMetadataTests
Add to `Function handles all configuration types` test in Lua script and assertions.

### 2. LuaScriptConfigurationInputFactoryTest
Add to `allTypesMetadata.config` and `allTypesConfig`.

### 3. LuaScriptConfigurationEncoderTest
Add to class-level inputs and `encodeConfiguration handles all` test.

### 4. ConfigurationValueParserTest
Add to `parseConfigurationValues covers all` test.

### 5. FunctionGraphSerializerTest
Add to `createTestFunctionGraph()` configuration list.

## Notes

- **Naming**: PascalCase for `@SerialName`, snake_case in Lua
- **Tests use reflection**: Will fail if you miss a type
- **State**: Always use `MutableState<T>` for UI reactivity
- **Defaults**: Handle null values gracefully (saved value → default → fallback)
