/*
 *  This file is part of Track & Graph
 *
 *  Track & Graph is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Track & Graph is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Track & Graph.  If not, see <https://www.gnu.org/licenses/>.
 */

package com.samco.trackandgraph

import com.samco.trackandgraph.data.database.dto.DataType
import com.samco.trackandgraph.data.database.dto.Function
import com.samco.trackandgraph.data.database.dto.TrackerSuggestionOrder
import com.samco.trackandgraph.data.database.dto.TrackerSuggestionType
import com.samco.trackandgraph.data.interactor.DataInteractor
import kotlinx.serialization.json.Json

// Function graph JSON strings for Functions Tutorial group

// Function: Exercice - combines Running and Cycling features
private fun exercice_function_graph(runningFeatureId: Long, cyclingFeatureId: Long) =
    """{"nodes":[{"type":"FeatureNode","x":-1502.8608,"y":-311.32462,"id":2,"featureId":""" + runningFeatureId + """},{"type":"FeatureNode","x":-1491.5502,"y":310.2478,"id":3,"featureId":""" + cyclingFeatureId + """}],"outputNode":{"x":0.0,"y":0.0,"id":1,"dependencies":[{"connectorIndex":0,"nodeId":3},{"connectorIndex":0,"nodeId":2}]},"isDuration":false}"""

// Function: Exercice This Week - filters Exercice with periodic generator
private fun exercice_this_week_function_graph(runningFeatureId: Long, cyclingFeatureId: Long) =
    """{"nodes":[{"type":"FeatureNode","x":-2716.8413,"y":-839.1216,"id":2,"featureId":""" + runningFeatureId + """},{"type":"FeatureNode","x":-2716.825,"y":-408.99643,"id":3,"featureId":""" + cyclingFeatureId + """},{"type":"LuaScriptNode","x":-2708.9812,"y":339.08325,"id":4,"script":"-- Lua Function to generate periodic data points at regular intervals\n-- This function creates data points with value=1 at deterministic timestamps\nlocal core = require(\"tng.core\")\nlocal enum = require(\"tng.config\").enum\nlocal uint = require(\"tng.config\").uint\nlocal instant = require(\"tng.config\").instant\nlocal now_time = core.time()\nlocal now = now_time and now_time.timestamp or 0\nreturn {\n    -- Configuration metadata\n    id = \"periodic-data-points\",\n    version = \"1.1.1\",\n    inputCount = 0, -- This is a generator, not a transformer\n    categories = { \"_generators\" },\n    title = {\n        [\"en\"] = \"Periodic Data Points\",\n        [\"de\"] = \"Periodische Datenpunkte\",\n        [\"es\"] = \"Puntos de Datos Periódicos\",\n        [\"fr\"] = \"Points de Données Périodiques\",\n    },\n    description = {\n        [\"en\"] = [[\nGenerates data points with value=1 at regular intervals going back in time.\nConfiguration:\n- **Period**: Time period unit (Day, Week, Month, Year)\n- **Period Multiplier**: Generate data point every N periods (e.g., every 2 days)\n- **Cutoff**: Stop generating data points at this date/time\nGenerated data points will have:\n- value = 1.0\n- label = \"\" (empty)\n- note = \"\" (empty)]],\n        [\"de\"] = [[\nGeneriert Datenpunkte mit Wert=1 in regelmäßigen Abständen zurück in der Zeit.\nKonfiguration:\n- **Periode**: Zeitperiodeneinheit (Tag, Woche, Monat, Jahr)\n- **Periodenmultiplikator**: Datenpunkt alle N Perioden generieren (z.B. alle 2 Tage)\n- **Grenzwert**: Generierung bei diesem Datum/Zeit stoppen\nGenerierte Datenpunkte haben:\n- Wert = 1.0\n- Label = \"\" (leer)\n- Notiz = \"\" (leer)]],\n        [\"es\"] = [[\nGenera puntos de datos con valor=1 a intervalos regulares retrocediendo en el tiempo.\nConfiguración:\n- **Período**: Unidad de período de tiempo (Día, Semana, Mes, Año)\n- **Multiplicador de Período**: Generar punto de datos cada N períodos (ej. cada 2 días)\n- **Límite**: Detener generación de puntos de datos en esta fecha/hora\nLos puntos de datos generados tendrán:\n- valor = 1.0\n- etiqueta = \"\" (vacío)\n- nota = \"\" (vacío)]],\n        [\"fr\"] = [[\nGénère des points de données avec valeur=1 à intervalles réguliers en remontant dans le temps.\nConfiguration:\n- **Période**: Unité de période de temps (Jour, Semaine, Mois, Année)\n- **Multiplicateur de Période**: Générer un point de données tous les N périodes (ex. tous les 2 jours)\n- **Limite**: Arrêter la génération de points de données à cette date/heure\nLes points de données générés auront:\n- valeur = 1.0\n- étiquette = \"\" (vide)\n- note = \"\" (vide)]],\n    },\n    config = {\n        enum {\n            id = \"period\",\n            name = \"_period\",\n            options = { \"_day\", \"_week\", \"_month\", \"_year\" },\n            default = \"_day\",\n        },\n        uint {\n            id = \"period_multiplier\",\n            name = \"_period_multiplier\",\n            default = 1,\n        },\n        instant {\n            id = \"cutoff\",\n            name = \"_cutoff\",\n            default = now - (365 * core.DURATION.DAY),\n        },\n    },\n    -- Generator function\n    generator = function(_, config)\n        -- Parse configuration with defaults\n        local period_str = config and config.period or error(\"Period configuration is required\")\n        local period_multiplier = (config and config.period_multiplier) or 1\n        -- Don't allow 0 multiplier, fallback to 1\n        if period_multiplier == 0 then\n            period_multiplier = 1\n        end\n        local cutoff_timestamp = config and config.cutoff or error(\"Cutoff configuration is required\")\n        -- Map enum string to core.PERIOD constant\n        local period_map = {\n            [\"_day\"] = core.PERIOD.DAY,\n            [\"_week\"] = core.PERIOD.WEEK,\n            [\"_month\"] = core.PERIOD.MONTH,\n            [\"_year\"] = core.PERIOD.YEAR,\n        }\n        local period = period_map[period_str]\n        -- Get current time for comparison\n        local now = core.time().timestamp\n        -- If cutoff is in the future, no data points to generate\n        if cutoff_timestamp > now then\n            return function()\n                return nil\n            end\n        end\n        -- Estimate number of periods elapsed since anchor\n        local elapsed_ms = now - cutoff_timestamp\n        local estimated_periods\n        local period_duration_ms\n        if period == core.PERIOD.DAY then\n            period_duration_ms = period_multiplier * core.DURATION.DAY\n        elseif period == core.PERIOD.WEEK then\n            period_duration_ms = period_multiplier * core.DURATION.WEEK\n        elseif period == core.PERIOD.MONTH then\n            -- Average month length: 30.44 days\n            period_duration_ms = period_multiplier * 30.44 * core.DURATION.DAY\n        elseif period == core.PERIOD.YEAR then\n            -- Average year length: 365.25 days\n            period_duration_ms = period_multiplier * 365.25 * core.DURATION.DAY\n        else\n            error(\"Invalid period: \" .. tostring(period_str))\n        end\n        estimated_periods = math.floor(elapsed_ms / period_duration_ms)\n        local cutoff_date = core.date(cutoff_timestamp)\n        -- Jump close to now with one large shift\n        local candidate = core.shift(cutoff_date, period, estimated_periods * period_multiplier)\n        -- Fine-tune: shift forward until we pass \"now\"\n        while candidate.timestamp <= now do\n            candidate = core.shift(candidate, period, period_multiplier)\n        end\n        -- Back up one step to get the most recent data point <= now\n        local current = core.shift(candidate, period, -period_multiplier)\n        -- Return iterator function\n        return function()\n            -- Check if we've gone past the cutoff (with 1 second tolerance for millisecond precision loss)\n            if current.timestamp < cutoff_timestamp - 1000 then\n                return nil\n            end\n            -- Create data point at current timestamp\n            local data_point = {\n                timestamp = current.timestamp,\n                offset = current.offset,\n                value = 1.0,\n                label = \"\",\n                note = \"\",\n            }\n            -- Shift backwards by period * period_multiplier for next iteration\n            current = core.shift(current, period, -period_multiplier)\n            return data_point\n        end\n    end,\n}\n","inputConnectorCount":0,"configuration":[{"configType":"Enum","id":"period","value":"_week"},{"configType":"UInt","id":"period_multiplier","value":1},{"configType":"Instant","id":"cutoff","epochMilli":1731283213735}],"translations":{"_generators":{"type":"Translations","translations":{"en":"Generators","es":"Generadores","de":"Generatoren","fr":"Générateurs"}},"_period":{"type":"Translations","translations":{"en":"Period","es":"Período","de":"Periode","fr":"Période"}},"_day":{"type":"Translations","translations":{"en":"Day","es":"Día","de":"Tag","fr":"Jour"}},"_week":{"type":"Translations","translations":{"en":"Week","es":"Semana","de":"Woche","fr":"Semaine"}},"_month":{"type":"Translations","translations":{"en":"Month","es":"Mes","de":"Monat","fr":"Mois"}},"_year":{"type":"Translations","translations":{"en":"Year","es":"Año","de":"Jahr","fr":"Année"}},"_period_multiplier":{"type":"Translations","translations":{"en":"Period Multiplier","es":"Multiplicador de Período","de":"Periodenmultiplikator","fr":"Multiplicateur de Période"}},"_cutoff":{"type":"Translations","translations":{"en":"Cutoff","es":"Límite","de":"Grenzwert","fr":"Limite"}}},"catalogFunctionId":"periodic-data-points","catalogVersion":"1.1.1","dependencies":[]},{"type":"LuaScriptNode","x":-1309.9325,"y":-6.859787,"id":5,"script":"-- Lua Function to filter data points after a reference point\n-- Outputs all data points from the first source that come after the last point in the second source\nreturn {\n\t-- Configuration metadata\n\tid = \"filter-after-last\",\n\tversion = \"1.0.0\",\n\tinputCount = 2,\n\tcategories = {\"_filter\"},\n\ttitle = {\n\t\t[\"en\"] = \"Filter After Last\",\n\t\t[\"de\"] = \"Filtern nach Letztem\",\n\t\t[\"es\"] = \"Filtrar después del último\",\n\t\t[\"fr\"] = \"Filtrer après le dernier\",\n\t},\n\tdescription = {\n\t\t[\"en\"] = [[\nFilters data points from the first input source to only include those that occur after the last data point in the second input source.\nThis is useful for filtering data based on a reference event or timestamp from another tracker.\n]],\n\t\t[\"de\"] = [[\nFiltert Datenpunkte aus der ersten Eingabequelle, um nur diejenigen einzuschließen, die nach dem letzten Datenpunkt in der zweiten Eingabequelle auftreten.\nDies ist nützlich zum Filtern von Daten basierend auf einem Referenzereignis oder Zeitstempel von einem anderen Tracker.\n]],\n\t\t[\"es\"] = [[\nFiltra puntos de datos de la primera fuente de entrada para incluir solo aquellos que ocurren después del último punto de datos en la segunda fuente de entrada.\nEsto es útil para filtrar datos basados en un evento de referencia o marca de tiempo de otro rastreador.\n]],\n\t\t[\"fr\"] = [[\nFiltre les points de données de la première source d'entrée pour n'inclure que ceux qui se produisent après le dernier point de données de la deuxième source d'entrée.\nCeci est utile pour filtrer les données basées sur un événement de référence ou un horodatage d'un autre tracker.\n]],\n\t},\n\tconfig = {},\n\t-- Generator function\n\tgenerator = function(sources, config)\n\t\tlocal source1 = sources[1]\n\t\tlocal source2 = sources[2]\n\t\tlocal cutoff_timestamp = nil\n\t\treturn function()\n\t\t\t-- Initialize cutoff on first call\n\t\t\tif cutoff_timestamp == nil then\n\t\t\t\tlocal reference_point = source2.dp()\n\t\t\t\tcutoff_timestamp = reference_point and reference_point.timestamp\n\t\t\tend\n\t\t\t-- Get next point from source1 and check if it's after cutoff\n\t\t\tlocal data_point = source1.dp()\n\t\t\tif not data_point then\n\t\t\t\treturn nil\n\t\t\tend\n\t\t\t-- Data points are in reverse chronological order, so \"after\" means greater timestamp\n\t\t\tif not cutoff_timestamp or data_point.timestamp > cutoff_timestamp then\n\t\t\t\treturn data_point\n\t\t\tend\n\t\t\t-- If the data point is not after the cutoff we're done\n\t\t\treturn nil\n\t\tend\n\tend,\n}\n","inputConnectorCount":2,"translations":{"_filter":{"type":"Translations","translations":{"en":"Filter","es":"Filtro","de":"Filter","fr":"Filtre"}}},"catalogFunctionId":"filter-after-last","catalogVersion":"1.0.0","dependencies":[{"connectorIndex":0,"nodeId":3},{"connectorIndex":0,"nodeId":2},{"connectorIndex":1,"nodeId":4}]}],"outputNode":{"x":0.0,"y":0.0,"id":1,"dependencies":[{"connectorIndex":0,"nodeId":5}]},"isDuration":false}"""

// Function: Weight (Kg) - converts Weight (lbs) to kg
private fun weight_kg_function_graph(weightLbsFeatureId: Long) =
    """{"nodes":[{"type":"FeatureNode","x":-2415.193,"y":-14.743446,"id":2,"featureId":""" + weightLbsFeatureId + """},{"type":"LuaScriptNode","x":-1305.376,"y":-15.362221,"id":3,"script":"-- Lua Function to multiply data point values by a configurable number\n-- This function multiplies all incoming data point values by a specified multiplier\nlocal number = require(\"tng.config\").number\nreturn {\n    -- Configuration metadata\n    id = \"multiply\",\n    version = \"1.0.0\",\n    inputCount = 1,\n    categories = {\"_arithmetic\"},\n    title = {\n        [\"en\"] = \"Multiply Values\",\n        [\"de\"] = \"Werte multiplizieren\",\n        [\"es\"] = \"Multiplicar Valores\",\n        [\"fr\"] = \"Multiplier les Valeurs\"\n    },\n    description = {\n        [\"en\"] = [[\nMultiplies all incoming data point values by a specified multiplier.\nConfiguration:\n- **Multiplier**: The number to multiply all values by (default: 1.0)\n]],\n        [\"de\"] = [[\nMultipliziert alle eingehenden Datenpunktwerte mit einem bestimmten Multiplikator.\nKonfiguration:\n- **Multiplikator**: Die Zahl, mit der alle Werte multipliziert werden (Standard: 1.0)\n]],\n        [\"es\"] = [[\nMultiplica todos los valores de puntos de datos entrantes por un multiplicador especificado.\nConfiguración:\n- **Multiplicador**: El número por el cual multiplicar todos los valores (predeterminado: 1.0)\n]],\n        [\"fr\"] = [[\nMultiplie toutes les valeurs de points de données entrantes par un multiplicateur spécifié.\nConfiguration:\n- **Multiplicateur**: Le nombre par lequel multiplier toutes les valeurs (par défaut: 1.0)\n]]\n    },\n    config = {\n        number {\n            id = \"multiplier\",\n            name = {\n                [\"en\"] = \"Multiplier\",\n                [\"de\"] = \"Multiplikator\",\n                [\"es\"] = \"Multiplicador\",\n                [\"fr\"] = \"Multiplicateur\"\n            }\n        }\n    },\n    -- Generator function\n    generator = function(source, config)\n        local multiplier = config and config.multiplier or 1.0\n        return function()\n            local data_point = source.dp()\n            if not data_point then return nil end\n            data_point.value = data_point.value * multiplier\n            return data_point\n        end\n    end\n}\n","inputConnectorCount":1,"configuration":[{"configType":"Number","id":"multiplier","value":0.453}],"translations":{"_arithmetic":{"type":"Translations","translations":{"en":"Arithmetic","es":"Aritmética","de":"Arithmetik","fr":"Arithmétique"}}},"catalogFunctionId":"multiply","catalogVersion":"1.0.0","dependencies":[{"connectorIndex":0,"nodeId":2}]}],"outputNode":{"x":-27.202393,"y":-15.375336,"id":1,"dependencies":[{"connectorIndex":0,"nodeId":3}]},"isDuration":false}"""

// Function: All Weight (Kg) - merges weight from both kg and lbs labels
private fun all_weight_kg_function_graph(weightFeatureId: Long) =
    """{"nodes":[{"type":"FeatureNode","x":-4292.038,"y":234.4284,"id":2,"featureId":""" + weightFeatureId + """},{"type":"LuaScriptNode","x":-1422.5459,"y":-547.7311,"id":3,"script":"-- Lua Function to multiply data point values by a configurable number\n-- This function multiplies all incoming data point values by a specified multiplier\nlocal number = require(\"tng.config\").number\nreturn {\n    -- Configuration metadata\n    id = \"multiply\",\n    version = \"1.0.0\",\n    inputCount = 1,\n    categories = {\"_arithmetic\"},\n    title = {\n        [\"en\"] = \"Multiply Values\",\n        [\"de\"] = \"Werte multiplizieren\",\n        [\"es\"] = \"Multiplicar Valores\",\n        [\"fr\"] = \"Multiplier les Valeurs\"\n    },\n    description = {\n        [\"en\"] = [[\nMultiplies all incoming data point values by a specified multiplier.\nConfiguration:\n- **Multiplier**: The number to multiply all values by (default: 1.0)\n]],\n        [\"de\"] = [[\nMultipliziert alle eingehenden Datenpunktwerte mit einem bestimmten Multiplikator.\nKonfiguration:\n- **Multiplikator**: Die Zahl, mit der alle Werte multipliziert werden (Standard: 1.0)\n]],\n        [\"es\"] = [[\nMultiplica todos los valores de puntos de datos entrantes por un multiplicador especificado.\nConfiguración:\n- **Multiplicador**: El número por el cual multiplicar todos los valores (predeterminado: 1.0)\n]],\n        [\"fr\"] = [[\nMultiplie toutes les valeurs de points de données entrantes par un multiplicateur spécifié.\nConfiguration:\n- **Multiplicateur**: Le nombre par lequel multiplier toutes les valeurs (par défaut: 1.0)\n]]\n    },\n    config = {\n        number {\n            id = \"multiplier\",\n            name = {\n                [\"en\"] = \"Multiplier\",\n                [\"de\"] = \"Multiplikator\",\n                [\"es\"] = \"Multiplicador\",\n                [\"fr\"] = \"Multiplicateur\"\n            }\n        }\n    },\n    -- Generator function\n    generator = function(source, config)\n        local multiplier = config and config.multiplier or 1.0\n        return function()\n            local data_point = source.dp()\n            if not data_point then return nil end\n            data_point.value = data_point.value * multiplier\n            return data_point\n        end\n    end\n}\n","inputConnectorCount":1,"configuration":[{"configType":"Number","id":"multiplier","value":0.453}],"translations":{"_arithmetic":{"type":"Translations","translations":{"en":"Arithmetic","es":"Aritmética","de":"Arithmetik","fr":"Arithmétique"}}},"catalogFunctionId":"multiply","catalogVersion":"1.0.0","dependencies":[{"connectorIndex":0,"nodeId":4}]},{"type":"LuaScriptNode","x":-2817.5232,"y":-540.92615,"id":4,"script":"-- Example Lua Function with Input Count and Configuration\n-- This function filters data points by label\nlocal tng_config = require(\"tng.config\")\nlocal text = tng_config.text\nlocal checkbox = tng_config.checkbox\nlocal function match(data_point, filter_label, case_sensitive, match_exactly)\n    if filter_label == nil then\n        return true\n    end\n    local data_label = data_point.label\n    if not data_label then return false end\n    -- Apply case sensitivity\n    if not case_sensitive then\n        data_label = string.lower(data_label)\n        filter_label = string.lower(filter_label)\n    end\n    -- Apply matching mode\n    if match_exactly then\n        return data_label == filter_label\n    else\n        return string.find(data_label, filter_label, 1, true) ~= nil\n    end\nend\nreturn {\n    -- Configuration metadata\n    id = \"filter-by-label\",\n    version = \"1.0.1\",\n    inputCount = 1,\n    categories = {\"_filter\"},\n    title = {\n        [\"en\"] = \"Filter by Label\",\n        [\"de\"] = \"Filtern nach Etikett\",\n        [\"es\"] = \"Filtrar por Etiqueta\",\n        [\"fr\"] = \"Filtrer par Étiquette\"\n    },\n    description = {\n        [\"en\"] = [[\nFilters data points by their label field. Only data points matching the filter criteria will pass through.\nConfiguration:\n- **Filter Label**: The text to search for in labels\n- **Case Sensitive**: Match case exactly (default: false)\n- **Match Exactly**: Require exact match instead of substring (default: false)\n- **Invert**: Keep data points that DON'T match instead (default: false)\n]],\n        [\"de\"] = [[\nFiltert Datenpunkte nach ihrem Label-Feld. Nur Datenpunkte, die den Filterkriterien entsprechen, werden durchgelassen.\nKonfiguration:\n- **Filter-Label**: Der Text, nach dem in Labels gesucht werden soll\n- **Groß-/Kleinschreibung beachten**: Groß-/Kleinschreibung exakt beachten (Standard: false)\n- **Exakt übereinstimmen**: Exakte Übereinstimmung statt Teilstring erforderlich (Standard: false)\n- **Invertieren**: Datenpunkte behalten, die NICHT übereinstimmen (Standard: false)\n]],\n        [\"es\"] = [[\nFiltra puntos de datos por su campo de etiqueta. Solo los puntos de datos que coincidan con los criterios del filtro pasarán.\nConfiguración:\n- **Filtrar Etiqueta**: El texto a buscar en las etiquetas\n- **Sensible a Mayúsculas**: Coincidir exactamente con mayúsculas y minúsculas (predeterminado: false)\n- **Coincidir Exactamente**: Requerir coincidencia exacta en lugar de subcadena (predeterminado: false)\n- **Invertir**: Mantener puntos de datos que NO coincidan (predeterminado: false)\n]],\n        [\"fr\"] = [[\nFiltre les points de données par leur champ d'étiquette. Seuls les points de données correspondant aux critères du filtre passeront.\nConfiguration:\n- **Filtrer l'Étiquette**: Le texte à rechercher dans les étiquettes\n- **Sensible à la Casse**: Correspondance exacte de la casse (par défaut: false)\n- **Correspondance Exacte**: Nécessite une correspondance exacte au lieu d'une sous-chaîne (par défaut: false)\n- **Inverser**: Conserver les points de données qui NE correspondent PAS (par défaut: false)\n]]\n    },\n    config = {\n        text {\n            id = \"filter_label\",\n            name = {\n                [\"en\"] = \"Filter Label\",\n                [\"de\"] = \"Filter-Label\",\n                [\"es\"] = \"Filtrar Etiqueta\",\n                [\"fr\"] = \"Filtrer l'Étiquette\"\n            }\n        },\n        checkbox {\n            id = \"case_sensitive\",\n            name = \"_case_sensitive\",\n        },\n        checkbox {\n            id = \"match_exactly\",\n            name = \"_match_exactly\",\n        },\n        checkbox {\n            id = \"invert\",\n            name = {\n                [\"en\"] = \"Invert\",\n                [\"de\"] = \"Invertieren\",\n                [\"es\"] = \"Invertir\",\n                [\"fr\"] = \"Inverser\"\n            }\n        }\n    },\n    -- Generator function\n    generator = function(source, config)\n        local filter_label = config and config.filter_label\n        local case_sensitive = config and config.case_sensitive or false\n        local match_exactly = config and config.match_exactly or false\n        local invert = config and config.invert or false\n        return function()\n            local data_point = source.dp()\n            local should_match = not invert\n            while data_point and (match(data_point, filter_label, case_sensitive, match_exactly) ~= should_match) do\n                data_point = source.dp()\n            end\n            return data_point\n        end\n    end\n}\n","inputConnectorCount":1,"configuration":[{"configType":"Text","id":"filter_label","value":"lb"},{"configType":"Checkbox","id":"case_sensitive","value":false},{"configType":"Checkbox","id":"match_exactly","value":false},{"configType":"Checkbox","id":"invert","value":false}],"translations":{"_filter":{"type":"Translations","translations":{"en":"Filter","es":"Filtro","de":"Filter","fr":"Filtre"}},"_case_sensitive":{"type":"Translations","translations":{"en":"Case Sensitive","es":"Distinguir Mayúsculas","de":"Groß-/Kleinschreibung beachten","fr":"Sensible à la Casse"}},"_match_exactly":{"type":"Translations","translations":{"en":"Match Exactly","es":"Coincidir Exactamente","de":"Exakt übereinstimmen","fr":"Correspondance Exacte"}}},"catalogFunctionId":"filter-by-label","catalogVersion":"1.0.1","dependencies":[{"connectorIndex":0,"nodeId":2}]},{"type":"LuaScriptNode","x":-2810.216,"y":817.02655,"id":5,"script":"-- Example Lua Function with Input Count and Configuration\n-- This function filters data points by label\nlocal tng_config = require(\"tng.config\")\nlocal text = tng_config.text\nlocal checkbox = tng_config.checkbox\nlocal function match(data_point, filter_label, case_sensitive, match_exactly)\n    if filter_label == nil then\n        return true\n    end\n    local data_label = data_point.label\n    if not data_label then return false end\n    -- Apply case sensitivity\n    if not case_sensitive then\n        data_label = string.lower(data_label)\n        filter_label = string.lower(filter_label)\n    end\n    -- Apply matching mode\n    if match_exactly then\n        return data_label == filter_label\n    else\n        return string.find(data_label, filter_label, 1, true) ~= nil\n    end\nend\nreturn {\n    -- Configuration metadata\n    id = \"filter-by-label\",\n    version = \"1.0.1\",\n    inputCount = 1,\n    categories = {\"_filter\"},\n    title = {\n        [\"en\"] = \"Filter by Label\",\n        [\"de\"] = \"Filtern nach Etikett\",\n        [\"es\"] = \"Filtrar por Etiqueta\",\n        [\"fr\"] = \"Filtrer par Étiquette\"\n    },\n    description = {\n        [\"en\"] = [[\nFilters data points by their label field. Only data points matching the filter criteria will pass through.\nConfiguration:\n- **Filter Label**: The text to search for in labels\n- **Case Sensitive**: Match case exactly (default: false)\n- **Match Exactly**: Require exact match instead of substring (default: false)\n- **Invert**: Keep data points that DON'T match instead (default: false)\n]],\n        [\"de\"] = [[\nFiltert Datenpunkte nach ihrem Label-Feld. Nur Datenpunkte, die den Filterkriterien entsprechen, werden durchgelassen.\nKonfiguration:\n- **Filter-Label**: Der Text, nach dem in Labels gesucht werden soll\n- **Groß-/Kleinschreibung beachten**: Groß-/Kleinschreibung exakt beachten (Standard: false)\n- **Exakt übereinstimmen**: Exakte Übereinstimmung statt Teilstring erforderlich (Standard: false)\n- **Invertieren**: Datenpunkte behalten, die NICHT übereinstimmen (Standard: false)\n]],\n        [\"es\"] = [[\nFiltra puntos de datos por su campo de etiqueta. Solo los puntos de datos que coincidan con los criterios del filtro pasarán.\nConfiguración:\n- **Filtrar Etiqueta**: El texto a buscar en las etiquetas\n- **Sensible a Mayúsculas**: Coincidir exactamente con mayúsculas y minúsculas (predeterminado: false)\n- **Coincidir Exactamente**: Requerir coincidencia exacta en lugar de subcadena (predeterminado: false)\n- **Invertir**: Mantener puntos de datos que NO coincidan (predeterminado: false)\n]],\n        [\"fr\"] = [[\nFiltre les points de données par leur champ d'étiquette. Seuls les points de données correspondant aux critères du filtre passeront.\nConfiguration:\n- **Filtrer l'Étiquette**: Le texte à rechercher dans les étiquettes\n- **Sensible à la Casse**: Correspondance exacte de la casse (par défaut: false)\n- **Correspondance Exacte**: Nécessite une correspondance exacte au lieu d'une sous-chaîne (par défaut: false)\n- **Inverser**: Conserver les points de données qui NE correspondent PAS (par défaut: false)\n]]\n    },\n    config = {\n        text {\n            id = \"filter_label\",\n            name = {\n                [\"en\"] = \"Filter Label\",\n                [\"de\"] = \"Filter-Label\",\n                [\"es\"] = \"Filtrar Etiqueta\",\n                [\"fr\"] = \"Filtrer l'Étiquette\"\n            }\n        },\n        checkbox {\n            id = \"case_sensitive\",\n            name = \"_case_sensitive\",\n        },\n        checkbox {\n            id = \"match_exactly\",\n            name = \"_match_exactly\",\n        },\n        checkbox {\n            id = \"invert\",\n            name = {\n                [\"en\"] = \"Invert\",\n                [\"de\"] = \"Invertieren\",\n                [\"es\"] = \"Invertir\",\n                [\"fr\"] = \"Inverser\"\n            }\n        }\n    },\n    -- Generator function\n    generator = function(source, config)\n        local filter_label = config and config.filter_label\n        local case_sensitive = config and config.case_sensitive or false\n        local match_exactly = config and config.match_exactly or false\n        local invert = config and config.invert or false\n        return function()\n            local data_point = source.dp()\n            local should_match = not invert\n            while data_point and (match(data_point, filter_label, case_sensitive, match_exactly) ~= should_match) do\n                data_point = source.dp()\n            end\n            return data_point\n        end\n    end\n}\n","inputConnectorCount":1,"configuration":[{"configType":"Text","id":"filter_label","value":"kg"},{"configType":"Checkbox","id":"case_sensitive","value":false},{"configType":"Checkbox","id":"match_exactly","value":false},{"configType":"Checkbox","id":"invert","value":false}],"translations":{"_filter":{"type":"Translations","translations":{"en":"Filter","es":"Filtro","de":"Filter","fr":"Filtre"}},"_case_sensitive":{"type":"Translations","translations":{"en":"Case Sensitive","es":"Distinguir Mayúsculas","de":"Groß-/Kleinschreibung beachten","fr":"Sensible à la Casse"}},"_match_exactly":{"type":"Translations","translations":{"en":"Match Exactly","es":"Coincidir Exactamente","de":"Exakt übereinstimmen","fr":"Correspondance Exacte"}}},"catalogFunctionId":"filter-by-label","catalogVersion":"1.0.1","dependencies":[{"connectorIndex":0,"nodeId":2}]},{"type":"LuaScriptNode","x":-236.48523,"y":8.267029,"id":6,"script":"-- Lua Function to override the label of all data points with a configurable string\n-- This function sets all incoming data point labels to a specified value\nlocal text = require(\"tng.config\").text\nreturn {\n    -- Configuration metadata\n    id = \"override-label\",\n    version = \"1.0.0\",\n    inputCount = 1,\n    categories = {\"_transform\"},\n    title = {\n        [\"en\"] = \"Override Label\",\n        [\"de\"] = \"Label überschreiben\",\n        [\"es\"] = \"Sobrescribir Etiqueta\",\n        [\"fr\"] = \"Remplacer l'Étiquette\",\n    },\n    description = {\n        [\"en\"] = [[\nSets all incoming data point labels to a specified value\n]],\n        [\"de\"] = [[\nSetzt alle eingehenden Datenpunkt-Labels auf einen bestimmten Wert\n]],\n        [\"es\"] = [[\nEstablece todas las etiquetas de puntos de datos entrantes en un valor especificado\n]],\n        [\"fr\"] = [[\nDéfinit toutes les étiquettes de points de données entrantes sur une valeur spécifiée\n]],\n    },\n    config = {\n        text {\n            id = \"new_label\",\n            name = {\n                [\"en\"] = \"New Label\",\n                [\"de\"] = \"Neues Label\",\n                [\"es\"] = \"Nueva Etiqueta\",\n                [\"fr\"] = \"Nouvelle Étiquette\",\n            },\n        },\n    },\n    -- Generator function\n    generator = function(source, config)\n        local new_label = config and config.new_label\n        return function()\n            local data_point = source.dp()\n            if not data_point then\n                return nil\n            end\n            if not new_label then\n                return data_point\n            end\n            data_point.label = new_label\n            return data_point\n        end\n    end,\n}\n","inputConnectorCount":1,"configuration":[{"configType":"Text","id":"new_label","value":"Kg"}],"translations":{"_transform":{"type":"Translations","translations":{"en":"Transform","es":"Transformar","de":"Transformieren","fr":"Transformer"}}},"catalogFunctionId":"override-label","catalogVersion":"1.0.0","dependencies":[{"connectorIndex":0,"nodeId":3},{"connectorIndex":0,"nodeId":5}]}],"outputNode":{"x":928.75366,"y":16.543365,"id":1,"dependencies":[{"connectorIndex":0,"nodeId":6}]},"isDuration":false}"""

/**
 * Creates the Functions Tutorial group with trackers and functions to demonstrate the functions feature.
 * This group contains:
 * - 4 Trackers: Weight, Running, Weight (lbs), and Cycling
 * - 4 Functions: Exercice, Exercice This Week, Weight (Kg), and All Weight (Kg)
 */
suspend fun createFunctionsTutorialGroup(dataInteractor: DataInteractor) {
    val groupId = dataInteractor.insertGroup(createGroup("FunctionsTutorial"))

    // Create trackers in display order
    val weightTrackerId = dataInteractor.insertTracker(
        createTracker(
            name = "Weight",
            groupId = groupId,
            displayIndex = 4,
            dataType = DataType.CONTINUOUS,
            hasDefaultValue = false,
            suggestionType = TrackerSuggestionType.LABEL_ONLY,
            suggestionOrder = TrackerSuggestionOrder.LABEL_ASCENDING
        )
    )
    val weightFeatureId = dataInteractor.getTrackerById(weightTrackerId)!!.featureId

    val runningTrackerId = dataInteractor.insertTracker(
        createTracker(
            name = "Running ",
            groupId = groupId,
            displayIndex = 5,
            dataType = DataType.DURATION,
            hasDefaultValue = false,
            suggestionType = TrackerSuggestionType.LABEL_ONLY,
            suggestionOrder = TrackerSuggestionOrder.LABEL_ASCENDING
        )
    )
    val runningFeatureId = dataInteractor.getTrackerById(runningTrackerId)!!.featureId

    val weightLbsTrackerId = dataInteractor.insertTracker(
        createTracker(
            name = "Weight (lbs)",
            groupId = groupId,
            displayIndex = 6,
            dataType = DataType.CONTINUOUS,
            hasDefaultValue = false,
            suggestionType = TrackerSuggestionType.LABEL_ONLY,
            suggestionOrder = TrackerSuggestionOrder.LABEL_ASCENDING
        )
    )
    val weightLbsFeatureId = dataInteractor.getTrackerById(weightLbsTrackerId)!!.featureId

    val cyclingTrackerId = dataInteractor.insertTracker(
        createTracker(
            name = "Cycling ",
            groupId = groupId,
            displayIndex = 7,
            dataType = DataType.DURATION,
            hasDefaultValue = false,
            suggestionType = TrackerSuggestionType.LABEL_ONLY,
            suggestionOrder = TrackerSuggestionOrder.LABEL_ASCENDING
        )
    )
    val cyclingFeatureId = dataInteractor.getTrackerById(cyclingTrackerId)!!.featureId

    // Create functions
    val json = Json { ignoreUnknownKeys = true }

    // Function: Exercice - combines Running and Cycling
    dataInteractor.insertFunction(
        Function(
            name = "Exercice ",
            groupId = groupId,
            displayIndex = 0,
            description = "",
            functionGraph = json.decodeFromString(
                exercice_function_graph(runningFeatureId, cyclingFeatureId)
            ),
            inputFeatureIds = listOf(runningFeatureId, cyclingFeatureId)
        )
    )

    // Function: Exercice This Week - filters Exercice data to current week
    dataInteractor.insertFunction(
        Function(
            name = "Exercice This Week",
            groupId = groupId,
            displayIndex = 1,
            description = "",
            functionGraph = json.decodeFromString(
                exercice_this_week_function_graph(runningFeatureId, cyclingFeatureId)
            ),
            inputFeatureIds = listOf(runningFeatureId, cyclingFeatureId)
        )
    )

    // Function: Weight (Kg) - converts Weight (lbs) to kg
    dataInteractor.insertFunction(
        Function(
            name = "Weight (Kg)",
            groupId = groupId,
            displayIndex = 2,
            description = "",
            functionGraph = json.decodeFromString(
                weight_kg_function_graph(weightLbsFeatureId)
            ),
            inputFeatureIds = listOf(weightLbsFeatureId)
        )
    )

    // Function: All Weight (Kg) - merges weight from both kg and lbs labels
    dataInteractor.insertFunction(
        Function(
            name = "All Weight (Kg)",
            groupId = groupId,
            displayIndex = 3,
            description = "",
            functionGraph = json.decodeFromString(
                all_weight_kg_function_graph(weightFeatureId)
            ),
            inputFeatureIds = listOf(weightFeatureId)
        )
    )
}
