/*
 *  This file is part of Track & Graph
 *
 *  Track & Graph is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Track & Graph is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Track & Graph.  If not, see <https://www.gnu.org/licenses/>.
 */
package com.samco.trackandgraph.functions.node_editor

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Brush
import androidx.compose.ui.platform.LocalClipboardManager
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import android.net.Uri
import androidx.compose.foundation.border
import androidx.compose.foundation.horizontalScroll
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.rememberScrollState
import androidx.compose.material3.TextField
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.focus.FocusManager
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.samco.trackandgraph.R
import com.samco.trackandgraph.data.lua.dto.LuaFunctionMetadata
import com.samco.trackandgraph.data.localisation.TranslatedString
import com.samco.trackandgraph.functions.node_editor.viewmodel.LuaScriptConfigurationInput
import com.samco.trackandgraph.functions.node_editor.viewmodel.Node
import com.samco.trackandgraph.ui.compose.theming.TnGComposeTheme
import com.samco.trackandgraph.ui.compose.theming.tngTypography
import com.samco.trackandgraph.ui.compose.ui.HalfDialogInputSpacing
import com.samco.trackandgraph.ui.compose.ui.buttonSize
import com.samco.trackandgraph.ui.compose.ui.cardPadding
import com.samco.trackandgraph.ui.compose.ui.dialogInputSpacing
import com.samco.trackandgraph.ui.compose.ui.IconTextButton
import com.samco.trackandgraph.ui.compose.ui.luaCodeVisualTransformation
import com.samco.trackandgraph.ui.compose.ui.slimOutlinedTextField
import com.samco.trackandgraph.ui.compose.ui.InputSpacingXLarge
import com.samco.trackandgraph.ui.compose.ui.LuaScriptEditDialog
import com.samco.trackandgraph.ui.compose.ui.inputSpacingLarge
import com.samco.trackandgraph.ui.compose.ui.resolve
import com.samco.trackandgraph.ui.compose.ui.InputSpacingLarge
import com.samco.trackandgraph.ui.compose.ui.smallIconSize
import io.github.z4kn4fein.semver.toVersion

@Composable
internal fun LuaScriptNode(
    node: Node.LuaScript,
    onDeleteNode: () -> Unit = {},
    onUpdateScript: (String) -> Unit = {},
    onUpdateScriptFromFile: (Uri?) -> Unit = {},
) {
    var showDialog by rememberSaveable { mutableStateOf(false) }
    var showInfoDialog by rememberSaveable { mutableStateOf(false) }
    var tempScript by remember { mutableStateOf(TextFieldValue(node.script)) }

    // Update temp script when node script changes
    LaunchedEffect(node.script) {
        tempScript = TextFieldValue(node.script)
    }

    Column(
        Modifier
            .widthIn(max = nodeCardContentWidth)
            .padding(horizontal = connectorSize / 2, vertical = cardPadding),
        verticalArrangement = Arrangement.spacedBy(dialogInputSpacing)
    ) {
        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically,
        ) {
            Column(
                modifier = Modifier.weight(1f),
            ) {
                Text(
                    text = getNodeTitle(node),
                    style = MaterialTheme.typography.titleMedium,
                )

                if (node.metadata?.version != null) {
                    Text(
                        text = node.metadata.version.toString(),
                        style = MaterialTheme.typography.labelMedium,
                        color = MaterialTheme.colorScheme.onSurface.copy(alpha = 0.6f)
                    )
                }
            }

            InputSpacingLarge()

            Row {
                // Show info button if metadata has title and description
                if (node.metadata != null &&
                    node.metadata.title != null &&
                    node.metadata.description != null
                ) {
                    IconButton(
                        modifier = Modifier.size(buttonSize),
                        onClick = { showInfoDialog = true }
                    ) {
                        Icon(
                            painter = painterResource(id = R.drawable.about_icon),
                            contentDescription = stringResource(R.string.info),
                            modifier = Modifier.size(smallIconSize),
                            tint = MaterialTheme.colorScheme.onSurfaceVariant
                        )
                    }
                }

                IconButton(
                    modifier = Modifier.size(buttonSize),
                    onClick = onDeleteNode
                ) {
                    Icon(
                        painter = painterResource(id = R.drawable.delete_icon),
                        contentDescription = stringResource(R.string.delete)
                    )
                }
            }
        }

        if (node.metadata?.version != null) {
            HalfDialogInputSpacing()
        }

        if (node.showEditTools) {
            EditorTools(
                node = node,
                onUpdateScript = onUpdateScript,
                onUpdateScriptFromFile = onUpdateScriptFromFile,
                onShowDialog = { showDialog = true },
                onUpdateTempScript = { textFieldValue ->
                    tempScript = tempScript.copy(
                        selection = textFieldValue.selection
                    )
                }
            )
        }

        val focusManager = LocalFocusManager.current

        // Configuration inputs
        if (node.configuration.isNotEmpty()) {
            ConfigurationInputs(
                focusManager = focusManager,
                configuration = node.configuration
            )
        }

        // Dialog
        if (showDialog) {
            LuaScriptEditDialog(
                script = tempScript,
                onDismiss = {
                    showDialog = false
                    // Update the view model with the final script when dialog is closed
                    onUpdateScript(tempScript.text)
                },
                onValueChanged = { newValue ->
                    tempScript = newValue
                }
            )
        }

        // Info dialog
        if (showInfoDialog && node.metadata != null) {
            NodeDescriptionDialog(
                infoDisplay = InfoDisplay.Function(node.metadata),
                onDismiss = { showInfoDialog = false }
            )
        }
    }
}

@Composable
private fun getNodeTitle(node: Node.LuaScript) =
    node.title.resolve() ?: stringResource(R.string.lua_script)

@Composable
private fun EditorTools(
    node: Node.LuaScript,
    onUpdateScript: (String) -> Unit,
    onUpdateScriptFromFile: (Uri?) -> Unit,
    onShowDialog: () -> Unit,
    onUpdateTempScript: (TextFieldValue) -> Unit
) {
    Column(
        verticalArrangement = Arrangement.spacedBy(dialogInputSpacing)
    ) {
        Buttons(
            onUpdateScriptFromClipboard = onUpdateScript,
            onReadFile = onUpdateScriptFromFile
        )

        ScriptPreviewTextField(
            scriptPreview = node.script,
            onScriptPreviewClicked = { textFieldValue ->
                onShowDialog()
                onUpdateTempScript(textFieldValue)
            }
        )
    }
}

@Composable
private fun ScriptPreviewTextField(
    scriptPreview: String,
    onScriptPreviewClicked: (TextFieldValue) -> Unit,
) = Box {
    val localFocusManager = LocalFocusManager.current
    val surfaceColor = MaterialTheme.colorScheme.surface

    val maxVisibleLines = 8

    val showOverlay = remember(scriptPreview) {
        scriptPreview.lines().size > maxVisibleLines
    }

    TextField(
        modifier = Modifier
            .fillMaxWidth()
            .slimOutlinedTextField()
            .border(1.dp, MaterialTheme.colorScheme.outline, shape = MaterialTheme.shapes.small)
            .horizontalScroll(rememberScrollState())
            .drawWithContent {
                drawContent()
                drawRect(color = surfaceColor.copy(alpha = 0.3f))

                // Draw gradient overlay at bottom if content overflows
                if (showOverlay) {
                    val gradientHeight = size.height * 0.3f // 30% of field height
                    val gradient = Brush.verticalGradient(
                        colors = listOf(
                            surfaceColor.copy(alpha = 0.0f),
                            surfaceColor.copy(alpha = 1.0f)
                        ),
                        startY = size.height - gradientHeight,
                        endY = size.height
                    )
                    drawRect(
                        brush = gradient,
                        topLeft = Offset(0f, size.height - gradientHeight),
                        size = androidx.compose.ui.geometry.Size(size.width, gradientHeight)
                    )
                }
            },
        maxLines = 8,
        colors = OutlinedTextFieldDefaults.colors(
            focusedContainerColor = MaterialTheme.colorScheme.surface,
            unfocusedContainerColor = MaterialTheme.colorScheme.surface
        ),
        value = TextFieldValue(scriptPreview),
        onValueChange = { textFieldValue ->
            onScriptPreviewClicked(textFieldValue)
            localFocusManager.clearFocus()
        },
        placeholder = {
            Text(
                text = stringResource(R.string.lua_script_input_hint),
                style = MaterialTheme.typography.bodyLarge,
            )
        },
        visualTransformation = luaCodeVisualTransformation(),
        textStyle = MaterialTheme.tngTypography.code,
        singleLine = false,
        keyboardOptions = KeyboardOptions(
            capitalization = KeyboardCapitalization.None,
            autoCorrectEnabled = false,
        )
    )
}

@Composable
private fun Buttons(
    onReadFile: (Uri?) -> Unit,
    onUpdateScriptFromClipboard: (String) -> Unit,
) = Row(
    modifier = Modifier.fillMaxWidth(),
    horizontalArrangement = Arrangement.Center,
) {

    val clipboardManager = LocalClipboardManager.current
    IconTextButton(
        onClick = {
            val text = clipboardManager.getText()?.text
                ?: return@IconTextButton
            onUpdateScriptFromClipboard(text)
        },
        icon = R.drawable.content_paste,
        text = stringResource(R.string.paste)
    )

    InputSpacingXLarge()

    val launcher = rememberLauncherForActivityResult(
        contract = ActivityResultContracts.GetContent()
    ) { onReadFile(it) }

    IconTextButton(
        onClick = { launcher.launch("*/*") },
        icon = R.drawable.folder_open,
        text = stringResource(R.string.file)
    )
}

@Composable
private fun ConfigurationInputs(
    focusManager: FocusManager,
    configuration: Map<String, LuaScriptConfigurationInput>
) {
    Column(
        modifier = Modifier.fillMaxWidth(),
        verticalArrangement = Arrangement.spacedBy(inputSpacingLarge)
    ) {
        configuration.forEach { (_, input) ->
            ConfigurationInputField(
                modifier = Modifier.fillMaxWidth(),
                focusManager = focusManager,
                input = input,
            )
        }
    }
}


@Preview(showBackground = true)
@Composable
private fun LuaScriptNodePreview() {
    TnGComposeTheme {
        Box(
            modifier = Modifier
                .background(MaterialTheme.colorScheme.background)
                .padding(16.dp)
        ) {
            val sampleNode = Node.LuaScript(
                id = 1,
                inputConnectorCount = 2,
                script = """
                    function main(input1, input2)
                        return input1 + input2 + input3 + input4 + input5 + input6 + input7 + input8 + input9 + input10
                    end
                """.trimIndent(),
                configuration = mapOf(
                    "threshold" to LuaScriptConfigurationInput.Number(
                        name = TranslatedString.Simple("Threshold"),
                        value = remember { mutableStateOf(TextFieldValue("10.5")) }
                    ),
                    "label" to LuaScriptConfigurationInput.Text(
                        name = TranslatedString.Simple("Label"),
                        value = remember { mutableStateOf(TextFieldValue("Sample Label")) }
                    )
                ),
                showEditTools = true,
                title = null
            )

            LuaScriptNode(
                node = sampleNode,
                onDeleteNode = { }
            )
        }
    }
}

@Preview(showBackground = true)
@Composable
private fun LuaScriptNodeEmptyPreview() {
    TnGComposeTheme {
        Box(
            modifier = Modifier
                .background(MaterialTheme.colorScheme.background)
                .padding(16.dp)
        ) {
            val sampleNode = Node.LuaScript(
                id = 2,
                inputConnectorCount = 1,
                script = "",
                configuration = emptyMap(),
                showEditTools = true,
                title = null
            )

            LuaScriptNode(
                node = sampleNode,
                onDeleteNode = { }
            )
        }
    }
}

@Preview(showBackground = true)
@Composable
private fun LuaScriptNodeWithVersionPreview() {
    TnGComposeTheme {
        Box(
            modifier = Modifier
                .background(MaterialTheme.colorScheme.background)
                .padding(16.dp)
        ) {
            val metadata = LuaFunctionMetadata(
                script = "function main(input) return input end",
                id = "filter_by_label",
                description = TranslatedString.Simple(
                    "Filters data points based on their label. " +
                            "Only data points with the specified label will be included in the output."
                ),
                version = "1.0.0".toVersion(),
                title = TranslatedString.Translations(
                    mapOf(
                        "en" to "Filter by Label",
                        "de" to "Filtern nach Etikett",
                        "es" to "Filtrar por Etiqueta",
                        "fr" to "Filtrer par Étiquette"
                    )
                ),
                inputCount = 1,
                config = listOf(),
                categories = mapOf("filter" to TranslatedString.Simple("Filter"))
            )

            val sampleNode = Node.LuaScript(
                id = 3,
                inputConnectorCount = 1,
                script = metadata.script,
                configuration = mapOf(
                    "filter_label" to LuaScriptConfigurationInput.Text(
                        name = TranslatedString.Simple("Filter Label"),
                        value = remember { mutableStateOf(TextFieldValue("work")) }
                    )
                ),
                showEditTools = false, // Version provided, so hide edit tools
                title = metadata.title,
                metadata = metadata
            )

            LuaScriptNode(
                node = sampleNode,
                onDeleteNode = { }
            )
        }
    }
}
