package com.example.util.simpletimetracker.feature_statistics_detail.viewModel.delegate

import com.example.util.simpletimetracker.core.base.ViewModelDelegate
import com.example.util.simpletimetracker.core.extension.shiftTimeStamp
import com.example.util.simpletimetracker.core.extension.toModel
import com.example.util.simpletimetracker.core.interactor.RecordFilterInteractor
import com.example.util.simpletimetracker.core.mapper.RangeViewDataMapper
import com.example.util.simpletimetracker.core.mapper.TimeMapper
import com.example.util.simpletimetracker.core.viewData.RangeSelectionOptionsListItem
import com.example.util.simpletimetracker.domain.daysOfWeek.interactor.GetProcessedLastDaysCountInteractor
import com.example.util.simpletimetracker.domain.prefs.interactor.PrefsInteractor
import com.example.util.simpletimetracker.domain.record.model.Range
import com.example.util.simpletimetracker.domain.record.model.RecordsFilter
import com.example.util.simpletimetracker.domain.statistics.model.RangeLength
import com.example.util.simpletimetracker.navigation.Router
import com.example.util.simpletimetracker.navigation.params.screen.CustomRangeSelectionParams
import com.example.util.simpletimetracker.navigation.params.screen.DateTimeDialogParams
import com.example.util.simpletimetracker.navigation.params.screen.DateTimeDialogType
import com.example.util.simpletimetracker.navigation.params.screen.DurationDialogParams
import com.example.util.simpletimetracker.navigation.params.screen.StatisticsDetailParams
import kotlinx.coroutines.launch
import javax.inject.Inject

class StatisticsDetailRangeViewModelDelegate @Inject constructor(
    private val router: Router,
    private val rangeViewDataMapper: RangeViewDataMapper,
    private val prefsInteractor: PrefsInteractor,
    private val timeMapper: TimeMapper,
    private val recordFilterInteractor: RecordFilterInteractor,
    private val getProcessedLastDaysCountInteractor: GetProcessedLastDaysCountInteractor,
) : StatisticsDetailViewModelDelegate, ViewModelDelegate() {

    private var parent: StatisticsDetailViewModelDelegate.Parent? = null
    private var rangeLength: RangeLength = RangeLength.All
    private var rangePosition: Int = 0

    override fun attach(parent: StatisticsDetailViewModelDelegate.Parent) {
        this.parent = parent
    }

    fun initialize(extra: StatisticsDetailParams) {
        rangeLength = extra.range.toModel()
        rangePosition = extra.shift
    }

    fun onBackToTodayClick() {
        updatePosition(0)
    }

    fun onRangeSelected(id: RangeSelectionOptionsListItem) {
        when (id) {
            is RangeSelectionOptionsListItem.SelectDate -> {
                onSelectDateClick()
            }
            is RangeSelectionOptionsListItem.Custom -> {
                onSelectCustomRangeClick()
            }
            is RangeSelectionOptionsListItem.Last -> {
                onSelectLastDaysClick()
            }
            is RangeSelectionOptionsListItem.Simple -> {
                onRangeUpdated(id.rangeLengthParams.toModel())
            }
        }
    }

    fun onDateTimeSet(timestamp: Long, tag: String?) = delegateScope.launch {
        val startOfDayShift = prefsInteractor.getStartOfDayShift()
        when (tag) {
            DATE_TAG -> {
                timeMapper.toTimestampShift(
                    toTime = timestamp.shiftTimeStamp(startOfDayShift),
                    range = rangeLength,
                    firstDayOfWeek = prefsInteractor.getFirstDayOfWeek(),
                ).toInt().let(::updatePosition)
            }
        }
    }

    fun onCustomRangeSelected(range: Range) {
        onRangeUpdated(RangeLength.Custom(range))
    }

    fun onCountSet(count: Long, tag: String?) {
        if (tag != LAST_DAYS_COUNT_TAG) return

        val lastDaysCount = getProcessedLastDaysCountInteractor.execute(count)
        onRangeUpdated(RangeLength.Last(lastDaysCount))
    }

    fun onSelectDateClick() = delegateScope.launch {
        val useMilitaryTime = prefsInteractor.getUseMilitaryTimeFormat()
        val firstDayOfWeek = prefsInteractor.getFirstDayOfWeek()
        val startOfDayShift = prefsInteractor.getStartOfDayShift()
        val current = timeMapper.toTimestampShifted(
            rangesFromToday = rangePosition,
            range = rangeLength,
            startOfDayShift = startOfDayShift,
        )

        router.navigate(
            DateTimeDialogParams(
                tag = DATE_TAG,
                type = DateTimeDialogType.DATE,
                timestamp = current,
                useMilitaryTime = useMilitaryTime,
                firstDayOfWeek = firstDayOfWeek,
            ),
        )
    }

    fun getDateFilter(): List<RecordsFilter> {
        return recordFilterInteractor.mapDateFilter(
            rangeLength = rangeLength,
            rangePosition = rangePosition,
        ).let(::listOf)
    }

    fun provideRangeLength(): RangeLength {
        return rangeLength
    }

    fun provideRangePosition(): Int {
        return rangePosition
    }

    private fun onSelectCustomRangeClick() = delegateScope.launch {
        val currentCustomRange = (rangeLength as? RangeLength.Custom)?.range

        CustomRangeSelectionParams(
            rangeStart = currentCustomRange?.timeStarted,
            rangeEnd = currentCustomRange?.timeEnded,
        ).let(router::navigate)
    }

    // TODO add custom range reopen same as last days
    private fun onSelectLastDaysClick() = delegateScope.launch {
        DurationDialogParams(
            tag = LAST_DAYS_COUNT_TAG,
            value = DurationDialogParams.Value.Count(
                getCurrentLastDaysCount().toLong(),
            ),
            hideDisableButton = true,
        ).let(router::navigate)
    }

    fun onSelectRangeClick() = delegateScope.launch {
        val data = rangeViewDataMapper.mapToRangesOptions(
            currentRange = rangeLength,
            addSelection = true,
            lastDaysCount = getCurrentLastDaysCount(),
        )
        router.navigate(data)
    }

    private suspend fun getCurrentLastDaysCount(): Int {
        return (rangeLength as? RangeLength.Last)?.days
            ?: prefsInteractor.getStatisticsDetailLastDays()
    }

    private fun onRangeUpdated(newRange: RangeLength) = delegateScope.launch {
        prefsInteractor.setStatisticsDetailRange(newRange)

        if (newRange != rangeLength) {
            rangeLength = newRange
            parent?.onRangeChanged()
            updatePosition(0)
        }
    }

    fun updatePosition(newPosition: Int) {
        rangePosition = newPosition
        parent?.updateViewData()
    }

    companion object {
        private const val LAST_DAYS_COUNT_TAG = "statistics_detail_last_days_count_tag"
        private const val DATE_TAG = "statistics_detail_date_tag"
    }
}