package com.example.util.simpletimetracker.feature_statistics_detail.mapper

import android.annotation.SuppressLint
import android.graphics.Color
import com.example.util.simpletimetracker.core.extension.removeTrailingZeroes
import com.example.util.simpletimetracker.core.mapper.CategoryViewDataMapper
import com.example.util.simpletimetracker.core.mapper.ColorMapper
import com.example.util.simpletimetracker.core.mapper.IconMapper
import com.example.util.simpletimetracker.core.mapper.RecordTagViewDataMapper
import com.example.util.simpletimetracker.core.mapper.TimeMapper
import com.example.util.simpletimetracker.core.repo.ResourceRepo
import com.example.util.simpletimetracker.domain.base.DurationFormat
import com.example.util.simpletimetracker.domain.base.MULTITASK_ITEM_ID
import com.example.util.simpletimetracker.domain.base.OneShotValue
import com.example.util.simpletimetracker.domain.base.UNTRACKED_ITEM_ID
import com.example.util.simpletimetracker.domain.category.model.Category
import com.example.util.simpletimetracker.domain.daysOfWeek.model.DayOfWeek
import com.example.util.simpletimetracker.domain.extension.orZero
import com.example.util.simpletimetracker.domain.record.model.Range
import com.example.util.simpletimetracker.domain.recordTag.model.RecordTag
import com.example.util.simpletimetracker.domain.recordType.model.RecordType
import com.example.util.simpletimetracker.domain.recordType.model.RecordTypeGoal
import com.example.util.simpletimetracker.domain.statistics.model.RangeLength
import com.example.util.simpletimetracker.feature_base_adapter.ViewHolderType
import com.example.util.simpletimetracker.feature_statistics_detail.R
import com.example.util.simpletimetracker.feature_statistics_detail.adapter.StatisticsDetailBlock
import com.example.util.simpletimetracker.feature_statistics_detail.adapter.StatisticsDetailButtonViewData
import com.example.util.simpletimetracker.feature_statistics_detail.conts.TAG_VALUE_PRECISION
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartBarDataDuration
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartBarDataRange
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartGrouping
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartLength
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartMode
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartSplitSortMode
import com.example.util.simpletimetracker.feature_statistics_detail.model.ChartValueMode
import com.example.util.simpletimetracker.feature_statistics_detail.model.SplitChartGrouping
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailCardInternalViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailChartCompositeViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailChartLengthViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailChartValueModeViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailChartViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailGroupingViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailPreviewViewData
import com.example.util.simpletimetracker.feature_statistics_detail.viewData.StatisticsDetailSplitGroupingViewData
import com.example.util.simpletimetracker.feature_views.barChart.BarChartView
import com.example.util.simpletimetracker.feature_views.viewData.RecordTypeIcon
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import kotlin.math.abs
import kotlin.math.roundToLong

class StatisticsDetailViewDataMapper @Inject constructor(
    private val iconMapper: IconMapper,
    private val colorMapper: ColorMapper,
    private val timeMapper: TimeMapper,
    private val resourceRepo: ResourceRepo,
    private val categoryViewDataMapper: CategoryViewDataMapper,
    private val recordTagViewDataMapper: RecordTagViewDataMapper,
) {

    fun mapToPreview(
        recordType: RecordType,
        isDarkTheme: Boolean,
        isFirst: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        return StatisticsDetailPreviewViewData(
            id = recordType.id,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = recordType.name.takeIf { isFirst }.orEmpty(),
            iconId = recordType.icon
                .let(iconMapper::mapIcon),
            color = recordType.color
                .let { colorMapper.mapToColorInt(it, isDarkTheme) },
        )
    }

    fun mapToCategorizedPreview(
        category: Category,
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        return StatisticsDetailPreviewViewData(
            id = category.id,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = category.name,
            iconId = null,
            color = category.color
                .let { colorMapper.mapToColorInt(it, isDarkTheme) },
        )
    }

    fun mapToUncategorizedPreview(
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        val item = categoryViewDataMapper.mapToUncategorizedItem(
            isDarkTheme = isDarkTheme,
            isFiltered = false,
        )

        return StatisticsDetailPreviewViewData(
            id = item.id,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = item.name,
            iconId = RecordTypeIcon.Image(R.drawable.untagged),
            color = item.color,
        )
    }

    fun mapToTaggedPreview(
        tag: RecordTag,
        types: Map<Long, RecordType>,
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        val icon = recordTagViewDataMapper.mapIcon(tag, types)
        val color = recordTagViewDataMapper.mapColor(tag, types)

        return StatisticsDetailPreviewViewData(
            id = tag.id,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = tag.name,
            iconId = icon?.let(iconMapper::mapIcon),
            color = color.let { colorMapper.mapToColorInt(it, isDarkTheme) },
        )
    }

    fun mapToUntaggedPreview(
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        val item = categoryViewDataMapper.mapToUntaggedItem(
            isDarkTheme = isDarkTheme,
            isFiltered = false,
        )

        return StatisticsDetailPreviewViewData(
            id = item.id,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = item.name,
            iconId = item.icon,
            color = item.color,
        )
    }

    fun mapUntrackedPreview(
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        return StatisticsDetailPreviewViewData(
            id = UNTRACKED_ITEM_ID,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = resourceRepo.getString(R.string.untracked_time_name),
            iconId = RecordTypeIcon.Image(R.drawable.unknown),
            color = colorMapper.toUntrackedColor(isDarkTheme),
        )
    }

    fun mapMultitaskPreview(
        isDarkTheme: Boolean,
        isForComparison: Boolean,
    ): StatisticsDetailPreviewViewData {
        return StatisticsDetailPreviewViewData(
            id = MULTITASK_ITEM_ID,
            type = if (isForComparison) {
                StatisticsDetailPreviewViewData.Type.COMPARISON
            } else {
                StatisticsDetailPreviewViewData.Type.FILTER
            },
            name = resourceRepo.getString(R.string.multitask_time_name),
            iconId = RecordTypeIcon.Image(R.drawable.multitask),
            color = colorMapper.toUntrackedColor(isDarkTheme),
        )
    }

    fun mapToPreviewEmpty(
        isDarkTheme: Boolean,
    ): StatisticsDetailPreviewViewData {
        return StatisticsDetailPreviewViewData(
            id = 0,
            type = StatisticsDetailPreviewViewData.Type.FILTER,
            name = "",
            iconId = RecordTypeIcon.Image(R.drawable.unknown),
            color = colorMapper.toUntrackedColor(isDarkTheme),
        )
    }

    fun mapToChartViewData(
        data: List<ChartBarDataDuration>,
        prevData: List<ChartBarDataDuration>,
        splitByActivity: Boolean,
        canSplitByActivity: Boolean,
        canComparisonSplitByActivity: Boolean,
        splitSortMode: ChartSplitSortMode,
        goalValue: Long,
        compareData: List<ChartBarDataDuration>,
        compareGoalValue: Long,
        showComparison: Boolean,
        rangeLength: RangeLength,
        availableChartGroupings: List<ChartGrouping>,
        appliedChartGrouping: ChartGrouping,
        availableChartLengths: List<ChartLength>,
        appliedChartLength: ChartLength,
        chartMode: ChartMode,
        durationFormat: DurationFormat,
        showSeconds: Boolean,
        isDarkTheme: Boolean,
    ): StatisticsDetailChartCompositeViewData {
        val chartIsSplitByActivity = splitByActivity && canSplitByActivity
        val chartComparisonIsSplitByActivity = splitByActivity && canComparisonSplitByActivity

        val chartData = mapChartData(
            data = data,
            goal = goalValue,
            rangeLength = rangeLength,
            chartMode = chartMode,
            showSelectedBarOnStart = true,
            useSingleColor = !chartIsSplitByActivity,
            drawRoundCaps = !chartIsSplitByActivity,
        )
        val compareChartData = mapChartData(
            data = compareData,
            goal = compareGoalValue,
            rangeLength = rangeLength,
            chartMode = chartMode,
            showSelectedBarOnStart = false,
            useSingleColor = !chartComparisonIsSplitByActivity,
            drawRoundCaps = !chartComparisonIsSplitByActivity,
        )
        val (title, rangeAverages) = getRangeAverages(
            data = data,
            prevData = prevData,
            compareData = compareData,
            showComparison = showComparison,
            rangeLength = rangeLength,
            chartGrouping = appliedChartGrouping,
            chartMode = chartMode,
            durationFormat = durationFormat,
            showSeconds = showSeconds,
            isDarkTheme = isDarkTheme,
        )
        val chartGroupingViewData = mapToChartGroupingViewData(
            availableChartGroupings = availableChartGroupings,
            appliedChartGrouping = appliedChartGrouping,
        )
        val chartLengthViewData = mapToChartLengthViewData(
            availableChartLengths = availableChartLengths,
            appliedChartLength = appliedChartLength,
        )
        val splitByActivityItems = if (canSplitByActivity || canComparisonSplitByActivity) {
            mapSplitByActivityItems(
                splitByActivity = splitByActivity,
                splitSortMode = splitSortMode,
                isDarkTheme = isDarkTheme,
            )
        } else {
            emptyList()
        }
        val additionalChartButtonItems = mutableListOf<ViewHolderType>()
        additionalChartButtonItems += splitByActivityItems

        return StatisticsDetailChartCompositeViewData(
            chartData = chartData,
            compareChartData = compareChartData,
            showComparison = showComparison,
            rangeAveragesTitle = title,
            rangeAverages = rangeAverages,
            appliedChartGrouping = appliedChartGrouping,
            chartGroupingViewData = chartGroupingViewData,
            chartGroupingVisible = chartGroupingViewData.size > 1,
            appliedChartLength = appliedChartLength,
            chartLengthViewData = chartLengthViewData,
            chartLengthVisible = chartLengthViewData.isNotEmpty(),
            additionalChartButtonItems = additionalChartButtonItems,
        )
    }

    fun mapToEmptyChartViewData(
        ranges: List<ChartBarDataRange>,
        availableChartGroupings: List<ChartGrouping>,
        availableChartLengths: List<ChartLength>,
    ): StatisticsDetailChartCompositeViewData {
        val emptyChart = StatisticsDetailChartViewData(
            visible = false,
            data = emptyList(),
            legendSuffix = "",
            addLegendToSelectedBar = false,
            shouldDrawHorizontalLegends = false,
            showSelectedBarOnStart = false,
            selectedBarPosition = null,
            goalValue = 0f,
            drawRoundCaps = true,
            useSingleColor = true,
            animate = OneShotValue(true),
        )

        return StatisticsDetailChartCompositeViewData(
            chartData = emptyChart.copy(
                visible = ranges.size > 1,
            ),
            compareChartData = emptyChart,
            showComparison = false,
            rangeAveragesTitle = " ",
            rangeAverages = if (ranges.size < 2) {
                emptyList()
            } else {
                mapToEmptyRangeAverages()
            },
            appliedChartGrouping = ChartGrouping.DAILY,
            chartGroupingViewData = emptyList(),
            chartGroupingVisible = availableChartGroupings.size > 1,
            appliedChartLength = ChartLength.TEN,
            chartLengthViewData = emptyList(),
            chartLengthVisible = availableChartLengths.isNotEmpty(),
            additionalChartButtonItems = emptyList(),
        )
    }

    fun mapToDailyChartViewData(
        data: Map<Int, Float>,
        firstDayOfWeek: DayOfWeek,
        isVisible: Boolean,
    ): StatisticsDetailChartViewData {
        val days = timeMapper.getWeekOrder(firstDayOfWeek)

        val viewData = days.map { day ->
            val calendarDay = timeMapper.toCalendarDayOfWeek(day)
            BarChartView.ViewData(
                id = 0,
                value = listOf(data[calendarDay].orZero() to Color.TRANSPARENT),
                legend = timeMapper.toShortDayOfWeekName(day),
            )
        }

        return StatisticsDetailChartViewData(
            visible = isVisible,
            data = viewData,
            legendSuffix = SPLIT_CHART_LEGEND,
            addLegendToSelectedBar = false,
            shouldDrawHorizontalLegends = true,
            showSelectedBarOnStart = false,
            selectedBarPosition = null,
            goalValue = 0f,
            useSingleColor = true,
            drawRoundCaps = true,
            animate = OneShotValue(true),
        )
    }

    fun mapToHourlyChartViewData(
        data: Map<Int, Float>,
        isVisible: Boolean,
    ): StatisticsDetailChartViewData {
        val hourLegends = (0 until 24).map {
            it to it.toString()
        }

        val viewData = hourLegends
            .map { (hour, legend) ->
                BarChartView.ViewData(
                    id = 0,
                    value = listOf(data[hour].orZero() to Color.TRANSPARENT),
                    legend = legend,
                )
            }

        return StatisticsDetailChartViewData(
            visible = isVisible,
            data = viewData,
            legendSuffix = SPLIT_CHART_LEGEND,
            addLegendToSelectedBar = false,
            shouldDrawHorizontalLegends = true,
            showSelectedBarOnStart = false,
            selectedBarPosition = null,
            goalValue = 0f,
            useSingleColor = true,
            drawRoundCaps = true,
            animate = OneShotValue(true),
        )
    }

    fun mapToSplitChartGroupingViewData(
        rangeLength: RangeLength,
        splitChartGrouping: SplitChartGrouping,
    ): List<ViewHolderType> {
        val groupings = when (rangeLength) {
            is RangeLength.Day -> emptyList()
            else -> listOf(
                SplitChartGrouping.HOURLY,
                SplitChartGrouping.DAILY,
            )
        }

        return groupings.map {
            StatisticsDetailSplitGroupingViewData(
                splitChartGrouping = it,
                name = mapToSplitGroupingName(it),
                isSelected = it == splitChartGrouping,
            )
        }
    }

    fun mapToDurationsSlipChartViewData(
        data: Map<Range, Float>,
        isVisible: Boolean,
    ): StatisticsDetailChartViewData {
        val viewData = data
            .map { (range, percent) ->
                val started = timeMapper.formatDuration(range.timeStarted / 1000)
                val ended = timeMapper.formatDuration(range.timeEnded / 1000)
                range to BarChartView.ViewData(
                    id = 0,
                    value = listOf(percent to Color.TRANSPARENT),
                    legend = ended,
                    selectedBarLegend = "$started - $ended",
                )
            }.sortedBy { (range, _) ->
                range.timeStarted
            }.map { (_, data) ->
                data
            }

        return StatisticsDetailChartViewData(
            visible = isVisible,
            data = viewData,
            legendSuffix = SPLIT_CHART_LEGEND,
            addLegendToSelectedBar = true,
            shouldDrawHorizontalLegends = true,
            showSelectedBarOnStart = false,
            selectedBarPosition = null,
            goalValue = 0f,
            useSingleColor = true,
            drawRoundCaps = true,
            animate = OneShotValue(true),
        )
    }

    @SuppressLint("DefaultLocale")
    fun processPercentageString(value: Float): String {
        val text = when {
            value >= 10 -> value.toLong()
            (value * 10).roundToLong() % 10L == 0L -> value.toLong()
            else -> String.format("%.1f", value)
        }
        return "$text%"
    }

    fun getRangeAverages(
        data: List<ChartBarDataDuration>,
        prevData: List<ChartBarDataDuration>,
        compareData: List<ChartBarDataDuration>,
        showComparison: Boolean,
        rangeLength: RangeLength,
        chartGrouping: ChartGrouping,
        chartMode: ChartMode,
        durationFormat: DurationFormat,
        showSeconds: Boolean,
        isDarkTheme: Boolean,
    ): Pair<String, List<StatisticsDetailCardInternalViewData>> {
        // No reason to show average of one value.
        if (data.size < 2 && compareData.size < 2) return "" to emptyList()

        fun getAverage(data: List<ChartBarDataDuration>): Float {
            if (data.isEmpty()) return 0f
            val sum = data.sumOf { point -> point.durations.sumOf { it.first } }.toFloat()
            return sum / data.size
        }

        @SuppressLint("DefaultLocale")
        fun formatDecimalValue(value: Float): String {
            val abs = abs(value)
            return when {
                abs >= 1000f -> value.toLong().toString()
                abs >= 100f -> String.format("%.1f", value)
                abs >= 10f -> String.format("%.2f", value)
                else -> String.format("%.3f", value)
            }.toString().removeTrailingZeroes()
        }

        fun formatInterval(interval: Float): String {
            return when (chartMode) {
                is ChartMode.DURATIONS -> timeMapper.formatInterval(
                    interval = interval.toLong(),
                    forceSeconds = showSeconds,
                    durationFormat = durationFormat,
                )
                is ChartMode.COUNTS -> formatDecimalValue(interval)
                is ChartMode.TAG_VALUE -> formatDecimalValue(interval / TAG_VALUE_PRECISION)
            }
        }

        fun filterNonEmptyData(
            data: List<ChartBarDataDuration>,
        ): List<ChartBarDataDuration> {
            return when (chartMode) {
                is ChartMode.DURATIONS,
                is ChartMode.COUNTS,
                -> data.filter { it.totalDuration.orZero() != 0L }
                is ChartMode.TAG_VALUE,
                -> data.filter { it.durations.isNotEmpty() }
            }
        }

        val average = getAverage(data)
        val nonEmptyData = filterNonEmptyData(data)
        val averageByNonEmpty = getAverage(nonEmptyData)

        val comparisonAverage = getAverage(compareData)
        val comparisonNonEmptyData = filterNonEmptyData(compareData)
        val comparisonAverageByNonEmpty = getAverage(comparisonNonEmptyData)

        val prevAverage = getAverage(prevData)
        val prevNonEmptyData = filterNonEmptyData(prevData)
        val prevAverageByNonEmpty = getAverage(prevNonEmptyData)

        val title = resourceRepo.getString(
            R.string.statistics_detail_range_averages_title,
            mapToGroupingName(chartGrouping),
        )

        val rangeAverages = listOf(
            StatisticsDetailCardInternalViewData(
                value = formatInterval(average),
                valueChange = mapValueChange(
                    average = average,
                    prevAverage = prevAverage,
                    rangeLength = rangeLength,
                    isDarkTheme = isDarkTheme,
                ),
                secondValue = formatInterval(comparisonAverage)
                    .let { "($it)" }
                    .takeIf { showComparison }
                    .orEmpty(),
                description = resourceRepo.getString(R.string.statistics_detail_range_averages),
                titleTextSizeSp = 14,
                subtitleTextSizeSp = 12,
            ),
            StatisticsDetailCardInternalViewData(
                value = formatInterval(averageByNonEmpty),
                valueChange = mapValueChange(
                    average = averageByNonEmpty,
                    prevAverage = prevAverageByNonEmpty,
                    rangeLength = rangeLength,
                    isDarkTheme = isDarkTheme,
                ),
                secondValue = formatInterval(comparisonAverageByNonEmpty)
                    .let { "($it)" }
                    .takeIf { showComparison }
                    .orEmpty(),
                description = resourceRepo.getString(R.string.statistics_detail_range_averages_non_empty),
                titleTextSizeSp = 14,
                subtitleTextSizeSp = 12,
            ),
        )

        return title to rangeAverages
    }

    private fun mapToEmptyRangeAverages(): List<StatisticsDetailCardInternalViewData> {
        val emptyValue by lazy { resourceRepo.getString(R.string.statistics_detail_empty) }

        return listOf(
            StatisticsDetailCardInternalViewData(
                value = emptyValue,
                valueChange = StatisticsDetailCardInternalViewData.ValueChange.None,
                secondValue = "",
                description = resourceRepo.getString(R.string.statistics_detail_range_averages),
                titleTextSizeSp = 14,
                subtitleTextSizeSp = 12,
            ),
            StatisticsDetailCardInternalViewData(
                value = emptyValue,
                valueChange = StatisticsDetailCardInternalViewData.ValueChange.None,
                secondValue = "",
                description = resourceRepo.getString(R.string.statistics_detail_range_averages_non_empty),
                titleTextSizeSp = 14,
                subtitleTextSizeSp = 12,
            ),
        )
    }

    private fun mapValueChange(
        average: Float,
        prevAverage: Float,
        rangeLength: RangeLength,
        isDarkTheme: Boolean,
    ): StatisticsDetailCardInternalViewData.ValueChange {
        if (rangeLength == RangeLength.All) {
            return StatisticsDetailCardInternalViewData.ValueChange.None
        }

        val change: Float = when {
            prevAverage.orZero() == 0f && average.orZero() == 0f -> 0f
            prevAverage.orZero() == 0f && average.orZero() > 0f -> 100f
            prevAverage.orZero() == 0f && average.orZero() < 0f -> -100f
            prevAverage != 0f -> (average.orZero() - prevAverage) * 100f / abs(prevAverage)
            else -> 0f
        }

        // Lowest precision is one decimal.
        @SuppressLint("DefaultLocale")
        fun formatChange(value: Float): String {
            val abs = abs(value)
            val text = when {
                abs >= 1_000_000f -> "∞"
                abs >= 1_000f -> "${(abs / 1000).toLong()}K"
                abs >= 10 -> abs.toLong().toString()
                (abs * 10).roundToLong() % 10L == 0L -> abs.toLong().toString()
                else -> String.format("%.1f", abs)
            }
            return if (value >= 0) "+$text%" else "-$text%"
        }

        return StatisticsDetailCardInternalViewData.ValueChange.Present(
            text = formatChange(change),
            color = if (change >= 0f) {
                colorMapper.toPositiveColor(isDarkTheme)
            } else {
                colorMapper.toNegativeColor(isDarkTheme)
            },
        )
    }

    fun mapChartData(
        data: List<ChartBarDataDuration>,
        goal: Long,
        rangeLength: RangeLength,
        chartMode: ChartMode,
        showSelectedBarOnStart: Boolean,
        useSingleColor: Boolean,
        drawRoundCaps: Boolean,
    ): StatisticsDetailChartViewData {
        val (legendSuffix, isMinutes) = when (chartMode) {
            is ChartMode.DURATIONS -> mapLegendSuffix(data)
            is ChartMode.COUNTS -> "" to false
            is ChartMode.TAG_VALUE -> "" to false
        }

        fun formatInterval(interval: Long): Float {
            return when (chartMode) {
                is ChartMode.DURATIONS -> formatInterval(interval, isMinutes)
                is ChartMode.COUNTS -> interval.toFloat()
                is ChartMode.TAG_VALUE -> interval.toFloat() / TAG_VALUE_PRECISION
            }
        }

        return StatisticsDetailChartViewData(
            visible = data.size > 1,
            data = data.map {
                val value = it.durations.map { (duration, color) ->
                    formatInterval(duration) to color
                }
                BarChartView.ViewData(
                    id = 0,
                    value = value,
                    legend = it.legend,
                )
            },
            legendSuffix = legendSuffix,
            addLegendToSelectedBar = true,
            shouldDrawHorizontalLegends = when (rangeLength) {
                is RangeLength.Day -> false
                is RangeLength.Week -> true
                is RangeLength.Month -> false
                is RangeLength.Year -> data.size <= 12
                is RangeLength.All,
                is RangeLength.Custom,
                is RangeLength.Last,
                -> data.size <= 10
            },
            showSelectedBarOnStart = showSelectedBarOnStart,
            selectedBarPosition = null,
            goalValue = formatInterval(goal),
            useSingleColor = useSingleColor,
            drawRoundCaps = drawRoundCaps,
            animate = OneShotValue(true),
        )
    }

    fun mapLegendSuffix(
        data: List<ChartBarDataDuration>,
    ): Pair<String, Boolean> {
        val isMinutes = data
            .maxOfOrNull { barPart -> abs(barPart.totalDuration.orZero()) }
            .orZero()
            .let(TimeUnit.MILLISECONDS::toHours) == 0L

        val legendSuffix = if (isMinutes) {
            R.string.statistics_detail_legend_minute_suffix
        } else {
            R.string.statistics_detail_legend_hour_suffix
        }.let(resourceRepo::getString)

        return legendSuffix to isMinutes
    }

    fun formatInterval(interval: Long, isMinutes: Boolean): Float {
        val hr: Long = TimeUnit.MILLISECONDS.toHours(
            interval,
        )
        val min: Long = TimeUnit.MILLISECONDS.toMinutes(
            interval - TimeUnit.HOURS.toMillis(hr),
        )
        val sec: Long = TimeUnit.MILLISECONDS.toSeconds(
            interval - TimeUnit.HOURS.toMillis(hr) - TimeUnit.MINUTES.toMillis(min),
        )

        return if (isMinutes) {
            hr * 60f + min + sec / 60f
        } else {
            hr + min / 60f
        }
    }

    fun mapToChartMode(
        goal: RecordTypeGoal?,
    ): ChartMode {
        return when (goal?.type) {
            is RecordTypeGoal.Type.Duration -> ChartMode.DURATIONS
            is RecordTypeGoal.Type.Count -> ChartMode.COUNTS
            null -> ChartMode.DURATIONS
        }
    }

    private fun mapSplitByActivityItems(
        splitByActivity: Boolean,
        splitSortMode: ChartSplitSortMode,
        isDarkTheme: Boolean,
    ): List<ViewHolderType> {
        return StatisticsDetailButtonViewData(
            marginTopDp = 0, // Set later depending on previous items in list.
            data = StatisticsDetailButtonViewData.Button(
                block = StatisticsDetailBlock.ChartSplitByActivity,
                text = resourceRepo.getString(R.string.statistics_detail_chart_split),
                color = if (splitByActivity) {
                    resourceRepo.getThemedAttr(R.attr.appActiveColor, isDarkTheme)
                } else {
                    resourceRepo.getThemedAttr(R.attr.appInactiveColor, isDarkTheme)
                },
            ),
            dataSecond = if (splitByActivity) {
                StatisticsDetailButtonViewData.Button(
                    block = StatisticsDetailBlock.ChartSplitByActivitySort,
                    text = when (splitSortMode) {
                        ChartSplitSortMode.ACTIVITY_ORDER ->
                            resourceRepo.getString(R.string.settings_sort_activity)
                        ChartSplitSortMode.DURATION ->
                            resourceRepo.getString(R.string.records_all_sort_duration)
                    },
                    color = resourceRepo.getThemedAttr(R.attr.appInactiveColor, isDarkTheme),
                )
            } else {
                null
            },
        ).let(::listOf)
    }

    fun mapToChartGroupingViewData(
        availableChartGroupings: List<ChartGrouping>,
        appliedChartGrouping: ChartGrouping,
    ): List<ViewHolderType> {
        return availableChartGroupings.map {
            StatisticsDetailGroupingViewData(
                chartGrouping = it,
                name = mapToGroupingName(it),
                isSelected = it == appliedChartGrouping,
                textSizeSp = if (availableChartGroupings.size >= 3) 12 else null,
            )
        }
    }

    fun mapToChartLengthViewData(
        availableChartLengths: List<ChartLength>,
        appliedChartLength: ChartLength,
    ): List<ViewHolderType> {
        return availableChartLengths.map {
            StatisticsDetailChartLengthViewData(
                chartLength = it,
                name = mapToLengthName(it),
                isSelected = it == appliedChartLength,
            )
        }
    }

    fun mapToChartValueModeViewData(
        availableChartValueModes: List<ChartValueMode>,
        chartValueMode: ChartValueMode,
    ): List<ViewHolderType> {
        return availableChartValueModes.map {
            StatisticsDetailChartValueModeViewData(
                chartValueMode = it,
                name = mapToChartValueModeGroupingName(it),
                isSelected = it == chartValueMode,
            )
        }
    }

    private fun mapToGroupingName(chartGrouping: ChartGrouping): String {
        return when (chartGrouping) {
            ChartGrouping.DAILY -> R.string.statistics_detail_chart_daily
            ChartGrouping.WEEKLY -> R.string.statistics_detail_chart_weekly
            ChartGrouping.MONTHLY -> R.string.statistics_detail_chart_monthly
            ChartGrouping.YEARLY -> R.string.statistics_detail_chart_yearly
        }.let(resourceRepo::getString)
    }

    private fun mapToSplitGroupingName(splitChartGrouping: SplitChartGrouping): String {
        return when (splitChartGrouping) {
            SplitChartGrouping.HOURLY -> R.string.statistics_detail_chart_hourly
            SplitChartGrouping.DAILY -> R.string.statistics_detail_chart_daily
        }.let(resourceRepo::getString)
    }

    private fun mapToLengthName(chartLength: ChartLength): String {
        return when (chartLength) {
            ChartLength.TEN -> R.string.statistics_detail_length_ten
            ChartLength.FIFTY -> R.string.statistics_detail_length_fifty
            ChartLength.HUNDRED -> R.string.statistics_detail_length_hundred
        }.let(resourceRepo::getString)
    }

    private fun mapToChartValueModeGroupingName(
        chartValueMode: ChartValueMode,
    ): String {
        return when (chartValueMode) {
            ChartValueMode.TOTAL -> R.string.statistics_detail_total_duration
            ChartValueMode.AVERAGE -> R.string.statistics_detail_average_record
        }.let(resourceRepo::getString)
    }

    companion object {
        private const val SPLIT_CHART_LEGEND = "%"
    }
}