package com.raiiware.forcestophelper.applicationlist;

import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.provider.Settings;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.BaseInputConnection;
import android.widget.EditText;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.SearchView;
import androidx.core.view.MenuHost;
import androidx.core.view.MenuProvider;
import androidx.fragment.app.Fragment;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.ViewModelProvider;
import androidx.recyclerview.widget.DividerItemDecoration;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.raiiware.forcestophelper.R;

import java.util.Objects;


public class ApplicationListFragment extends Fragment {

    private ApplicationListViewModel   viewModel;
    private ApplicationListAdapter     listAdapter;
    private View                       loadingIndicator;
    private View                       stickyHeader;
    private TextView                   stickyHeaderTextView;
    private RecyclerView.LayoutManager layoutManager;
    private RecyclerView               recyclerView;
    private MenuItem                   sortingOrderApplicationLabelMenuItem;
    private MenuItem                   sortingOrderFirstInstallTimeMenuItem;
    private MenuItem                   sortingOrderLastUpdateTimeMenuItem;

    public static ApplicationListFragment newInstance() {
        return new ApplicationListFragment();
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        this.viewModel = new ViewModelProvider(this).get(ApplicationListViewModel.class);
    }

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container,
            @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.application_list_fragment, container, false);
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        setUpMenu(requireActivity());
        setUpListAdapter();
        setUpRecyclerView(view);
        setUpLoadingIndicator(view);
        setUpStickyHeader(view);

        observeViewModel();

        if (savedInstanceState == null) {
            reloadApplicationList();
        }
    }

    private void setUpMenu(@NonNull MenuHost menuHost) {
        menuHost.addMenuProvider(new MenuProvider() {
            @Override
            public void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
                ApplicationListFragment.this.onCreateMenu(menu, menuInflater);
            }
            @Override
            public void onPrepareMenu(@NonNull Menu menu) {
                ApplicationListFragment.this.onPrepareMenu(menu);
            }
            @Override
            public boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
                return ApplicationListFragment.this.onMenuItemSelected(menuItem);
            }
        }, getViewLifecycleOwner(), Lifecycle.State.RESUMED);
    }

    private void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
        menuInflater.inflate(R.menu.application_list, menu);

        final var searchMenuItem = menu.findItem(R.id.search);
        final var searchView     = (SearchView) searchMenuItem.getActionView();
        Objects.requireNonNull(searchView);

        new Handler(Looper.getMainLooper())
                .post(() -> setUpSearchView(searchView, searchMenuItem));

        this.sortingOrderApplicationLabelMenuItem = menu.findItem(R.id.sorting_order_application_label);
        this.sortingOrderFirstInstallTimeMenuItem = menu.findItem(R.id.sorting_order_first_install_time);
        this.sortingOrderLastUpdateTimeMenuItem   = menu.findItem(R.id.sorting_order_last_update_time);
    }

    private void setUpSearchView(@NonNull SearchView searchView, @NonNull MenuItem searchMenuItem) {
        final var query = viewModel.getQuery();
        if (! TextUtils.isEmpty(query)) {
            searchMenuItem.expandActionView();
            searchView.setQuery(query, false);
        }

        final var searchEditText = (EditText) searchView.findViewById(androidx.appcompat.R.id.search_src_text);

        searchEditText.addTextChangedListener(new TextWatcher() {
            @Override public void beforeTextChanged(CharSequence s, int start, int count, int after) {}
            @Override public void onTextChanged(CharSequence s, int start, int before, int count) {}

            @Override
            public void afterTextChanged(Editable s) {
                if ((s.length() == 0) && isActivityMaybePausing()) return;
                if (BaseInputConnection.getComposingSpanStart(s) >= 0) return;
                requestSearch(s.toString(), 300);
            }
        });

        searchView.setOnQueryTextListener(new SearchView.OnQueryTextListener() {
            @Override
            public boolean onQueryTextSubmit(String query) {
                requestSearch(query, 0);
                searchView.clearFocus();
                return true;
            }

            @Override
            public boolean onQueryTextChange(String newText) {
                return false;
            }
        });
    }

    private boolean isActivityMaybePausing() {
        final var activity = getActivity();
        if (activity == null) return true;

        return !(activity.getLifecycle().getCurrentState().isAtLeast(Lifecycle.State.RESUMED));
    }

    private void onPrepareMenu(@NonNull Menu menu) {
        final var showStoppedAppsMenuItem = menu.findItem(R.id.show_stopped_apps);
        refreshShowStoppedAppsMenuItem(showStoppedAppsMenuItem);

        refreshSortingOrderMenuItems();
    }

    private void refreshShowStoppedAppsMenuItem(@NonNull MenuItem showStoppedAppsMenuItem) {
        showStoppedAppsMenuItem.setChecked(viewModel.shouldShowStoppedApps());
    }

    private void refreshSortingOrderMenuItems() {
        switch (viewModel.getSortingOrder()) {
            case APPLICATION_LABEL:
                sortingOrderApplicationLabelMenuItem.setChecked(true);
                break;

            case FIRST_INSTALL_TIME_DESCENDING:
                sortingOrderFirstInstallTimeMenuItem.setChecked(true);
                break;

            case LAST_UPDATE_TIME_DESCENDING:
                sortingOrderLastUpdateTimeMenuItem.setChecked(true);
                break;
        }
    }

    private boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
        final var itemId = menuItem.getItemId();

        if (itemId == R.id.show_stopped_apps) {
            viewModel.toggleShowStoppedApps();
            refreshShowStoppedAppsMenuItem(menuItem);
            return true;
        }

        if (itemId == R.id.sorting_order_application_label) {
            viewModel.setSortingOrder(ApplicationSortingOrder.APPLICATION_LABEL);
            refreshSortingOrderMenuItems();
            return true;
        }

        if (itemId == R.id.sorting_order_first_install_time) {
            viewModel.setSortingOrder(ApplicationSortingOrder.FIRST_INSTALL_TIME_DESCENDING);
            refreshSortingOrderMenuItems();
            return true;
        }

        if (itemId == R.id.sorting_order_last_update_time) {
            viewModel.setSortingOrder(ApplicationSortingOrder.LAST_UPDATE_TIME_DESCENDING);
            refreshSortingOrderMenuItems();
            return true;
        }

        return false;
    }

    private void setUpListAdapter() {
        final ApplicationItemListener itemListener = new ApplicationItemListener() {
            @Override
            public void onApplicationItemClicked(Application application, boolean pinned) {
                onItemClicked(application, pinned);
            }

            @Override
            public boolean onApplicationItemLongClicked(Application application, boolean pinned, View itemView) {
                return onItemLongClicked(application, pinned, itemView);
            }
        };

        this.listAdapter = new ApplicationListAdapter(itemListener);
    }

    private void setUpRecyclerView(View view) {
        this.layoutManager = new LinearLayoutManager(requireContext());

        this.recyclerView = view.findViewById(R.id.application_list);
        recyclerView.setLayoutManager(layoutManager);
        recyclerView.addItemDecoration(new DividerItemDecoration(requireContext(), LinearLayoutManager.VERTICAL));

        recyclerView.setAdapter(listAdapter);

        recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                refreshStickyHeader();
            }
        });
    }

    private void setUpLoadingIndicator(View view) {
        this.loadingIndicator = view.findViewById(R.id.loading_indicator);
    }

    private void setUpStickyHeader(View view) {
        this.stickyHeader         = view.findViewById(R.id.sticky_header);
        this.stickyHeaderTextView = stickyHeader.findViewById(R.id.sticky_header_text);

        stickyHeader.setVisibility(View.INVISIBLE);
    }

    private void refreshStickyHeader() {
        //noinspection LoopStatementThatDoesntLoop
        do {
            if (viewModel.isLoading()) break;

            if (!(listAdapter.containsHeaders())) break;

            if (layoutManager.getChildCount() <= 0) break;

            final var topVisibleChild = layoutManager.getChildAt(0);
            if (topVisibleChild == null || topVisibleChild.getTop() >= 1) break;

            final var topVisiblePosition = recyclerView.getChildAdapterPosition(topVisibleChild);
            if (topVisiblePosition < 0) break;

            if (listAdapter.isItemPinned(topVisiblePosition)) {
                stickyHeaderTextView.setText(R.string.application_list_header_text_pinned);

            } else {
                stickyHeaderTextView.setText(R.string.application_list_header_text_unpinned);
            }

            stickyHeader.setVisibility(View.VISIBLE);
            stickyHeaderTextView.requestLayout();
            return;

        } while (false);

        stickyHeader.setVisibility(View.GONE);
    }

    private void observeViewModel() {
        viewModel.observeApplicationListData(getViewLifecycleOwner(), applicationListData -> {
            listAdapter.submitApplicationListData(applicationListData);
        });

        viewModel.observeLoading(getViewLifecycleOwner(), loading -> {
            loadingIndicator.setVisibility(loading ? View.VISIBLE : View.GONE);
            refreshStickyHeader();
        });
    }

    private void reloadApplicationList() {
        viewModel.reloadApplicationList();
    }

    private void requestSearch(String query, int delayMillis) {
        viewModel.setQuery(query, delayMillis);
    }

    @Override
    public void onResume() {
        super.onResume();
        reloadApplicationList();
    }

    private void onItemClicked(Application application, boolean pinned) {
        launchApplicationDetailsScreen(application.getPackageName());
    }

    private boolean onItemLongClicked(Application application, boolean pinned, View itemView) {
        final var popupMenu = new ApplicationItemPopupMenu(
                requireContext()
                , itemView
                , viewModel
                , application
                , pinned
        );

        popupMenu.show();
        return true;
    }

    private void launchApplicationDetailsScreen(String packageName) {
        final Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS);

        final Uri uri = Uri.fromParts("package", packageName, null);
        intent.setData(uri);
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);

        try {
            startActivity(intent);

        } catch (ActivityNotFoundException ignored) {
        }
    }
}
