package com.raiiware.forcestophelper.applicationlist;

import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.StringRes;
import androidx.recyclerview.widget.DiffUtil;
import androidx.recyclerview.widget.ListAdapter;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.color.MaterialColors;
import com.raiiware.forcestophelper.R;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;


final class ApplicationListAdapter extends ListAdapter<ApplicationListAdapter.Item, ApplicationListAdapter.ViewHolder> {

    private static final int VIEW_TYPE_APPLICATION_ITEM          = 1;
    private static final int VIEW_TYPE_APPLICATION_ITEM_STOPPED  = 2;
    private static final int VIEW_TYPE_APPLICATION_ITEM_DISABLED = 3;
    private static final int VIEW_TYPE_HEADER_ITEM               = 4;

    private final @NonNull ApplicationItemListener itemListener;
    private                boolean                 containsHeaders = false;

    ApplicationListAdapter(@NonNull ApplicationItemListener itemListener) {
        super(DIFF_CALLBACK);
        this.itemListener = itemListener;
    }

    private static final DiffUtil.ItemCallback<Item> DIFF_CALLBACK = new DiffUtil.ItemCallback<>() {
        @Override
        public boolean areItemsTheSame(@NonNull Item oldItem, @NonNull Item newItem) {
            return oldItem.isSameItem(newItem);
        }

        @Override
        public boolean areContentsTheSame(@NonNull Item oldItem, @NonNull Item newItem) {
            return oldItem.hasSameContents(newItem);
        }
    };

    void submitApplicationListData(ApplicationListData applicationListData) {
        final List<Application> pinnedList   = applicationListData.getPinnedList();
        final List<Application> unpinnedList = applicationListData.getUnpinnedList();

        final List<Item> itemList = new ArrayList<>(1 + pinnedList.size() + 1 + unpinnedList.size());

        if (pinnedList.isEmpty()) {
            for (final var application : unpinnedList) {
                itemList.add(new ApplicationItem(application, false));
            }

            this.containsHeaders = false;

        } else {
            itemList.add(new HeaderItem(1L, R.string.application_list_header_text_pinned, true));

            for (final var application : pinnedList) {
                itemList.add(new ApplicationItem(application, true));
            }

            if (! unpinnedList.isEmpty()) {
                itemList.add(new HeaderItem(2L, R.string.application_list_header_text_unpinned, false));

                for (final var application : unpinnedList) {
                    itemList.add(new ApplicationItem(application, false));
                }
            }

            this.containsHeaders = true;
        }

        submitList(itemList);
    }

    boolean containsHeaders() {
        return containsHeaders;
    }

    @Override
    public int getItemViewType(int position) {
        return getItem(position).getViewType();
    }

    @Override
    public @NonNull ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        final LayoutInflater inflater = LayoutInflater.from(parent.getContext());

        if (viewType == VIEW_TYPE_APPLICATION_ITEM) {
            final View view = inflater.inflate(R.layout.application_list_application_item, parent, false);
            return new ApplicationItemViewHolder(view, VIEW_TYPE_APPLICATION_ITEM);
        }

        if (viewType == VIEW_TYPE_APPLICATION_ITEM_STOPPED) {
            final View view = inflater.inflate(R.layout.application_list_application_item_stopped, parent, false);
            return new ApplicationItemViewHolder(view, VIEW_TYPE_APPLICATION_ITEM_STOPPED);
        }

        if (viewType == VIEW_TYPE_APPLICATION_ITEM_DISABLED) {
            final View view = inflater.inflate(R.layout.application_list_application_item_with_state_indicator, parent, false);
            return new ApplicationItemViewHolder(view, VIEW_TYPE_APPLICATION_ITEM_DISABLED);
        }

        if (viewType == VIEW_TYPE_HEADER_ITEM) {
            final View view = inflater.inflate(R.layout.application_list_header_item, parent, false);
            return new HeaderItemViewHolder(view);
        }

        throw new IllegalArgumentException();
    }

    @Override
    public void onBindViewHolder(@NonNull ViewHolder holder, int position) {
        holder.bind(getItem(position), itemListener);
    }

    @Override
    public void onViewRecycled(@NonNull ViewHolder holder) {
        holder.recycle();
        super.onViewRecycled(holder);
    }

    boolean isItemPinned(int position) {
        return getItem(position).isPinned();
    }

    static abstract class Item {
        abstract boolean isPinned();

        abstract int getViewType();

        abstract boolean isSameItem(Item item);
        abstract boolean hasSameContents(Item item);
    }

    static abstract class ViewHolder extends RecyclerView.ViewHolder {
        ViewHolder(@NonNull View itemView) {
            super(itemView);
        }

        abstract void bind(Item item, final ApplicationItemListener itemListener);

        void recycle() {
        }
    }

    private static final class ApplicationItem extends Item {
        private final Application application;
        private final boolean     pinned;
        private final boolean     stopped;
        private final boolean     enabled;

        ApplicationItem(Application application, boolean pinned) {
            this.application = application;
            this.pinned      = pinned;
            this.stopped     = application.isStopped();
            this.enabled     = application.isEnabled();
        }

        Application getApplication() {
            return application;
        }

        boolean isPinned() {
            return pinned;
        }

        @Override
        int getViewType() {
            if (stopped) return VIEW_TYPE_APPLICATION_ITEM_STOPPED;
            if (! enabled) return VIEW_TYPE_APPLICATION_ITEM_DISABLED;

            return VIEW_TYPE_APPLICATION_ITEM;
        }

        boolean isSameItem(Item item) {
            if (item instanceof ApplicationItem) {
                final ApplicationItem that = (ApplicationItem) item;
                if (pinned != that.pinned) return false;
                if (stopped != that.stopped) return false;
                if (enabled != that.enabled) return false;
                return Objects.equals(application.getPackageName(), that.application.getPackageName());

            } else {
                return false;
            }
        }

        boolean hasSameContents(Item item) {
            if (item instanceof ApplicationItem) {
                final ApplicationItem that = (ApplicationItem) item;
                if (pinned != that.pinned) return false;
                return Objects.equals(application, that.getApplication());

            } else {
                return false;
            }
        }
    }

    private static final class ApplicationItemViewHolder extends ViewHolder {
        private final ImageView applicationIconImageview;
        private final TextView  applicationLabelTextView;
        private final TextView  packageNameTextView;
        private final TextView  statusIndicatorTextView;

        ApplicationItemViewHolder(@NonNull View itemView, int viewType) {
            super(itemView);
            this.applicationIconImageview = itemView.findViewById(R.id.application_icon);
            this.applicationLabelTextView = itemView.findViewById(R.id.application_label);
            this.packageNameTextView      = itemView.findViewById(R.id.package_name);
            this.statusIndicatorTextView  = itemView.findViewById(R.id.application_state_indicator);

            if (viewType == VIEW_TYPE_APPLICATION_ITEM_STOPPED) {
                float overlayAlpha   = 0.1f;
                int   colorSurface   = MaterialColors.getColor(itemView, com.google.android.material.R.attr.colorSurface);
                int   colorOnSurface = MaterialColors.getColor(itemView, com.google.android.material.R.attr.colorOnSurface);
                int   compositeColor = MaterialColors.layer(colorSurface, colorOnSurface, overlayAlpha);
                itemView.setBackgroundColor(compositeColor);
            }
        }

        @Override
        void bind(Item item, final ApplicationItemListener itemListener) {
            bindImpl((ApplicationItem) item, itemListener);
        }
        private void bindImpl(ApplicationItem item, final ApplicationItemListener itemListener) {
            final Application application     = item.getApplication();
            final boolean     pinned          = item.isPinned();
            final Drawable    applicationIcon = application.getApplicationIcon();
            final String      applicationLabel = application.getApplicationLabel();
            final String      packageName      = application.getPackageName();

            if (application.isEnabled()) {
                applicationIconImageview.setImageDrawable(applicationIcon);

            } else {
                final var applicationIconDisabled = applicationIcon.mutate();
                applicationIconDisabled.setColorFilter(
                        new PorterDuffColorFilter(
                                Color.argb(127, 0, 0, 0)
                                , PorterDuff.Mode.SRC_ATOP)
                );

                applicationIconImageview.setImageDrawable(applicationIconDisabled);
            }

            if (TextUtils.isEmpty(applicationLabel)) {
                applicationLabelTextView.setText("");
                applicationLabelTextView.setVisibility(View.GONE);
                packageNameTextView.setText(packageName);

            } else {
                applicationLabelTextView.setText(applicationLabel);
                applicationLabelTextView.setVisibility(View.VISIBLE);
                packageNameTextView.setText(packageName);
            }

            if (statusIndicatorTextView != null) {
                if (application.isStopped()) {
                    if (application.isEnabled()) {
                        statusIndicatorTextView.setText(R.string.application_list_application_state_indicator_stopped);
                    } else {
                        statusIndicatorTextView.setText(R.string.application_list_application_state_indicator_stopped_disabled);
                    }

                } else if (! application.isEnabled()) {
                    statusIndicatorTextView.setText(R.string.application_list_application_state_indicator_disabled);

                } else {
                    statusIndicatorTextView.setText("");
                }
            }

            if (itemView.isClickable()) {
                itemView.setOnClickListener(v -> itemListener.onApplicationItemClicked(application, pinned));
            }

            if (itemView.isLongClickable()) {
                itemView.setOnLongClickListener(v -> itemListener.onApplicationItemLongClicked(application, pinned, itemView));
            }
        }

        @Override
        void recycle() {
            applicationIconImageview.setImageDrawable(null);

            itemView.setOnClickListener(null);
            itemView.setOnLongClickListener(null);
        }
    }

    private static final class HeaderItem extends Item {
        private final            long    id;
        private final @StringRes int     headerTextResId;
        private final            boolean pinned;

        HeaderItem(long id, @StringRes int headerTextResId, boolean pinned) {
            this.id              = id;
            this.headerTextResId = headerTextResId;
            this.pinned          = pinned;
        }

        @StringRes int getHeaderTextResId() {
            return headerTextResId;
        }

        boolean isPinned() {
            return pinned;
        }

        @Override
        int getViewType() {
            return VIEW_TYPE_HEADER_ITEM;
        }

        @Override
        boolean isSameItem(Item item) {
            if (item instanceof HeaderItem) {
                final HeaderItem that = (HeaderItem) item;
                return id == that.id;

            } else {
                return false;
            }
        }

        @Override
        boolean hasSameContents(Item item) {
            if (item instanceof HeaderItem) {
                final HeaderItem that = (HeaderItem) item;
                return id == that.id;

            } else {
                return false;
            }
        }
    }

    private static final class HeaderItemViewHolder extends ViewHolder {
        private final TextView  headerTextTextView;

        HeaderItemViewHolder(@NonNull View itemView) {
            super(itemView);
            this.headerTextTextView = itemView.findViewById(R.id.header_text);
        }

        @Override
        void bind(Item item, final ApplicationItemListener itemListener) {
            bindImpl((HeaderItem) item);
        }
        private void bindImpl(HeaderItem item) {
            final @StringRes int headerTextResId = item.getHeaderTextResId();

            headerTextTextView.setText(headerTextResId);
        }
    }
}
