package com.raiiware.forcestophelper.applicationlist;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.annotation.WorkerThread;
import androidx.lifecycle.AndroidViewModel;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.MutableLiveData;
import androidx.lifecycle.Observer;

import com.raiiware.forcestophelper.database.AppDatabase;
import com.raiiware.forcestophelper.database.PinnedApplicationDao;

import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


@MainThread
public final class ApplicationListViewModel extends AndroidViewModel {

    private final ExecutorService                      executor;
    private final PinnedApplicationDao                 pinnedApplicationDao;
    private final ApplicationListDataBuilder           applicationListDataBuilder;
    private final ApplicationCriteriaBuilder           applicationCriteriaBuilder;
    private       ApplicationSortingOrder              applicationSortingOrder;
    private final MutableLiveData<ApplicationListData> applicationListDataLiveData;
    private final MutableLiveData<Boolean>             loadingLiveData;

    public ApplicationListViewModel(@NonNull android.app.Application application) {
        super(application);
        this.executor                    = Executors.newSingleThreadExecutor();
        this.pinnedApplicationDao        = AppDatabase.getDatabase(application.getApplicationContext()).pinnedApplicationDao();
        this.applicationListDataBuilder  = new ApplicationListDataBuilder(application.getPackageManager());
        this.applicationSortingOrder     = ApplicationSortingOrder.APPLICATION_LABEL;
        this.applicationCriteriaBuilder  = new ApplicationCriteriaBuilder();
        this.applicationListDataLiveData = new MutableLiveData<>();
        this.loadingLiveData             = new MutableLiveData<>(false);
    }

    void observeApplicationListData(@NonNull LifecycleOwner lifecycleOwner, @NonNull Observer<ApplicationListData> observer) {
        applicationListDataLiveData.observe(lifecycleOwner, observer);
    }

    void observeLoading(@NonNull LifecycleOwner lifecycleOwner, @NonNull Observer<Boolean> observer) {
        loadingLiveData.observe(lifecycleOwner, observer);
    }

    boolean isLoading() {
        return Boolean.TRUE.equals(loadingLiveData.getValue());
    }

    boolean shouldShowStoppedApps() {
        return applicationCriteriaBuilder.shouldIncludeStoppedApps();
    }

    void toggleShowStoppedApps() {
        applicationCriteriaBuilder.toggleIncludeStoppedApps();

        reloadApplicationList();
    }

    ApplicationSortingOrder getSortingOrder() {
        return applicationSortingOrder;
    }

    void setSortingOrder(ApplicationSortingOrder sortingOrder) {
        if (this.applicationSortingOrder == sortingOrder) return;

        this.applicationSortingOrder = sortingOrder;

        reloadApplicationList();
    }

    void setQuery(String query, int delayMillis) {
        applicationCriteriaBuilder.setQuery(query);

        reloadApplicationListDelayed(delayMillis);
    }

    String getQuery() {
        return applicationCriteriaBuilder.getQuery();
    }

    void reloadApplicationList() {
        final var applicationCriteria   = applicationCriteriaBuilder.build();
        final var applicationComparator = applicationSortingOrder.getApplicationComparator();

        executor.execute(() -> reloadApplicationListImpl(applicationCriteria, applicationComparator));
    }

    private
    void reloadApplicationListDelayed(long delayMillis) {
        if (delayMillis <= 0) {
            reloadApplicationList();

        } else {
            messageQueue.removeMessages(MSG_RELOAD);
            messageQueue.sendEmptyMessageDelayed(MSG_RELOAD, delayMillis);
        }
    }

    void setApplicationPinned(Application application, final boolean pinned) {
        final String packageName = application.getPackageName();

        final var applicationCriteria   = applicationCriteriaBuilder.build();
        final var applicationComparator = applicationSortingOrder.getApplicationComparator();

        executor.execute(() -> {
            if (pinned) {
                pinnedApplicationDao.registerPinnedApplication(packageName);

            } else {
                pinnedApplicationDao.unregisterPinnedApplication(packageName);
            }

            reloadApplicationListImpl(applicationCriteria, applicationComparator);
        });
    }

    private static final int MSG_RELOAD = 1;

    private final Handler messageQueue = new Handler(Looper.getMainLooper(), this::handleMessage);

    private boolean handleMessage(@NonNull Message message) {
        //noinspection SwitchStatementWithTooFewBranches
        switch (message.what) {
            case MSG_RELOAD:
                reloadApplicationList();
                break;
        }
        return false;
    }

    @WorkerThread
    private void reloadApplicationListImpl(ApplicationCriteria applicationCriteria, Comparator<Application> applicationComparator) {
        loadingLiveData.postValue(true);

        final Set<String> pinnedPackageNames = new HashSet<>(pinnedApplicationDao.getPinnedPackageNames());

        final ApplicationListData applicationListData = applicationListDataBuilder.buildApplicationListData(
                applicationCriteria
                , applicationComparator
                , pinnedPackageNames
        );

        applicationListDataLiveData.postValue(applicationListData);
        loadingLiveData.postValue(false);
    }
}
