//
// Created by pent on 4/19/25.
//

#ifndef PAGAN_SAMPLEHANDLE_H
#define PAGAN_SAMPLEHANDLE_H
#include <string>
#include <sstream>
#include <unordered_map>
#include "PitchedBuffer.h"
#include "EffectProfileBuffer.h"
#include "VolumeEnvelope.h"
#include <cmath>
#include "Oscillator.h"

class NoFrameDataException: public std::exception {};

int SampleHandleUUIDGen = 0;
float MAX_VOLUME = 1 / 1.27;

// Values correspond with values defined in EffectType.kt
const int TYPE_VOLUME = 1;
const int TYPE_LOWPASS = 2;
const int TYPE_DELAY = 3;
const int TYPE_PAN = 5;

//const int TYPE_FREQUENCY_DOMAIN = 1024;
//const int TYPE_REVERB = TYPE_FREQUENCY_DOMAIN | 2;
//const int TYPE_EQUALIZER = TYPE_FREQUENCY_DOMAIN | 3;

// TODO Modulations
// modulation_envelope, modulation_lfo, modulators
class SampleHandle {
    float initial_frame_factor;
    // int uuid;

    public:
        int uuid;
        SampleData* data;
        int sample_rate;
        jfloat initial_attenuation;
        int loop_start;
        int loop_end;
        int stereo_mode;
        VolumeEnvelope* volume_envelope;
        float pitch_shift;
        float filter_cutoff;
        float pan;
        PitchedBuffer** data_buffers;
        int buffer_count;

        int working_frame;
        int release_frame;
        int kill_frame;
        bool is_dead;
        int active_buffer;

        Oscillator* vibrato_oscillator;
        int vibrato_delay;
        float vibrato_pitch;

        explicit SampleHandle(
                SampleData* data,
                int sample_rate,
                jfloat initial_attenuation,
                jint loop_start,
                jint loop_end,
                int stereo_mode,
                VolumeEnvelope* volume_envelope,
                float pitch_shift,
                float filter_cutoff,
                float pan,
                float vibrato_frequency,
                float vibrato_delay,
                float vibrato_pitch
        ) {
            this->uuid = SampleHandleUUIDGen++;
            this->data = data;
            this->sample_rate = sample_rate;
            this->initial_attenuation = initial_attenuation;
            this->loop_end = loop_end;
            this->loop_start = loop_start;
            this->stereo_mode = stereo_mode;
            this->volume_envelope = volume_envelope;
            this->pitch_shift = pitch_shift;
            this->filter_cutoff = filter_cutoff;
            this->pan = pan;
            this->vibrato_delay = (int)(vibrato_delay * this->sample_rate);
            this->vibrato_pitch = vibrato_pitch;

            if (vibrato_frequency != 0 && vibrato_pitch != 1) {
                this->vibrato_oscillator = new Oscillator(this->sample_rate, vibrato_frequency);
            } else {
                this->vibrato_oscillator = nullptr;
            }

            this->secondary_setup(nullptr, 0);
        }

        explicit SampleHandle(
                SampleData* data,
                int sample_rate,
                jfloat initial_attenuation,
                jint loop_start,
                jint loop_end,
                int stereo_mode,
                VolumeEnvelope* volume_envelope,
                float pitch_shift,
                float filter_cutoff,
                float pan,
                PitchedBuffer** data_buffers,
                int buffer_count,
                Oscillator* vibrato_oscillator,
                float vibrato_delay,
                float vibrato_pitch
        ) {
            this->uuid = SampleHandleUUIDGen++;
            this->data = data;
            this->sample_rate = sample_rate;
            this->initial_attenuation = initial_attenuation;
            this->loop_end = loop_end;
            this->loop_start = loop_start;
            this->stereo_mode = stereo_mode;
            this->volume_envelope = volume_envelope;
            this->pitch_shift = pitch_shift;
            this->filter_cutoff = filter_cutoff;
            this->pan = pan;
            this->vibrato_oscillator = vibrato_oscillator;
            this->vibrato_delay = (int)(vibrato_delay * this->sample_rate);
            this->vibrato_pitch = vibrato_pitch;

            this->secondary_setup(data_buffers, buffer_count);
        }

        void secondary_setup(PitchedBuffer** input_buffers, int count) {
            this->initial_frame_factor = 1 / std::pow(10, this->initial_attenuation);
            this->working_frame = 0;
            this->release_frame = -1;
            this->kill_frame = -1;
            this->is_dead = false;
            this->active_buffer = 0;

            if (count > 0) {
                this->buffer_count = count;
                this->data_buffers = (PitchedBuffer**)malloc(sizeof(PitchedBuffer*) * count);
                for (int i = 0; i < count; i++) {
                    PitchedBuffer* buffer = input_buffers[i];
                    auto* ptr = (PitchedBuffer*)malloc(sizeof(PitchedBuffer));
                    new (ptr) PitchedBuffer(buffer);
                    this->data_buffers[i] = ptr;
                }
            } else if (this->loop_start > -1 && this->loop_start != this->loop_end) {
                this->data_buffers = (PitchedBuffer**)malloc(sizeof(PitchedBuffer*) * 3);
                auto* ptr = (PitchedBuffer*)malloc(sizeof(PitchedBuffer));
                new (ptr) PitchedBuffer(this->data, this->pitch_shift, 0, this->loop_start, false);
                this->data_buffers[0] = ptr;

                ptr = (PitchedBuffer*)malloc(sizeof(PitchedBuffer));
                new (ptr) PitchedBuffer(this->data, this->pitch_shift, this->loop_start, this->loop_end, true);
                this->data_buffers[1] = ptr;

                ptr = (PitchedBuffer*)malloc(sizeof(PitchedBuffer));
                new (ptr) PitchedBuffer(this->data, this->pitch_shift, this->loop_end, this->data->size, false);
                this->data_buffers[2] = ptr;

                this->buffer_count = 3;
            } else {
                this->data_buffers = (PitchedBuffer**)malloc(sizeof(PitchedBuffer*));
                auto* ptr = (PitchedBuffer*)malloc(sizeof(PitchedBuffer));
                new (ptr) PitchedBuffer(this->data, this->pitch_shift, 0, this->data->size, false);
                this->data_buffers[0] = ptr;

                this->buffer_count = 1;
            }
        }

        explicit SampleHandle(SampleHandle* original) {
            this->uuid = SampleHandleUUIDGen++;
            this->data = original->data;

            this->sample_rate = original->sample_rate;
            this->initial_attenuation = original->initial_attenuation;
            this->loop_end = original->loop_end;
            this->loop_start = original->loop_start;

            this->stereo_mode = original->stereo_mode;
            this->pitch_shift = original->pitch_shift;
            this->filter_cutoff = original->filter_cutoff;
            this->pan = original->pan;

            this->volume_envelope = (VolumeEnvelope*)malloc(sizeof(VolumeEnvelope));
            new (this->volume_envelope) VolumeEnvelope(original->volume_envelope);

            this->secondary_setup(original->data_buffers, original->buffer_count);

            this->working_frame = original->working_frame;
            this->release_frame = original->release_frame;
            this->kill_frame = original->kill_frame;
            this->is_dead = original->is_dead;

            if (original->vibrato_oscillator != nullptr) {
                this->vibrato_oscillator = new Oscillator(original->vibrato_oscillator->sample_rate, original->vibrato_oscillator->frequency);
                this->vibrato_pitch = original->vibrato_pitch;
                this->vibrato_delay = original->vibrato_delay;
            } else {
                this->vibrato_oscillator = nullptr;
                this->vibrato_pitch = 0;
                this->vibrato_delay = 0;
            }
        }

        ~SampleHandle() {
            for (int i = 0; i < this->buffer_count; i++) {
                delete this->data_buffers[i];
            }
            delete[] this->data_buffers;

            this->volume_envelope->~VolumeEnvelope();
            free(this->volume_envelope);
        }

        void set_release_frame(int frame) {
            this->release_frame = frame;
        }

        void set_working_frame(int frame) {
            this->working_frame = frame;
            if (this->kill_frame > -1 && (this->working_frame >= this->kill_frame || (this->working_frame >= this->release_frame + this->volume_envelope->frames_release))) {
                this->is_dead = true;
                return;
            }

            try {
                if (this->release_frame == -1 || this->release_frame > frame) {
                    if (this->loop_start == -1 || frame < this->data_buffers[0]->virtual_size) {
                        this->data_buffers[0]->set_position(frame);
                        this->active_buffer = 0;
                    } else {
                        this->data_buffers[1]->set_position((frame - this->data_buffers[0]->virtual_size));
                        this->active_buffer = 1;
                    }
                } else if (this->loop_start > -1 && this->loop_start < this->loop_end) {
                    if (frame < this->data_buffers[0]->virtual_size) {
                        this->data_buffers[0]->set_position(frame);
                        this->active_buffer = 0;
                    } else if (frame < this->data_buffers[1]->virtual_size) {
                        this->data_buffers[1]->set_position(frame - this->data_buffers[0]->virtual_size);
                        this->active_buffer = 1;
                    } else {
                        int remainder = frame - this->release_frame;
                        int loop_size = this->loop_end - this->loop_start;
                        if (remainder < loop_size) {
                            this->data_buffers[1]->set_position(remainder);
                            this->active_buffer = 1;
                        } else {
                            int loop_count = (this->release_frame - this->loop_start) / loop_size;
                            this->data_buffers[2]->set_position(frame - this->data_buffers[0]->virtual_size - (loop_count * this->data_buffers[1]->virtual_size));
                            this->active_buffer = 2;
                        }
                    }
                } else {
                    this->data_buffers[0]->set_position(frame);
                    this->active_buffer = 0;
                }
                this->is_dead = false;
            } catch (PitchedBufferOverflow& e) {
                this->is_dead = true;
            }
        }

        int get_release_duration() const {
            return this->volume_envelope->frames_release;
        }

        PitchedBuffer* get_active_data_buffer() const {
            return this->data_buffers[this->active_buffer];
        }

        // TODO: Calculate on init
        std::tuple<float, float> get_balance() {
            std::tuple<float, float> output;
            switch (this->stereo_mode & 0x000F) {
                case 0x01: {
                    output = std::make_tuple(
                        1 + this->pan,
                        -1 * (-1 + this->pan)
                    );
                    break;
                }

                case 0x02: {
                    output = std::make_tuple(
                        1 + this->pan,
                        0
                    );
                    break;
                }

                case 0x04: {
                    output = std::make_tuple(
                        0,
                        -1 * (-1 + this->pan)
                    );
                    break;
                }

                default: {
                    // TODO: LINKED, but treat as mono for now
                    output = std::make_tuple(
                            1 + this->pan,
                            -1 * (-1 + this->pan)
                    );
                    break;
                }
            }

            return output;
        }

        // Instead of multiplexing the channels, alternating between left and right
        // The 2 channels are split length-wise. the first entire half is the right channel, followed by the left
        // This is so its easier to FFT later in the process.
        void get_next_frames(float* buffer, int target_size, int left_padding) {
            int actual_size = target_size;
            std::tuple<float, float> working_pan = this->get_balance();

            // No need to smooth the left padding since the handle won't start, then have a gap, then continue
            for (int i = 0; i < left_padding; i++) {
                buffer[i] = 0;
                buffer[i + target_size] = 0;
            }

            for (int i = left_padding; i < target_size; i++) {
                float frame;
                try {
                    frame = this->get_next_frame();
                } catch (NoFrameDataException &e) {
                    actual_size = i;
                    break;
                }

                buffer[i] = frame * std::get<0>(working_pan);
                buffer[i + target_size] = frame * std::get<1>(working_pan);
            }

            for (int i = actual_size; i < target_size; i++) {
                buffer[i] = 0;
                buffer[i + target_size] = 0;
            }
        }

        float get_next_frame() {
            if (this->is_dead) {
                throw NoFrameDataException();
            }

            bool is_pressed = this->is_pressed();
            if (this->working_frame < this->volume_envelope->frames_delay) {
                this->working_frame += 1;
                return 0;
            }

            float frame_factor = this->initial_frame_factor;
            if (this->working_frame - this->volume_envelope->frames_delay < this->volume_envelope->frames_attack) {
                float r = ((float)this->working_frame - (float)this->volume_envelope->frames_delay) / (float)this->volume_envelope->frames_attack;
                frame_factor *= r;
            } else if (this->working_frame - this->volume_envelope->frames_attack - this->volume_envelope->frames_delay < this->volume_envelope->frames_hold) {
                // PASS
            } else if (this->volume_envelope->sustain_attenuation > 0) {
                int relative_frame = this->working_frame - this->volume_envelope->frames_delay - this->volume_envelope->frames_attack;
                if (relative_frame < this->volume_envelope->frames_decay) {
                    float r = ((float)relative_frame / (float)this->volume_envelope->frames_decay);
                    frame_factor /= std::pow((float)10, r * this->volume_envelope->sustain_attenuation);
                } else {
                    frame_factor /= this->volume_envelope->true_sustain_attenuation;
                }
            }

            if (this->get_active_data_buffer()->is_overflowing()) {
                if (!is_pressed || this->loop_start == -1) {
                    if (this->active_buffer < this->buffer_count - 1) {
                        this->active_buffer += 1;
                    } else {
                        this->is_dead = true;
                        throw NoFrameDataException();
                    }
                } else if (this->active_buffer == 0) {
                    this->active_buffer += 1;
                    this->get_active_data_buffer()->set_position(0);
                }
            }

            if (!is_pressed) {
                int pos = 0;
                for (int i = 0; i < this->buffer_count; i++) {
                    pos += this->data_buffers[i]->virtual_size;
                }
                int release_frame_count = std::min(this->volume_envelope->frames_release, pos - this->release_frame);
                int current_position_release = this->working_frame - this->release_frame;
                if (current_position_release < release_frame_count) {
                    frame_factor *= 1 - ((float)current_position_release / (float)release_frame_count);
                } else {
                    this->is_dead = true;
                    throw NoFrameDataException();
                }
            }

            this->working_frame += 1;
            if (this->active_buffer >= this->buffer_count) {
                this->is_dead = true;
                throw NoFrameDataException();
            }

            float frame_value;
            try {
                frame_value = this->get_active_data_buffer()->get();
            } catch (PitchedBufferOverflow& e) {
                this->is_dead = true;
                throw NoFrameDataException();
            }

            if (this->vibrato_oscillator != nullptr && this->vibrato_delay < this->working_frame) {
                this->get_active_data_buffer()->repitch(1 + ((this->vibrato_pitch - 1) * this->vibrato_oscillator->next()));
            }

            return frame_value * frame_factor * MAX_VOLUME;
        }

        void release_note() {
            this->set_release_frame(this->working_frame);
        }

        void set_kill_frame(int f) {
            this->kill_frame = f;
        }

        bool is_pressed() {
            return this->release_frame == -1 || this->release_frame > this->working_frame;
        }

        int get_duration() {
            if (this->release_frame > -1) {
                return this->release_frame + this->get_release_duration();
            } else {
                return -1;
            }
        }

        void repitch(float adjustment) const {
            for (int i = 0; i < this->buffer_count; i++) {
                this->data_buffers[i]->repitch(adjustment);
            }
        }
};

#endif //PAGAN_SAMPLEHANDLE_H
