package com.prajwalch.torrentsearch.ui.bookmarks

import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.RowScope
import androidx.compose.foundation.layout.consumeWindowInsets
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.SwipeToDismissBox
import androidx.compose.material3.SwipeToDismissBoxValue
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberSwipeToDismissBoxState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.retain.retain
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalClipboard
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.pluralStringResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.dp
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import androidx.lifecycle.compose.collectAsStateWithLifecycle

import com.prajwalch.torrentsearch.R
import com.prajwalch.torrentsearch.domain.models.MagnetUri
import com.prajwalch.torrentsearch.domain.models.Torrent
import com.prajwalch.torrentsearch.extensions.copyText
import com.prajwalch.torrentsearch.ui.components.AnimatedScrollToTopFAB
import com.prajwalch.torrentsearch.ui.components.ArrowBackIconButton
import com.prajwalch.torrentsearch.ui.components.CollapsibleSearchBar
import com.prajwalch.torrentsearch.ui.components.DeleteForeverIconButton
import com.prajwalch.torrentsearch.ui.components.EmptyPlaceholder
import com.prajwalch.torrentsearch.ui.components.LazyColumnWithScrollbar
import com.prajwalch.torrentsearch.ui.components.RoundedDropdownMenu
import com.prajwalch.torrentsearch.ui.components.SearchIconButton
import com.prajwalch.torrentsearch.ui.components.SortDropdownMenu
import com.prajwalch.torrentsearch.ui.components.SortIconButton
import com.prajwalch.torrentsearch.ui.components.TorrentActionsBottomSheet
import com.prajwalch.torrentsearch.ui.components.TorrentListItem
import com.prajwalch.torrentsearch.ui.components.rememberCollapsibleSearchBarState
import com.prajwalch.torrentsearch.ui.rememberTorrentListState
import com.prajwalch.torrentsearch.ui.theme.spaces

import kotlinx.coroutines.launch

/** Name of the export file. */
private const val BOOKMARKS_EXPORT_FILE_NAME = "bookmarks.json"

/** Export file type. */
private const val BOOKMARKS_EXPORT_FILE_TYPE = "application/json"

/** Activity contract which allows the user to choose the export location. */
private val CreateDocumentContract =
    ActivityResultContracts.CreateDocument(BOOKMARKS_EXPORT_FILE_TYPE)

/** Activity contract which allows the user to choose the exported file. */
private val GetContentContract = ActivityResultContracts.GetContent()

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun BookmarksScreen(
    onNavigateBack: () -> Unit,
    onNavigateToSettings: () -> Unit,
    onDownloadTorrent: (MagnetUri) -> Unit,
    onShareMagnetLink: (MagnetUri) -> Unit,
    onOpenDescriptionPage: (String) -> Unit,
    onShareDescriptionPageUrl: (String) -> Unit,
    modifier: Modifier = Modifier,
    viewModel: BookmarksViewModel = hiltViewModel(),
) {
    val uiState by viewModel.uiState.collectAsStateWithLifecycle()

    val context = LocalContext.current
    val contentResolver = context.contentResolver

    // TODO: Proper error handling.
    val importFileChooserLauncher = rememberLauncherForActivityResult(
        contract = GetContentContract,
    ) { fileUri ->
        fileUri
            ?.let(contentResolver::openInputStream)
            ?.let(viewModel::importBookmarks)
    }
    val exportLocationChooserLauncher = rememberLauncherForActivityResult(
        contract = CreateDocumentContract,
    ) { fileUri ->
        fileUri
            ?.let(contentResolver::openOutputStream)
            ?.let(viewModel::exportBookmarks)
    }

    val coroutineScope = rememberCoroutineScope()
    val scrollBehavior = TopAppBarDefaults.enterAlwaysScrollBehavior()
    val torrentListState = rememberTorrentListState(itemsCount = { uiState.bookmarks.size })

    // Bookmark related states.
    var selectedBookmark by retain { mutableStateOf<Torrent?>(null) }
    val snackbarHostState = remember { SnackbarHostState() }

    selectedBookmark?.let { bookmark ->
        val clipboard = LocalClipboard.current

        val magnetLinkCopiedMessage = stringResource(
            R.string.torrent_list_magnet_link_copied_message
        )
        val urlCopiedMessage = stringResource(
            R.string.torrent_list_url_copied_message,
        )

        TorrentActionsBottomSheet(
            onDismiss = { selectedBookmark = null },
            title = bookmark.name,
            showNSFWBadge = bookmark.isNSFW(),
            onDeleteBookmark = {
                viewModel.deleteBookmarkedTorrent(torrent = bookmark)
            },
            onDownloadTorrent = {
                onDownloadTorrent(bookmark.magnetUri())
            },
            onCopyMagnetLink = {
                coroutineScope.launch {
                    clipboard.copyText(text = bookmark.magnetUri())
                    snackbarHostState.showSnackbar(message = magnetLinkCopiedMessage)
                }
            },
            onShareMagnetLink = {
                onShareMagnetLink(bookmark.magnetUri())
            },
            onOpenDescriptionPage = {
                onOpenDescriptionPage(bookmark.descriptionPageUrl)
            },
            onCopyDescriptionPageUrl = {
                coroutineScope.launch {
                    clipboard.copyText(text = bookmark.descriptionPageUrl)
                    snackbarHostState.showSnackbar(message = urlCopiedMessage)
                }
            },
            onShareDescriptionPageUrl = {
                onShareDescriptionPageUrl(bookmark.descriptionPageUrl)
            },
            enableDescriptionPageActions = bookmark.descriptionPageUrl.isNotEmpty(),
        )
    }

    var showDeleteAllConfirmationDialog by rememberSaveable { mutableStateOf(false) }
    if (showDeleteAllConfirmationDialog) {
        DeleteAllConfirmationDialog(
            onDismiss = { showDeleteAllConfirmationDialog = false },
            onConfirm = {
                viewModel.deleteAllBookmarks()
                showDeleteAllConfirmationDialog = false
            },
        )
    }

    val searchBarState = rememberCollapsibleSearchBarState(visibleOnInitial = false)
    val topBarTitle: @Composable () -> Unit = @Composable {
        CollapsibleSearchBar(
            state = searchBarState,
            onQueryChange = viewModel::filterBookmarks,
            placeholder = { Text(text = stringResource(R.string.bookmarks_search_query_hint)) },
        )
        if (!searchBarState.isVisible) {
            Column {
                Text(text = stringResource(R.string.bookmarks_screen_title))

                if (uiState.bookmarks.isNotEmpty()) {
                    BookmarksCount(
                        totalBookmarksCount = uiState.bookmarks.size,
                        currentBookmarksCount = uiState.bookmarks.size,
                    )
                }
            }
        }
    }
    val topBarActions: @Composable RowScope.() -> Unit = @Composable {
        var showSortOptions by rememberSaveable(uiState.sortOptions) { mutableStateOf(false) }
        var showAdditionalActions by rememberSaveable { mutableStateOf(false) }
        val enableBookmarksActions = uiState.bookmarks.isNotEmpty() || !searchBarState.isTextBlank

        if (!searchBarState.isVisible) {
            SearchIconButton(
                onClick = { searchBarState.showSearchBar() },
                enabled = enableBookmarksActions,
            )
            SortIconButton(
                onClick = { showSortOptions = true },
                enabled = enableBookmarksActions,
            )
            SortDropdownMenu(
                expanded = showSortOptions,
                onDismissRequest = { showSortOptions = false },
                currentCriteria = uiState.sortOptions.criteria,
                onChangeCriteria = viewModel::setSortCriteria,
                currentOrder = uiState.sortOptions.order,
                onChangeOrder = viewModel::setSortOrder,
            )
            DeleteForeverIconButton(
                onClick = { showDeleteAllConfirmationDialog = true },
                contentDescription = R.string.bookmarks_action_delete_all,
                enabled = enableBookmarksActions,
            )
        }
        Box {
            IconButton(onClick = { showAdditionalActions = true }) {
                Icon(
                    painter = painterResource(R.drawable.ic_more_vert),
                    contentDescription = null,
                )
            }
            TopBarAdditionalActionsDropdownMenu(
                expanded = showAdditionalActions,
                onDismiss = { showAdditionalActions = false },
                onImportBookmarks = {
                    // When mime type is given it restricts other type of files
                    // from being selectable.
                    importFileChooserLauncher.launch(BOOKMARKS_EXPORT_FILE_TYPE)
                },
                onExportBookmarks = {
                    // Takes file name to create on the selected location.
                    exportLocationChooserLauncher.launch(BOOKMARKS_EXPORT_FILE_NAME)
                },
                onNavigateToSettings = onNavigateToSettings,
            )
        }
    }

    Scaffold(
        modifier = Modifier
            .fillMaxSize()
            .nestedScroll(scrollBehavior.nestedScrollConnection)
            .then(modifier),
        snackbarHost = { SnackbarHost(hostState = snackbarHostState) },
        topBar = {
            TopAppBar(
                title = topBarTitle,
                navigationIcon = { ArrowBackIconButton(onClick = onNavigateBack) },
                actions = topBarActions,
                scrollBehavior = scrollBehavior,
            )
        },
        floatingActionButton = {
            AnimatedScrollToTopFAB(
                visible = torrentListState.showScrollTopButton,
                onClick = { coroutineScope.launch { torrentListState.scrollToTop() } },
            )
        },
    ) { innerPadding ->
        if (uiState.bookmarks.isEmpty()) {
            EmptyPlaceholder(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(innerPadding),
                title = R.string.bookmarks_empty_message,
            )
        } else {
            BookmarkList(
                modifier = Modifier
                    .fillMaxSize()
                    .consumeWindowInsets(innerPadding),
                bookmarks = uiState.bookmarks,
                onBookmarkClick = { selectedBookmark = it },
                onDeleteBookmark = viewModel::deleteBookmarkedTorrent,
                contentPadding = innerPadding,
                lazyListState = torrentListState.lazyListState,
            )
        }
    }
}

@Composable
private fun TopBarAdditionalActionsDropdownMenu(
    expanded: Boolean,
    onDismiss: () -> Unit,
    onImportBookmarks: () -> Unit,
    onExportBookmarks: () -> Unit,
    onNavigateToSettings: () -> Unit,
    modifier: Modifier = Modifier,
) {
    fun actionWithDismiss(action: () -> Unit) = {
        action()
        onDismiss()
    }
    RoundedDropdownMenu(
        modifier = modifier,
        expanded = expanded,
        onDismissRequest = onDismiss,
    ) {
        DropdownMenuItem(
            text = { Text(stringResource(R.string.bookmarks_action_import)) },
            onClick = actionWithDismiss(onImportBookmarks),
            leadingIcon = {
                Icon(
                    painter = painterResource(R.drawable.ic_download),
                    contentDescription = stringResource(R.string.bookmarks_action_import),
                )
            }
        )
        DropdownMenuItem(
            text = { Text(stringResource(R.string.bookmarks_action_export)) },
            onClick = actionWithDismiss(onExportBookmarks),
            leadingIcon = {
                Icon(
                    painter = painterResource(R.drawable.ic_upload),
                    contentDescription = stringResource(R.string.bookmarks_action_export),
                )
            }
        )
        DropdownMenuItem(
            text = { Text(stringResource(R.string.bookmarks_action_settings)) },
            onClick = actionWithDismiss(onNavigateToSettings),
            leadingIcon = {
                Icon(
                    painter = painterResource(R.drawable.ic_settings),
                    contentDescription = stringResource(R.string.bookmarks_action_settings),
                )
            },
        )
    }
}

@Composable
private fun BookmarksCount(
    totalBookmarksCount: Int,
    currentBookmarksCount: Int,
    modifier: Modifier = Modifier,
    color: Color = MaterialTheme.colorScheme.onSurfaceVariant,
    style: TextStyle = MaterialTheme.typography.labelMedium,
) {
    Text(
        modifier = modifier,
        text = pluralStringResource(
            R.plurals.bookmarks_count_format,
            totalBookmarksCount,
            currentBookmarksCount,
        ),
        color = color,
        style = style,
    )
}

@Composable
private fun DeleteAllConfirmationDialog(
    onDismiss: () -> Unit,
    onConfirm: () -> Unit,
    modifier: Modifier = Modifier,
) {
    AlertDialog(
        modifier = modifier,
        onDismissRequest = onDismiss,
        icon = {
            Icon(
                painter = painterResource(R.drawable.ic_delete_forever),
                contentDescription = null,
            )
        },
        title = {
            Text(text = stringResource(R.string.bookmarks_dialog_title_delete_all))
        },
        text = {
            Text(text = stringResource(R.string.bookmarks_dialog_message_delete_all))
        },
        confirmButton = {
            TextButton(onClick = onConfirm) {
                Text(text = stringResource(R.string.bookmarks_button_delete))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(text = stringResource(R.string.button_cancel))
            }
        },
    )
}

@Composable
private fun BookmarkList(
    bookmarks: List<Torrent>,
    onBookmarkClick: (Torrent) -> Unit,
    onDeleteBookmark: (Torrent) -> Unit,
    modifier: Modifier = Modifier,
    contentPadding: PaddingValues = PaddingValues(0.dp),
    lazyListState: LazyListState,
) {
    LazyColumnWithScrollbar(
        modifier = modifier,
        state = lazyListState,
        contentPadding = contentPadding,
    ) {
        items(items = bookmarks, key = { it.id }, contentType = { it.category }) {
            BookmarkListItem(
                modifier = Modifier.animateItem(),
                bookmark = it,
                onClick = { onBookmarkClick(it) },
                onDelete = { onDeleteBookmark(it) },
            )
            HorizontalDivider()
        }
    }
}

@Composable
private fun BookmarkListItem(
    bookmark: Torrent,
    onClick: () -> Unit,
    onDelete: () -> Unit,
    modifier: Modifier = Modifier,
) {
    val swipeToDismissBoxState = rememberSwipeToDismissBoxState()
    val coroutineScope = rememberCoroutineScope()

    SwipeToDismissBox(
        modifier = modifier,
        state = swipeToDismissBoxState,
        backgroundContent = {
            Icon(
                modifier = Modifier
                    .fillMaxSize()
                    .background(color = MaterialTheme.colorScheme.errorContainer)
                    .wrapContentSize(align = Alignment.CenterEnd)
                    .padding(horizontal = MaterialTheme.spaces.large),
                painter = painterResource(R.drawable.ic_delete),
                contentDescription = null,
                tint = MaterialTheme.colorScheme.onErrorContainer,
            )
        },
        enableDismissFromStartToEnd = false,
        onDismiss = { direction ->
            if (direction == SwipeToDismissBoxValue.EndToStart) {
                onDelete()
            } else {
                coroutineScope.launch { swipeToDismissBoxState.reset() }
            }
        },
    ) {
        TorrentListItem(
            modifier = Modifier.clickable(onClick = onClick),
            name = bookmark.name,
            size = bookmark.size,
            seeders = bookmark.seeders,
            peers = bookmark.peers,
            uploadDate = bookmark.uploadDate,
            category = bookmark.category,
            providerName = bookmark.providerName,
            isNSFW = bookmark.isNSFW(),
        )
    }
}