"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Options = exports.OptionsManager = void 0;
const types_1 = require("./types");
/**
 * Verifies the type of the value matches the expected type
 *
 * @param name name of option to check value for
 * @param expectedType expected type off the value for the option
 * @param value value to check
 * @throws TypeError When value is invalid type for option
 */
const verifyType = (name, expectedType, value) => {
    if (typeof value !== expectedType) {
        throw new TypeError(`Invalid type ${typeof value} for ${name}. Expected ${expectedType}`);
    }
};
/**
 * Provides a way to convert numeric timestamp to a formatted string.
 *
 * A custom formatter may be registered.
 * If one isn't registered, the default formatter will be used and the data will be formatted as HH:mm:SS.fff
 */
class OptionsManager {
    /**
     * Create the options manager
     */
    constructor() {
        /**
         * Combine segments if the {@link Segment.startTime}, {@link Segment.endTime}, and {@link Segment.speaker} match
         * between the current and prior segments
         *
         * Can be used with {@link combineSegments}. The {@link combineEqualTimes} rule is applied first.
         *
         * Can be used with {@link speakerChange}. The {@link speakerChange} rule is applied last.
         *
         * Cannot be used with {@link combineSpeaker}
         */
        this.combineEqualTimes = false;
        /**
         * Character to use when {@link combineEqualTimes} is true.
         */
        this.combineEqualTimesSeparator = "\n";
        /**
         * Combine segments where speaker is the same and concatenated `body` fits in the {@link combineSegmentsLength}
         *
         * Can be used with {@link combineEqualTimes}. The {@link combineSegments} rule is applied first.
         *
         * Can be used with {@link speakerChange}. The {@link speakerChange} rule is applied last.
         *
         * Cannot be used with {@link combineSpeaker}
         */
        this.combineSegments = false;
        /**
         * Max length of body text to use when {@link combineSegments} is true
         */
        this.combineSegmentsLength = types_1.DEFAULT_COMBINE_SEGMENTS_LENGTH;
        /**
         * Combine consecutive segments from the same speaker.
         *
         * Note: If this is enabled, {@link combineEqualTimes} and {@link combineSegments} will not be applied.
         *
         * Warning: if the transcript does not contain speaker information, resulting segment will contain entire transcript text.
         */
        this.combineSpeaker = false;
        /**
         * Only include {@link Segment.speaker} when speaker changes
         *
         * May be used in combination with {@link combineSpeaker}, {@link combineEqualTimes}, or {@link combineSegments}
         */
        this.speakerChange = false;
        /**
         * Get option value from it's name
         *
         * @param name name of option to get
         * @returns value of option. If unknown, returns undefined
         */
        this.getOptionByName = (name) => {
            let actual;
            switch (name) {
                case "combineEqualTimes":
                    actual = this.combineEqualTimes;
                    break;
                case "combineEqualTimesSeparator":
                    actual = this.combineEqualTimesSeparator;
                    break;
                case "combineSegments":
                    actual = this.combineSegments;
                    break;
                case "combineSegmentsLength":
                    actual = this.combineSegmentsLength;
                    break;
                case "combineSpeaker":
                    actual = this.combineSpeaker;
                    break;
                case "speakerChange":
                    actual = this.speakerChange;
                    break;
                default:
                    break;
            }
            return actual;
        };
        /**
         * Set option value using it's name
         *
         * @param name name of option to set
         * @param value value to set option to
         */
        this.setOptionByName = (name, value) => {
            switch (name) {
                case "combineEqualTimes":
                    verifyType(name, "boolean", value);
                    this.combineEqualTimes = value;
                    break;
                case "combineEqualTimesSeparator":
                    verifyType(name, "string", value);
                    this.combineEqualTimesSeparator = value;
                    break;
                case "combineSegments":
                    verifyType(name, "boolean", value);
                    this.combineSegments = value;
                    break;
                case "combineSegmentsLength":
                    verifyType(name, "number", value);
                    this.combineSegmentsLength = value;
                    break;
                case "combineSpeaker":
                    verifyType(name, "boolean", value);
                    this.combineSpeaker = value;
                    break;
                case "speakerChange":
                    verifyType(name, "boolean", value);
                    this.speakerChange = value;
                    break;
                default:
                    break;
            }
        };
        /**
         * Set all options to their default value
         */
        this.restoreDefaultSettings = () => {
            this.setOptions({
                combineEqualTimes: false,
                combineEqualTimesSeparator: "\n",
                combineSegments: false,
                combineSegmentsLength: types_1.DEFAULT_COMBINE_SEGMENTS_LENGTH,
                combineSpeaker: false,
                speakerChange: false,
            }, false);
        };
        /**
         * Set one or more options
         *
         * @param options the options to set
         * @param setDefault true: set all values to the default before setting values specified by `options`
         */
        this.setOptions = (options, setDefault = true) => {
            if (setDefault) {
                this.restoreDefaultSettings();
            }
            if (options === undefined) {
                return;
            }
            ;
            Object.keys(options).forEach((option) => {
                this.setOptionByName(option, options[option]);
            });
        };
        /**
         * Helper to determine if at least one option should be applied
         *
         * @returns true: at least one option set
         */
        this.optionsSet = () => {
            const { combineEqualTimes, combineSegments, combineSpeaker, speakerChange } = this;
            return combineEqualTimes || combineSegments || combineSpeaker || speakerChange;
        };
        if (!OptionsManager._instance) {
            OptionsManager._instance = this;
        }
        // eslint-disable-next-line no-constructor-return
        return OptionsManager._instance;
    }
}
exports.OptionsManager = OptionsManager;
exports.Options = new OptionsManager();
