"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseJSON = exports.isJSON = void 0;
const segments_1 = require("../segments");
const speaker_1 = require("../speaker");
const timestamp_1 = require("../timestamp");
/**
 * Determines if the value of data is a valid JSON transcript format
 *
 * @param data The transcript data
 * @returns True: data is valid JSON transcript format
 */
const isJSON = (data) => {
    return (data.startsWith("{") && data.endsWith("}")) || (data.startsWith("[") && data.endsWith("]"));
};
exports.isJSON = isJSON;
/**
 * Parse JSON data where segments are in the `segments` Array and in the {@link JSONSegment} format
 *
 * @param data Parsed JSON data
 * @returns An array of Segments from the parsed data
 */
const parseDictSegmentsJSON = (data) => {
    let outSegments = [];
    data.segments.forEach((segment) => {
        outSegments = (0, segments_1.addSegment)({
            startTime: segment.startTime,
            startTimeFormatted: timestamp_1.TimestampFormatter.format(segment.startTime),
            endTime: segment.endTime,
            endTimeFormatted: timestamp_1.TimestampFormatter.format(segment.endTime),
            speaker: segment.speaker,
            body: segment.body,
        }, outSegments);
    });
    return outSegments;
};
/**
 * Parse JSON data where top level item is a dict/object
 *
 * @param data The transcript data
 * @returns An array of Segments from the parsed data
 * @throws {TypeError} When JSON data does not match one of the valid formats
 */
const parseDictJSON = (data) => {
    let outSegments = [];
    if (Object.keys(data).length === 0) {
        return outSegments;
    }
    if ("segments" in data) {
        outSegments = parseDictSegmentsJSON(data);
    }
    else {
        throw new TypeError(`Unknown JSON dict transcript format`);
    }
    return outSegments;
};
/**
 * Convert {@link SubtitleSegment} to the {@link Segment} format used here
 *
 * @param data Segment parsed from JSON data
 * @returns Segment representing `data`.
 * Returns {@link undefined} when data does not match {@link SubtitleSegment} format.
 */
const getSegmentFromSubtitle = (data) => {
    if ("start" in data && "end" in data && "text" in data) {
        const { speaker, message } = (0, speaker_1.parseSpeaker)(data.text);
        const startTime = data.start / 1000;
        const endTime = data.end / 1000;
        const segment = {
            startTime,
            startTimeFormatted: timestamp_1.TimestampFormatter.format(startTime),
            endTime,
            endTimeFormatted: timestamp_1.TimestampFormatter.format(endTime),
            speaker,
            body: message,
        };
        if (Number.isNaN(segment.startTime)) {
            console.warn(`Computed start time is NaN: ${segment.startTime}`);
            return undefined;
        }
        if (Number.isNaN(segment.endTime)) {
            console.warn(`Computed end time is NaN: ${segment.endTime}`);
            return undefined;
        }
        return segment;
    }
    return undefined;
};
/**
 * Parse JSON data where items in data are in the {@link SubtitleSegment} format
 *
 * @param data Parsed JSON data
 * @returns An array of Segments from the parsed data
 * @throws {TypeError} When item in `data` does not match the {@link SubtitleSegment} format
 */
const parseListJSONSubtitle = (data) => {
    let outSegments = [];
    let lastSpeaker = "";
    data.forEach((subtitle, count) => {
        const subtitleSegment = getSegmentFromSubtitle(subtitle);
        if (subtitleSegment !== undefined) {
            lastSpeaker = subtitleSegment.speaker ? subtitleSegment.speaker : lastSpeaker;
            subtitleSegment.speaker = lastSpeaker;
            outSegments = (0, segments_1.addSegment)(subtitleSegment, outSegments);
        }
        else {
            throw new TypeError(`Unable to parse segment for item ${count}`);
        }
    });
    return outSegments;
};
/**
 * Parse JSON data where top level item is an Array
 *
 * @param data The transcript data
 * @returns An array of Segments from the parsed data
 * @throws {TypeError} When JSON data does not match one of the valid formats
 */
const parseListJSON = (data) => {
    let outSegments = [];
    if (data.length === 0) {
        return outSegments;
    }
    const subtitleSegment = getSegmentFromSubtitle(data[0]);
    if (subtitleSegment !== undefined) {
        outSegments = parseListJSONSubtitle(data);
    }
    else {
        throw new TypeError(`Unknown JSON list transcript format`);
    }
    return outSegments;
};
/**
 * Parse JSON data to an Array of {@link Segment}
 *
 * @param data The transcript data
 * @returns An array of Segments from the parsed data
 * @throws {TypeError} When `data` is not valid JSON format
 */
const parseJSON = (data) => {
    const dataTrimmed = data.trim();
    let outSegments = [];
    if (!(0, exports.isJSON)(dataTrimmed)) {
        throw new TypeError(`Data is not valid JSON format`);
    }
    let parsed;
    try {
        parsed = JSON.parse(data);
    }
    catch (e) {
        throw new TypeError(`Data is not valid JSON: ${e}`);
    }
    if (parsed.constructor === Object) {
        outSegments = parseDictJSON(parsed);
    }
    else if (parsed.constructor === Array) {
        outSegments = parseListJSON(parsed);
    }
    return outSegments;
};
exports.parseJSON = parseJSON;
