var _NativeModules$RNFile;

import { NativeModules, Platform } from 'react-native';
import { FileAccessEventEmitter, FileAccessNative } from './native';
export { Util } from './util';
/**
 * ID tracking next fetch request.
 */

let nextRequestId = 1;
/**
 * Process fetch events for the request.
 */

function registerFetchListener(requestId, resolve, reject, onProgress) {
  const listener = FileAccessEventEmitter.addListener('FetchEvent', event => {
    if (event.requestId !== requestId) {
      return;
    }

    if (event.state === 'progress') {
      onProgress === null || onProgress === void 0 ? void 0 : onProgress(event.bytesRead, event.contentLength, event.done);
    } else if (event.state === 'error') {
      listener.remove();
      reject(new Error(event.message));
    } else if (event.state === 'complete') {
      listener.remove();
      resolve({
        headers: event.headers,
        ok: event.ok,
        redirected: event.redirected,
        status: event.status,
        statusText: event.statusText,
        url: event.url
      });
    }
  });
}

export const FileSystem = {
  /**
   * Append content to a file.
   *
   * Default encoding of `data` is assumed utf8.
   */
  appendFile(path, data, encoding = 'utf8') {
    return FileAccessNative.appendFile(path, data, encoding);
  },

  /**
   * Append a file to another file.
   *
   * Returns number of bytes written.
   */
  concatFiles(source, target) {
    return FileAccessNative.concatFiles(source, target);
  },

  /**
   * Copy a file.
   */
  cp(source, target) {
    return FileAccessNative.cp(source, target);
  },

  /**
   * Copy a bundled asset file.
   *
   * When using Android asset type 'resource', include the folder, but skip the
   * file extension. For example use 'raw/foo', for the file 'res/raw/foo.txt'.
   * When possible, prefer using the 'assets/' folder; files in 'res/' have
   * naming restrictions imposed by Android.
   * https://developer.android.com/guide/topics/resources/providing-resources.html#OriginalFiles
   */
  cpAsset(asset, target, type = 'asset') {
    return Platform.OS === 'android' ? FileAccessNative.cpAsset(asset, target, type) : FileAccessNative.cpAsset(asset, target);
  },

  /**
   * Copy a file to an externally controlled location.
   *
   * On Android API level < 29, may require permission WRITE_EXTERNAL_STORAGE.
   */
  cpExternal(source, targetName, dir) {
    return FileAccessNative.cpExternal(source, targetName, dir);
  },

  /**
   * Check device available space.
   */
  df() {
    return FileAccessNative.df();
  },

  /**
   * Check if a path exists.
   */
  exists(path) {
    return FileAccessNative.exists(path);
  },

  /**
   * Save a network request to a file.
   */
  async fetch(resource, init, onProgress) {
    const requestId = nextRequestId++;
    return new Promise((resolve, reject) => {
      registerFetchListener(requestId, resolve, reject, onProgress);
      FileAccessNative.fetch(requestId, resource, init);
    });
  },

  /**
   * Save a network request to a file.
   */
  fetchManaged(resource, init, onProgress) {
    const requestId = nextRequestId++;
    return {
      cancel: () => FileAccessNative.cancelFetch(requestId),
      result: new Promise((resolve, reject) => {
        registerFetchListener(requestId, resolve, reject, onProgress);
        FileAccessNative.fetch(requestId, resource, init);
      })
    };
  },

  /**
   * Return the local storage directory for app groups.
   *
   * This is an Apple only feature.
   */
  getAppGroupDir(groupName) {
    if (Platform.OS !== 'ios' && Platform.OS !== 'macos') {
      return Promise.reject(new Error('AppGroups are available on Apple devices only'));
    }

    return FileAccessNative.getAppGroupDir(groupName);
  },

  /**
   * Hash the file content.
   */
  hash(path, algorithm) {
    return FileAccessNative.hash(path, algorithm);
  },

  /**
   * Check if a path is a directory.
   */
  isDir(path) {
    return FileAccessNative.isDir(path);
  },

  /**
   * List files in a directory.
   */
  ls(path) {
    return FileAccessNative.ls(path);
  },

  /**
   * Make a new directory.
   *
   * Returns path of the created directory.
   */
  mkdir(path) {
    return FileAccessNative.mkdir(path);
  },

  /**
   * Move a file.
   */
  mv(source, target) {
    return FileAccessNative.mv(source, target);
  },

  /**
   * Read the content of a file.
   */
  readFile(path, encoding = 'utf8') {
    return FileAccessNative.readFile(path, encoding);
  },

  /**
   * Read file metadata.
   */
  stat(path) {
    return FileAccessNative.stat(path);
  },

  /**
   * Read metadata of all files in a directory.
   */
  statDir(path) {
    return FileAccessNative.statDir(path);
  },

  /**
   * Delete a file.
   */
  unlink(path) {
    return FileAccessNative.unlink(path);
  },

  /**
   * Extract a zip archive.
   */
  unzip(source, target) {
    return FileAccessNative.unzip(source, target);
  },

  /**
   * Write content to a file.
   *
   * Default encoding of `data` is assumed utf8.
   */
  writeFile(path, data, encoding = 'utf8') {
    return FileAccessNative.writeFile(path, data, encoding);
  }

};
/**
 * Directory constants.
 */

export const Dirs = (_NativeModules$RNFile = NativeModules.RNFileAccess) === null || _NativeModules$RNFile === void 0 ? void 0 : _NativeModules$RNFile.getConstants();
/**
 * Utility functions for working with Android scoped storage.
 */

export const AndroidScoped = {
  /**
   * Append a path segment to an Android scoped storage content uri.
   */
  appendPath(basePath, segment) {
    return basePath + encodeURIComponent('/' + segment);
  }

};
//# sourceMappingURL=index.js.map