"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "Util", {
  enumerable: true,
  get: function () {
    return _util.Util;
  }
});
exports.AndroidScoped = exports.Dirs = exports.FileSystem = void 0;

var _reactNative = require("react-native");

var _native = require("./native");

var _util = require("./util");

var _NativeModules$RNFile;

/**
 * ID tracking next fetch request.
 */
let nextRequestId = 1;
/**
 * Process fetch events for the request.
 */

function registerFetchListener(requestId, resolve, reject, onProgress) {
  const listener = _native.FileAccessEventEmitter.addListener('FetchEvent', event => {
    if (event.requestId !== requestId) {
      return;
    }

    if (event.state === 'progress') {
      onProgress === null || onProgress === void 0 ? void 0 : onProgress(event.bytesRead, event.contentLength, event.done);
    } else if (event.state === 'error') {
      listener.remove();
      reject(new Error(event.message));
    } else if (event.state === 'complete') {
      listener.remove();
      resolve({
        headers: event.headers,
        ok: event.ok,
        redirected: event.redirected,
        status: event.status,
        statusText: event.statusText,
        url: event.url
      });
    }
  });
}

const FileSystem = {
  /**
   * Append content to a file.
   *
   * Default encoding of `data` is assumed utf8.
   */
  appendFile(path, data, encoding = 'utf8') {
    return _native.FileAccessNative.appendFile(path, data, encoding);
  },

  /**
   * Append a file to another file.
   *
   * Returns number of bytes written.
   */
  concatFiles(source, target) {
    return _native.FileAccessNative.concatFiles(source, target);
  },

  /**
   * Copy a file.
   */
  cp(source, target) {
    return _native.FileAccessNative.cp(source, target);
  },

  /**
   * Copy a bundled asset file.
   *
   * When using Android asset type 'resource', include the folder, but skip the
   * file extension. For example use 'raw/foo', for the file 'res/raw/foo.txt'.
   * When possible, prefer using the 'assets/' folder; files in 'res/' have
   * naming restrictions imposed by Android.
   * https://developer.android.com/guide/topics/resources/providing-resources.html#OriginalFiles
   */
  cpAsset(asset, target, type = 'asset') {
    return _reactNative.Platform.OS === 'android' ? _native.FileAccessNative.cpAsset(asset, target, type) : _native.FileAccessNative.cpAsset(asset, target);
  },

  /**
   * Copy a file to an externally controlled location.
   *
   * On Android API level < 29, may require permission WRITE_EXTERNAL_STORAGE.
   */
  cpExternal(source, targetName, dir) {
    return _native.FileAccessNative.cpExternal(source, targetName, dir);
  },

  /**
   * Check device available space.
   */
  df() {
    return _native.FileAccessNative.df();
  },

  /**
   * Check if a path exists.
   */
  exists(path) {
    return _native.FileAccessNative.exists(path);
  },

  /**
   * Save a network request to a file.
   */
  async fetch(resource, init, onProgress) {
    const requestId = nextRequestId++;
    return new Promise((resolve, reject) => {
      registerFetchListener(requestId, resolve, reject, onProgress);

      _native.FileAccessNative.fetch(requestId, resource, init);
    });
  },

  /**
   * Save a network request to a file.
   */
  fetchManaged(resource, init, onProgress) {
    const requestId = nextRequestId++;
    return {
      cancel: () => _native.FileAccessNative.cancelFetch(requestId),
      result: new Promise((resolve, reject) => {
        registerFetchListener(requestId, resolve, reject, onProgress);

        _native.FileAccessNative.fetch(requestId, resource, init);
      })
    };
  },

  /**
   * Return the local storage directory for app groups.
   *
   * This is an Apple only feature.
   */
  getAppGroupDir(groupName) {
    if (_reactNative.Platform.OS !== 'ios' && _reactNative.Platform.OS !== 'macos') {
      return Promise.reject(new Error('AppGroups are available on Apple devices only'));
    }

    return _native.FileAccessNative.getAppGroupDir(groupName);
  },

  /**
   * Hash the file content.
   */
  hash(path, algorithm) {
    return _native.FileAccessNative.hash(path, algorithm);
  },

  /**
   * Check if a path is a directory.
   */
  isDir(path) {
    return _native.FileAccessNative.isDir(path);
  },

  /**
   * List files in a directory.
   */
  ls(path) {
    return _native.FileAccessNative.ls(path);
  },

  /**
   * Make a new directory.
   *
   * Returns path of the created directory.
   */
  mkdir(path) {
    return _native.FileAccessNative.mkdir(path);
  },

  /**
   * Move a file.
   */
  mv(source, target) {
    return _native.FileAccessNative.mv(source, target);
  },

  /**
   * Read the content of a file.
   */
  readFile(path, encoding = 'utf8') {
    return _native.FileAccessNative.readFile(path, encoding);
  },

  /**
   * Read file metadata.
   */
  stat(path) {
    return _native.FileAccessNative.stat(path);
  },

  /**
   * Read metadata of all files in a directory.
   */
  statDir(path) {
    return _native.FileAccessNative.statDir(path);
  },

  /**
   * Delete a file.
   */
  unlink(path) {
    return _native.FileAccessNative.unlink(path);
  },

  /**
   * Extract a zip archive.
   */
  unzip(source, target) {
    return _native.FileAccessNative.unzip(source, target);
  },

  /**
   * Write content to a file.
   *
   * Default encoding of `data` is assumed utf8.
   */
  writeFile(path, data, encoding = 'utf8') {
    return _native.FileAccessNative.writeFile(path, data, encoding);
  }

};
/**
 * Directory constants.
 */

exports.FileSystem = FileSystem;
const Dirs = (_NativeModules$RNFile = _reactNative.NativeModules.RNFileAccess) === null || _NativeModules$RNFile === void 0 ? void 0 : _NativeModules$RNFile.getConstants();
/**
 * Utility functions for working with Android scoped storage.
 */

exports.Dirs = Dirs;
const AndroidScoped = {
  /**
   * Append a path segment to an Android scoped storage content uri.
   */
  appendPath(basePath, segment) {
    return basePath + encodeURIComponent('/' + segment);
  }

};
exports.AndroidScoped = AndroidScoped;
//# sourceMappingURL=index.js.map