"use strict";
/* eslint-disable @typescript-eslint/no-unsafe-return */
/* eslint-disable @typescript-eslint/no-unsafe-assignment */
/* eslint-disable @typescript-eslint/no-unsafe-member-access */
/* eslint-disable @typescript-eslint/no-unsafe-call */
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleItem = exports.getImage = exports.getLink = exports.getContent = exports.getDescription = exports.getTitle = exports.getAuthor = exports.getEnclosure = exports.getGuid = exports.isValidItem = exports.ItunesEpisodeType = void 0;
const logger_1 = require("../logger");
const shared_1 = require("./shared");
const unescape_1 = require("./unescape");
var ItunesEpisodeType;
(function (ItunesEpisodeType) {
    ItunesEpisodeType["Full"] = "full";
    ItunesEpisodeType["Trailer"] = "trailer";
    ItunesEpisodeType["Bonus"] = "bonus";
})(ItunesEpisodeType = exports.ItunesEpisodeType || (exports.ItunesEpisodeType = {}));
function isValidItem(item, { allowMissingGuid = false } = {}) {
    // If there is no enclosure, just skip this item and move on to the next
    if (!getEnclosure(item)) {
        logger_1.logger.warn("Item has no enclosure, skipping it.");
        return false;
    }
    // If there is no guid in the item, then skip this item and move on
    if (!getGuid(item) && !allowMissingGuid) {
        logger_1.logger.warn("Item has no guid, skipping it.");
        return false;
    }
    if (!getGuid(item) && allowMissingGuid) {
        logger_1.logger.warn("Item has no guid, but flag passed to allow it.");
    }
    // If there is no title or description
    if (!getTitle(item) && !getDescription(item)) {
        logger_1.logger.warn("Item has no title or description, skipping it.");
        return false;
    }
    return true;
}
exports.isValidItem = isValidItem;
/** If an item guid is empty, its considered invalid and will be skipped */
function getGuid(item) {
    // The guid node also has a isPermaLink attribute which is being ignored
    // https://validator.w3.org/feed/docs/error/InvalidPermalink.html
    const node = shared_1.firstWithValue(item.guid);
    const textValue = shared_1.getText(node);
    if (textValue) {
        return textValue;
    }
    const numberValue = shared_1.getNumber(node);
    if (typeof numberValue === "number") {
        return numberValue.toString();
    }
    console.warn("Empty/missing guid, returning empty string");
    return "";
}
exports.getGuid = getGuid;
function getEnclosure(item) {
    var _a, _b;
    const node = shared_1.firstWithAttributes(item.enclosure, ["url"]);
    if (node) {
        return {
            url: unescape_1.unescape(shared_1.getKnownAttribute(node, "url")),
            length: parseInt((_a = shared_1.getAttribute(node, "length")) !== null && _a !== void 0 ? _a : "0", 10),
            type: (_b = shared_1.getAttribute(node, "type")) !== null && _b !== void 0 ? _b : shared_1.guessEnclosureType(shared_1.getKnownAttribute(node, "url")),
        };
    }
    return null;
}
exports.getEnclosure = getEnclosure;
function getAuthor(item) {
    const node = shared_1.firstWithValue(item.author);
    const fallbackNode = shared_1.firstWithValue(item["itunes:author"]);
    const author = shared_1.getText(node) || shared_1.getText(fallbackNode);
    if (author) {
        return { author };
    }
    return undefined;
}
exports.getAuthor = getAuthor;
function getTitle(item) {
    const node = shared_1.firstWithValue(item.title);
    const fallbackNode = shared_1.firstWithValue(item["itunes:title"]);
    const title = shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(shared_1.getText(node) || shared_1.getText(fallbackNode)));
    if (title) {
        return { title };
    }
    return undefined;
}
exports.getTitle = getTitle;
function getDescription(item) {
    const node = shared_1.firstWithValue(item.description);
    if (node) {
        const descriptionValue = shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(shared_1.getText(node)));
        if (descriptionValue) {
            return { description: descriptionValue };
        }
    }
    const contentValue = getContent(item);
    if (contentValue === null || contentValue === void 0 ? void 0 : contentValue.content) {
        return { description: contentValue.content };
    }
    const summary = getSummary(item);
    if (summary) {
        return { description: summary.summary };
    }
    return undefined;
}
exports.getDescription = getDescription;
function getContent(item) {
    const contentNode = shared_1.firstWithValue(item["content:encoded"]);
    if (contentNode) {
        const contentValue = shared_1.getText(contentNode);
        if (contentValue) {
            return { content: contentValue };
        }
    }
    return undefined;
}
exports.getContent = getContent;
function getLink(item) {
    const node = shared_1.firstWithValue(item.link) || shared_1.firstWithAttributes(item.link, ["href"]);
    const value = shared_1.getText(node) || shared_1.getAttribute(node, "href") || "";
    if (value) {
        return { link: value };
    }
    return undefined;
}
exports.getLink = getLink;
function getItunesImage(item) {
    const image = shared_1.ensureArray(item["itunes:image"]).find((n) => shared_1.getText(n) || shared_1.getAttribute(n, "href") || shared_1.getText(n === null || n === void 0 ? void 0 : n.url));
    if (!image) {
        return undefined;
    }
    return {
        itunesImage: shared_1.sanitizeUrl(shared_1.getText(image) || shared_1.getAttribute(image, "href") || shared_1.getText(image.url)),
    };
}
function getImage(item) {
    const node = shared_1.ensureArray(item.image).find((n) => shared_1.getText(n === null || n === void 0 ? void 0 : n.url));
    const image = shared_1.sanitizeUrl(shared_1.getText(node === null || node === void 0 ? void 0 : node.url));
    if (image) {
        return { image };
    }
    const itunesImage = getItunesImage(item);
    if (itunesImage) {
        return { image: itunesImage.itunesImage };
    }
    return undefined;
}
exports.getImage = getImage;
function getExplicit(item) {
    const node = shared_1.firstWithValue(item["itunes:explicit"]);
    const nodeText = shared_1.getText(node).toLowerCase();
    if (["yes", "true"].includes(nodeText)) {
        return true;
    }
    if (typeof node === "boolean" && node) {
        return true;
    }
    return false;
}
function getDuration(item) {
    var _a;
    const node = shared_1.firstWithValue(item["itunes:duration"]);
    const durationValue = (_a = (shared_1.getText(node) || shared_1.getNumber(node))) !== null && _a !== void 0 ? _a : 0;
    if (typeof durationValue === "string") {
        const seconds = shared_1.timeToSeconds(durationValue);
        return Number.isNaN(seconds) ? 0 : seconds;
    }
    return durationValue;
}
function getItunesEpisode(item) {
    const node = shared_1.firstWithValue(item["itunes:episode"]);
    const episodeValue = shared_1.getText(node) || shared_1.getNumber(node);
    if (typeof episodeValue === "string") {
        const parsedString = episodeValue.replace(/\D/g, "");
        if (parsedString) {
            return { itunesEpisode: parseInt(parsedString, 10) };
        }
    }
    else if (typeof episodeValue === "number") {
        return { itunesEpisode: episodeValue };
    }
    return undefined;
}
function getItunesSeason(item) {
    const node = shared_1.firstWithValue(item["itunes:season"]);
    const value = shared_1.getText(node) || shared_1.getNumber(node);
    if (typeof value === "string") {
        const parsedString = value.replace(/\D/g, "");
        if (parsedString) {
            return { itunesSeason: parseInt(parsedString, 10) };
        }
    }
    else if (typeof value === "number") {
        return { itunesSeason: value };
    }
    return undefined;
}
function getItunesEpisodeType(item) {
    const node = shared_1.firstWithValue(item["itunes:episodeType"]);
    const typeValue = shared_1.getText(node).toLowerCase();
    const episodeType = shared_1.lookup(ItunesEpisodeType, typeValue);
    if (episodeType) {
        return { itunesEpisodeType: episodeType };
    }
    return undefined;
}
function getKeywords(item) {
    const node = shared_1.firstWithValue(item["itunes:keywords"]);
    const keywords = shared_1.getText(node);
    if (keywords) {
        const parsed = keywords
            .split(",")
            .map((k) => k.trim())
            .map(shared_1.sanitizeMultipleSpaces)
            .filter(Boolean);
        if (parsed.length) {
            return { keywords: parsed };
        }
    }
    return undefined;
}
function getPubDate(item) {
    const node = shared_1.firstWithValue(item.pubDate);
    const value = shared_1.getText(node) || shared_1.getNumber(node);
    if (value) {
        const parsed = shared_1.pubDateToDate(value);
        if (parsed) {
            return { pubDate: parsed };
        }
    }
    return undefined;
}
function getSummary(item) {
    const node = shared_1.firstWithValue(item["itunes:summary"]);
    if (node) {
        const summaryValue = shared_1.getText(node);
        if (summaryValue) {
            return { summary: summaryValue };
        }
    }
    return undefined;
}
function getSubtitle(item) {
    const node = shared_1.firstWithValue(item["itunes:subtitle"]);
    if (node) {
        const value = shared_1.sanitizeNewLines(shared_1.sanitizeMultipleSpaces(shared_1.getText(node)));
        if (value) {
            return { subtitle: value };
        }
    }
    return undefined;
}
function handleItem(item, _feed) {
    return {
        guid: getGuid(item),
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        enclosure: getEnclosure(item),
        duration: getDuration(item),
        explicit: getExplicit(item),
        ...getTitle(item),
        ...getAuthor(item),
        ...getLink(item),
        ...getItunesImage(item),
        ...getItunesEpisode(item),
        ...getItunesEpisodeType(item),
        ...getItunesSeason(item),
        ...getKeywords(item),
        ...getPubDate(item),
        ...getImage(item),
        ...getSummary(item),
        ...getDescription(item),
        ...getSubtitle(item),
        ...getContent(item),
    };
}
exports.handleItem = handleItem;
//# sourceMappingURL=item.js.map