"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleFeed = void 0;
/* eslint-disable @typescript-eslint/no-unsafe-member-access */
/* eslint-disable @typescript-eslint/no-unsafe-assignment */
const logger_1 = require("../logger");
const shared_1 = require("./shared");
const types_1 = require("./types");
const itunes_categories_1 = require("./itunes-categories");
function getTitle(feed) {
    var _a;
    const node = shared_1.firstWithValue(feed.title);
    return shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(shared_1.getText(node) || `${(_a = shared_1.getNumber(node)) !== null && _a !== void 0 ? _a : ""}`));
}
function getDescription(feed) {
    const node = shared_1.firstWithValue(feed.description);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue));
        }
    }
    const fallbackNode = getSummary(feed);
    if (fallbackNode) {
        return fallbackNode.summary;
    }
    const atomFallback = getSubtitle(feed);
    if (atomFallback) {
        return atomFallback.subtitle;
    }
    return "";
}
function getCopyright(feed) {
    const node = shared_1.firstWithValue(feed.copyright);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return { copyright: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    return undefined;
}
function getWebmaster(feed) {
    const node = shared_1.firstWithValue(feed.webMaster);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return { webmaster: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    return undefined;
}
function getManagingEditor(feed) {
    const node = shared_1.firstWithValue(feed.managingEditor);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return { managingEditor: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    return undefined;
}
function getSummary(feed) {
    const node = shared_1.firstWithValue(feed["itunes:summary"]);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return { summary: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    return undefined;
}
function getSubtitle(feed) {
    const node = shared_1.firstWithValue(feed.subtitle);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return { subtitle: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    const fallbackNode = shared_1.firstWithValue(feed["itunes:subtitle"]);
    if (fallbackNode) {
        const nodeValue = shared_1.getText(fallbackNode);
        if (nodeValue) {
            return { subtitle: shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue)) };
        }
    }
    return undefined;
}
function getLink(feed) {
    const node = shared_1.firstWithValue(feed.link);
    if (node) {
        const nodeValue = shared_1.getText(node);
        if (nodeValue) {
            return shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue));
        }
    }
    const nodeWithAttribute = shared_1.firstWithAttributes(feed.link, ["href"]);
    if (nodeWithAttribute) {
        const nodeValue = shared_1.getKnownAttribute(nodeWithAttribute, "href");
        if (nodeValue) {
            return shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue));
        }
    }
    const fallbackNode = shared_1.firstWithValue(feed.url);
    if (fallbackNode) {
        const nodeValue = shared_1.getText(fallbackNode);
        if (nodeValue) {
            return shared_1.sanitizeMultipleSpaces(shared_1.sanitizeNewLines(nodeValue));
        }
    }
    return "";
}
function getBoolean(item, truthyValues) {
    const node = shared_1.firstWithValue(item);
    const nodeText = shared_1.getText(node).toLowerCase();
    const normalizedTruthy = truthyValues.map((x) => x.toLowerCase());
    if (normalizedTruthy.includes(nodeText)) {
        return true;
    }
    if (normalizedTruthy.includes("true") && typeof node === "boolean" && node) {
        return true;
    }
    return false;
}
function getExplicit(feed) {
    return getBoolean(feed["itunes:explicit"], ["yes", "true"]);
}
function getItunesBlock(feed) {
    return getBoolean(feed["itunes:block"], ["yes"]);
}
function getItunesComplete(feed) {
    return getBoolean(feed["itunes:complete"], ["yes"]);
}
function getLanguage(feed) {
    const node = shared_1.firstWithValue(feed.language);
    const value = shared_1.getText(node);
    if (value) {
        return { language: value };
    }
    return undefined;
}
function getItunesImage(feed) {
    const target = feed["itunes:image"];
    const node = shared_1.ensureArray(target).find((imageNode) => shared_1.getText(imageNode.url));
    if (node) {
        const value = shared_1.getText(node.url);
        if (value) {
            return { itunesImage: value };
        }
    }
    const fallback = shared_1.firstWithAttributes(target, ["href"]);
    if (fallback) {
        return { itunesImage: shared_1.getKnownAttribute(target, "href") };
    }
    const lastFallback = shared_1.firstWithValue(target);
    if (lastFallback) {
        const value = shared_1.getText(lastFallback);
        if (value) {
            return { itunesImage: value };
        }
    }
    return undefined;
}
function getItunesCategory(feed) {
    const getCategoriesNode = (node) => 
    // eslint-disable-next-line @typescript-eslint/no-unsafe-return
    shared_1.ensureArray(node["itunes:category"]);
    const normalize = (str) => str.toLowerCase().replace("&amp;", "&");
    const categoriesSet = new Set();
    const extractCategories = (currentNode, parents) => {
        const categories = getCategoriesNode(currentNode);
        if (categories.length > 0) {
            categories.forEach((cat) => {
                var _a;
                const categoryName = normalize((_a = shared_1.getAttribute(cat, "text")) !== null && _a !== void 0 ? _a : "");
                if (categoryName) {
                    extractCategories(cat, [...parents, categoryName]);
                }
            });
        }
        else if (parents.length > 0) {
            categoriesSet.add(parents.join(" > "));
        }
    };
    extractCategories(feed, []);
    if (categoriesSet.size === 0) {
        return undefined;
    }
    return {
        itunesCategory: Array.from(categoriesSet).map(itunes_categories_1.categoryLookup).filter(Boolean),
    };
}
function getGenerator(feed) {
    const node = shared_1.firstWithValue(feed.generator);
    const nodeValue = shared_1.getText(node);
    if (nodeValue) {
        return { generator: nodeValue };
    }
    return undefined;
}
function getPubDate(feed) {
    const node = shared_1.firstWithValue(feed.pubDate);
    const nodeValue = shared_1.getText(node);
    if (nodeValue) {
        const parsed = shared_1.pubDateToDate(nodeValue);
        if (parsed) {
            return { pubDate: parsed };
        }
    }
    const fallbackNode = shared_1.firstWithValue(feed.lastBuildDate);
    const fallbackNodeValue = shared_1.getText(fallbackNode);
    if (fallbackNodeValue) {
        const parsed = shared_1.pubDateToDate(fallbackNodeValue);
        if (parsed) {
            return { pubDate: parsed };
        }
    }
    const atomFallbackNode = shared_1.firstWithValue(feed.updated);
    const atomFallbackNodeValue = shared_1.getText(atomFallbackNode);
    if (atomFallbackNodeValue) {
        const parsed = shared_1.pubDateToDate(atomFallbackNodeValue);
        if (parsed) {
            return { pubDate: parsed };
        }
    }
    return undefined;
}
function getLastBuildDate(feed) {
    const node = shared_1.firstWithValue(feed.lastBuildDate);
    const nodeValue = shared_1.getText(node);
    if (nodeValue) {
        const parsed = shared_1.pubDateToDate(nodeValue);
        if (parsed) {
            return { lastBuildDate: parsed };
        }
    }
    return undefined;
}
function getItunesType(feed) {
    const node = shared_1.firstWithValue(feed["itunes:type"]);
    const nodeValue = shared_1.getText(node);
    if (nodeValue) {
        const parsed = shared_1.lookup(types_1.ItunesFeedType, nodeValue.toLowerCase());
        if (parsed) {
            return { itunesType: parsed };
        }
    }
    const fallbackNode = shared_1.firstWithAttributes(feed["itunes:type"], ["text"]);
    if (fallbackNode) {
        const parsed = shared_1.lookup(types_1.ItunesFeedType, shared_1.getKnownAttribute(fallbackNode, "text").toLowerCase());
        if (parsed) {
            return { itunesType: parsed };
        }
    }
    return undefined;
}
function getItunesNewFeedUrl(feed) {
    const node = shared_1.firstWithValue(feed["itunes:new-feed-url"]);
    const nodeValue = shared_1.getText(node);
    if (nodeValue) {
        return { itunesNewFeedUrl: nodeValue };
    }
    return undefined;
}
const allowedCategories = new Set([
    "arts",
    "books",
    "design",
    "fashion",
    "beauty",
    "food",
    "performing",
    "visual",
    "business",
    "careers",
    "entrepreneurship",
    "investing",
    "management",
    "marketing",
    "nonprofit",
    "comedy",
    "interviews",
    "improv",
    "standup",
    "education",
    "courses",
    "howto",
    "language",
    "learning",
    "selfimprovement",
    "fiction",
    "drama",
    "history",
    "health",
    "fitness",
    "alternative",
    "medicine",
    "mental",
    "nutrition",
    "sexuality",
    "kids",
    "family",
    "parenting",
    "pets",
    "animals",
    "stories",
    "leisure",
    "animation",
    "manga",
    "automotive",
    "aviation",
    "crafts",
    "games",
    "hobbies",
    "home",
    "garden",
    "videogames",
    "music",
    "commentary",
    "news",
    "daily",
    "entertainment",
    "government",
    "politics",
    "buddhism",
    "christianity",
    "hinduism",
    "islam",
    "judaism",
    "religion",
    "spirituality",
    "science",
    "astronomy",
    "chemistry",
    "earth",
    "life",
    "mathematics",
    "natural",
    "nature",
    "physics",
    "social",
    "society",
    "culture",
    "documentary",
    "personal",
    "journals",
    "philosophy",
    "places",
    "travel",
    "relationships",
    "sports",
    "baseball",
    "basketball",
    "cricket",
    "fantasy",
    "football",
    "golf",
    "hockey",
    "rugby",
    "running",
    "soccer",
    "swimming",
    "tennis",
    "volleyball",
    "wilderness",
    "wrestling",
    "technology",
    "truecrime",
    "tv",
    "film",
    "aftershows",
    "reviews",
    "climate",
    "weather",
    "tabletop",
    "role-playing",
    "cryptocurrency",
]);
const allowedCompoundCategories = new Map([
    ["video", { others: ["games"], result: "videogames" }],
    ["true", { others: ["crime"], result: "truecrime" }],
    ["after", { others: ["shows"], result: "aftershows" }],
    ["self", { others: ["improvement"], result: "selfimprovement" }],
    ["how", { others: ["to"], result: "howto" }],
]);
const handleCompoundCategory = (categoryName, categoriesList, categoriesSet) => {
    const compoundCategory = allowedCompoundCategories.get(categoryName);
    if (compoundCategory) {
        if (compoundCategory.others.every((o) => categoriesList.includes(o)) &&
            allowedCategories.has(compoundCategory.result)) {
            categoriesSet.add(compoundCategory.result);
        }
        else {
            logger_1.logger.warn(`Compound category wasn't in the allow list - ${compoundCategory.result}`);
        }
    }
};
function getCategories(feed) {
    const split = (str) => str
        .replace(/-/g, "")
        .split("&amp;")
        .reduce((acc, curr) => [...acc, ...curr.split(/\s+&\s+/)], [])
        .reduce((acc, curr) => [...acc, ...curr.split(/\s+/)], [])
        .map((s) => s.trim())
        .filter(Boolean);
    const getCategoriesNode = (node) => 
    // eslint-disable-next-line @typescript-eslint/no-unsafe-return
    shared_1.ensureArray(node["itunes:category"]);
    const normalize = (str) => str.toLowerCase();
    const categoriesSet = new Set();
    const extractCategories = (currentNode) => {
        const categories = getCategoriesNode(currentNode);
        if (categories.length > 0) {
            categories.forEach((cat) => {
                var _a;
                const categoryName = normalize((_a = shared_1.getAttribute(cat, "text")) !== null && _a !== void 0 ? _a : "");
                if (categoryName) {
                    split(categoryName).forEach((normalizedCategoryName, _idx, all) => {
                        if (allowedCompoundCategories.has(normalizedCategoryName)) {
                            handleCompoundCategory(normalizedCategoryName, all, categoriesSet);
                        }
                        else if (allowedCategories.has(normalizedCategoryName)) {
                            categoriesSet.add(normalizedCategoryName);
                        }
                    });
                    extractCategories(cat);
                }
            });
        }
    };
    extractCategories(feed);
    if (categoriesSet.size === 0) {
        return undefined;
    }
    return { categories: Array.from(categoriesSet) };
    return undefined;
}
function getPubSub(feed) {
    var _a, _b, _c;
    const getNode = (key, type) => 
    // eslint-disable-next-line @typescript-eslint/no-unsafe-return
    shared_1.ensureArray(feed[key]).find((n) => shared_1.getAttribute(n, "href") && shared_1.getAttribute(n, "rel") === type);
    const selfNode = (_a = getNode("atom:link", "self")) !== null && _a !== void 0 ? _a : getNode("link", "self");
    const hubNode = (_b = getNode("atom:link", "hub")) !== null && _b !== void 0 ? _b : getNode("link", "hub");
    const nextNode = (_c = getNode("atom:link", "next")) !== null && _c !== void 0 ? _c : getNode("link", "next");
    if (!selfNode && !hubNode && !nextNode) {
        return undefined;
    }
    const pubsub = {};
    if (selfNode) {
        pubsub.self = shared_1.getKnownAttribute(selfNode, "href");
    }
    if (hubNode) {
        pubsub.hub = shared_1.getKnownAttribute(hubNode, "href");
    }
    if (nextNode) {
        pubsub.next = shared_1.getKnownAttribute(nextNode, "href");
    }
    return { pubsub };
}
function getAuthor(feed) {
    const node = shared_1.firstWithValue(feed["itunes:author"]);
    if (node) {
        return { author: shared_1.getText(node) };
    }
    return undefined;
}
function getOwner(feed) {
    const node = shared_1.ensureArray(feed["itunes:owner"]).find((n) => shared_1.firstWithValue(n["itunes:name"]) && shared_1.firstWithValue(n["itunes:email"]));
    if (node) {
        const name = shared_1.firstWithValue(node["itunes:name"]);
        const email = shared_1.firstWithValue(node["itunes:email"]);
        return {
            owner: {
                name: shared_1.getText(name),
                email: shared_1.getText(email),
            },
        };
    }
    return undefined;
}
function getImage(feed) {
    var _a, _b;
    const nodeWithUrl = shared_1.ensureArray(feed.image).find((n) => shared_1.getText(n.url));
    if (nodeWithUrl) {
        const result = { image: { url: shared_1.getText(shared_1.firstWithValue(nodeWithUrl.url)) } };
        const title = shared_1.firstWithValue(nodeWithUrl.title);
        if (title) {
            result.image.title = shared_1.getText(title);
        }
        const link = shared_1.firstWithValue(nodeWithUrl.link);
        if (link) {
            result.image.link = shared_1.getText(link);
        }
        const width = shared_1.firstWithValue(nodeWithUrl.width);
        if (width) {
            result.image.width = (_a = shared_1.getNumber(width)) !== null && _a !== void 0 ? _a : 0;
        }
        const height = shared_1.firstWithValue(nodeWithUrl.height);
        if (height) {
            result.image.height = (_b = shared_1.getNumber(height)) !== null && _b !== void 0 ? _b : 0;
        }
        return result;
    }
    const itunesImage = getItunesImage(feed);
    if (itunesImage) {
        return {
            image: { url: itunesImage.itunesImage },
        };
    }
    const logoNode = shared_1.firstWithValue(feed.logo);
    if (logoNode) {
        return {
            image: {
                url: shared_1.getText(logoNode),
            },
        };
    }
    return undefined;
}
function getTimeToLive(feed) {
    const node = shared_1.firstWithValue(feed.ttl);
    if (node !== null) {
        const parsed = shared_1.getNumber(node);
        if (parsed !== null) {
            return { ttl: parsed };
        }
    }
    return undefined;
}
function handleFeed(feed, feedType) {
    return {
        lastUpdate: new Date(),
        type: feedType,
        title: getTitle(feed),
        description: getDescription(feed),
        link: getLink(feed),
        explicit: getExplicit(feed),
        itunesBlock: getItunesBlock(feed),
        itunesComplete: getItunesComplete(feed),
        ...getItunesImage(feed),
        ...getLanguage(feed),
        ...getItunesCategory(feed),
        ...getGenerator(feed),
        ...getPubDate(feed),
        ...getLastBuildDate(feed),
        ...getItunesType(feed),
        ...getItunesNewFeedUrl(feed),
        ...getCategories(feed),
        ...getPubSub(feed),
        ...getAuthor(feed),
        ...getOwner(feed),
        ...getImage(feed),
        ...getSummary(feed),
        ...getSubtitle(feed),
        ...getCopyright(feed),
        ...getWebmaster(feed),
        ...getManagingEditor(feed),
        ...getTimeToLive(feed),
        items: [],
        pc20support: {},
    };
}
exports.handleFeed = handleFeed;
//# sourceMappingURL=feed.js.map