"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkFeedByObject = exports.checkFeedByUri = exports.checkHttps = exports.checkHotlink = exports.checkCors = void 0;
const node_fetch_1 = __importDefault(require("node-fetch"));
const zip_1 = __importDefault(require("ramda/src/zip"));
const logger_1 = require("../logger");
const parser_1 = require("../parser");
const shared_1 = require("../shared");
const podcastCertification = "https://podcastcertification.org";
function checkCors(urlToCheck, methodToCheck = "GET", origin = podcastCertification) {
    return node_fetch_1.default(urlToCheck, {
        redirect: "follow",
        follow: 20,
        method: "OPTIONS",
        headers: {
            origin,
            "Access-Control-Request-Method": methodToCheck,
        },
    }).then((resp) => {
        return ["*", origin].includes(resp.headers.get("access-control-allow-origin"));
        // resp.headers.has("access-control-allow-methods") &&
        // (resp.headers.get("access-control-allow-methods") as string[]).includes(methodToCheck)
    }, (err) => {
        logger_1.logger.error(err);
        return false;
    });
}
exports.checkCors = checkCors;
function checkHotlink(urlToCheck, referer = podcastCertification) {
    return node_fetch_1.default(urlToCheck, {
        method: "GET",
        headers: {
            referer,
            "user-agent": "partytime/hotlink-check",
        },
    }).then((resp) => {
        return resp.status < 300 && resp.status >= 200;
    });
}
exports.checkHotlink = checkHotlink;
function checkHttps(urlToCheck, referer = podcastCertification) {
    return node_fetch_1.default(urlToCheck.replace(/^http:\/\//, "https://"), {
        method: "GET",
        headers: {
            referer,
            "user-agent": "partytime/https-check",
        },
    }).then((resp) => {
        return resp.status < 300 && resp.status >= 200;
    });
}
exports.checkHttps = checkHttps;
async function checkFeedByUri(uri) {
    const xml = await shared_1.getFeedText(uri);
    const feedObject = parser_1.parseFeed(xml);
    if (feedObject) {
        return checkFeedByObject({ uri, feedObject });
    }
    throw new Error(`Feed could not be parsed`);
}
exports.checkFeedByUri = checkFeedByUri;
// eslint-disable-next-line sonarjs/cognitive-complexity
async function checkFeedByObject({ uri, feedObject, }) {
    const toCheck = {};
    if (uri.startsWith("http")) {
        toCheck.corsFeedUri = uri;
    }
    const newestEpisode = feedObject.items.reduce((latest, curr) => {
        if (curr.pubDate && !latest.pubDate) {
            return curr;
        }
        if (!curr.pubDate && latest.pubDate) {
            return latest;
        }
        if (curr.pubDate && latest.pubDate && curr.pubDate > latest.pubDate) {
            return curr;
        }
        return latest;
    }, { pubDate: new Date(0) });
    if (newestEpisode.enclosure.url) {
        toCheck.corsEpisodeEnclosure = newestEpisode.enclosure.url;
    }
    if (newestEpisode.podcastChapters) {
        toCheck.corsEpisodeChapters = newestEpisode.podcastChapters.url;
    }
    if (Array.isArray(newestEpisode.podcastTranscripts) &&
        newestEpisode.podcastTranscripts.length > 0) {
        toCheck.corsEpisodeTranscript = newestEpisode.podcastTranscripts[0].url;
    }
    const corsSupport = await Promise.all(Object.values(toCheck).map((s) => checkCors(s)));
    const resultObject = Object.fromEntries(zip_1.default(Object.keys(toCheck), corsSupport));
    if (feedObject.image) {
        resultObject.hotlinkFeedImage = await checkHotlink(feedObject.image.url);
        resultObject.httpsFeedImage = await checkHttps(feedObject.image.url);
    }
    if (newestEpisode.image) {
        resultObject.hotlinkEpisodeImage = await checkHotlink(newestEpisode.image);
        resultObject.httpsEpisodeImage = await checkHttps(newestEpisode.image);
    }
    if (newestEpisode.podcastChapters) {
        resultObject.httpsEpisodeChapters = await checkHttps(newestEpisode.podcastChapters.url);
    }
    if (Array.isArray(newestEpisode.podcastTranscripts) &&
        newestEpisode.podcastTranscripts.length > 0) {
        resultObject.httpsEpisodeTranscript = await checkHttps(newestEpisode.podcastTranscripts[0].url);
    }
    return resultObject;
}
exports.checkFeedByObject = checkFeedByObject;
//# sourceMappingURL=index.js.map